(*pp camlp4o *)
(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License
**-----------------------------------------------------------------------
**
** File: parse_c_scade.ml
** Author: jahier@imag.fr
*)

open Lexing
open MyGenlex

open Gen_stubs_common

(****************************************************************************)
(* Debugging *)

let debug = false

let (debug_print_vn_ct_list : string -> file -> string -> vn_ct list -> unit) =
  fun io file node l ->
    if debug then
      (
	print_string ("\n ********* list of "^io^" vn_ct in file "^ 
		      file ^ " and node "
		      ^ node ^": \n");
	(List.iter
	   (fun (vn, ct) -> print_string (vn ^ ":" ^ (ctype_to_string ct) ^ "\n")) 
	   l);
	flush stdout
      )

let (print_debug_func_do : Lexing.lexbuf -> string -> token Stream.t -> unit) =
  fun ic msg tok ->
    (match Stream.peek tok with 
	 None -> print_string "End of file " 
       | Some token ->  print_genlex_token token
    );
    print_string ( " *** At character " ^ 
      string_of_int (ic.lex_curr_pos) ^ ", token " ^
	(string_of_int (Stream.count tok)) ^ "\t: " ^ msg);
    flush stdout
      

let (print_debug_func : Lexing.lexbuf -> string -> token Stream.t -> unit) =
  fun ic msg tok ->
    if debug then print_debug_func_do ic msg tok else ()
      

(****************************************************************************)
(****************************************************************************)

let lexer = make_lexer ["typedef"; "struct";
			"."; ","; "{"; "}"; ";"; ":"; "("; ")"; "["; "]";
			"/*"; "*/"; "#"]

let set_pos buf i =
  buf.lex_curr_pos <- i


(* returns all the struct definitions of the file *)
let rec (get_all_typedef_struct : Lexing.lexbuf -> (string * vn_ct list) list
	   -> token Stream.t -> (string * vn_ct list) list) = 
  fun ic acc tok -> 
    let _ = print_debug_func ic ("get_all_typedef_struct \n") tok in
    (match tok with parser
	 [< 'Kwd ((_,nic), "typedef") >] ->  get_all_typedef_struct2 (set_pos ic nic; ic) acc tok
       | [< 'Ident ((_,nic), _) >] -> get_all_typedef_struct (set_pos ic nic; ic) acc tok
       | [< 'Kwd ((_,nic), _) >] -> get_all_typedef_struct (set_pos ic nic; ic) acc tok
       | [< 'Int ((_,nic), _) >] -> get_all_typedef_struct (set_pos ic nic; ic) acc tok
       | [< 'Float ((_,nic), _) >] -> get_all_typedef_struct (set_pos ic nic; ic) acc tok
       | [< 'Char ((_,nic), _) >] -> get_all_typedef_struct (set_pos ic nic; ic) acc tok
       | [< 'String ((_,nic), _) >] -> get_all_typedef_struct (set_pos ic nic; ic) acc tok
       | [<  >] -> acc
    )
and get_all_typedef_struct2 ic acc tok =
    let _ = print_debug_func ic ("get_all_typedef_struct2 \n") tok in
    (match tok with parser
	 [< 
	   'Kwd ((_,nic), "struct");  'Kwd ((_,nic), "{");  
	   vtl = parse_struct_body (set_pos ic nic; ic) [];
	  'Ident((_,nic2), struct_name)
	 >] -> 
	    (get_all_typedef_struct (set_pos ic nic2; ic) ((struct_name, vtl)::acc) tok)
	   
       | [< 'Ident ((_,nic), _) >] -> get_all_typedef_struct (set_pos ic nic; ic) acc tok
       | [< 'Kwd ((_,nic), _) >] -> get_all_typedef_struct (set_pos ic nic; ic) acc tok
       | [< 'Int ((_,nic), _) >] -> get_all_typedef_struct (set_pos ic nic; ic) acc tok
       | [< 'Float ((_,nic), _) >] -> get_all_typedef_struct (set_pos ic nic; ic) acc tok
       | [< 'Char ((_,nic), _) >] -> get_all_typedef_struct (set_pos ic nic; ic) acc tok
       | [< 'String ((_,nic), _) >] -> get_all_typedef_struct (set_pos ic nic; ic) acc tok
    )
and parse_struct_body ic acc tok =
    let _ = print_debug_func ic ("parse_struct_body \n") tok in
      try
	(match tok with parser
	     [< 'Ident (_, ct);  'Ident (_, vn); 'Kwd((_,nic), ";")  >] -> 
	       parse_struct_body (set_pos ic nic; ic) ((vn,Simple ct)::acc) tok
		 
	   | [< 'Kwd (_, "}") >] -> acc
	   | [< 'Kwd (_, _) >] -> assert false
	   | [< 'Int (_, _) >] -> assert false
	   | [< 'Float (_, _) >] -> assert false
	   | [< 'Char (_, _) >] -> assert false
	   | [< 'String (_, _) >] -> assert false
               
	)
      with
	| Stream.Error e  -> 
	    print_string ("*** Error when parsing the struct body at character " ^ 
			    string_of_int (ic.lex_curr_pos) ^ "\n");
	    flush stdout;
	    acc
	    
	 

let (find_var_list : string -> string -> token Stream.t -> string -> vn_ct list) =
  fun file file_content tok node_name ->
    (* returns the list of var names and type parsed in the struct node_name *)
    let buff = Lexing.from_string file_content in
    let tdl = get_all_typedef_struct buff [] tok in

(*   List.iter (fun (m,vnct) -> debug_print_vn_ct_list "all" file m  vnct) tdl; *)
      try 
	  List.rev (List.assoc ("_C_" ^ node_name) tdl)
      with Not_found -> 
	(
	  output_string stdout (
	    "\n********** DID NOT FIND INFORMATION ABOUT NODE " ^ node_name ^
	    " IN FILE " ^ file ^ 
	    ". **********\n********** YOU SHOULD SELECT A ROOT NODE THAT USES "^ node_name^ 
	    " AND BUILD IT. **********\n" 
	  );
	  (* 	    output_string stderr str; *)
	  flush stdout;
	  flush stderr;
	  exit 1
	)

let (del_var_name_prefix : string -> string) =
  fun var ->
    (* scade var names are of the form "_I0_i1"; therefore we remove
       the prefix "_I0_" to get the user var name
    *)
    try
      let reg__ = Str.regexp "_" in
      let sptr1 = Str.search_forward reg__ var 0 in
      let sptr2 = Str.search_forward reg__ var (sptr1+1) in
	String.sub var (sptr2+1) ((String.length var) - sptr2 - 1)
    with
	Not_found -> var

let _ = assert ((del_var_name_prefix "_I0_i1") = "i1")
let _ = assert ((del_var_name_prefix "_I12434_weird_var_name") = "weird_var_name")


(****************************************************************************)
(****************************************************************************)

let (get_vn_and_ct_list2: string -> string -> string -> vn_ct list * vn_ct list) =
  fun file file_content node_name ->

    let ic = try open_in file with
	_ ->
	  (
	    print_string ("*** File " ^ file
			  ^ " does not exist. Please check its name.\n");
	    flush stdout;
	    exit 2
	  )
    in
    let tok = (lexer(Stream.of_channel ic)) in
    let var_list = find_var_list file file_content tok node_name in
      (* split inputs and outputs. *)
    let (vi, vlo) =
      List.partition (fun (name, _ct) -> (String.get name 1) = 'I') var_list
    in
    let (vo, _vl) =
      List.partition (fun (name, _ct) -> (String.get name 1) = 'O') var_list
    in
      (
	List.map (fun (vn, ct) -> ((del_var_name_prefix vn), ct)) vi,
	List.map (fun (vn, ct) -> ((del_var_name_prefix vn), ct)) vo
      )


(****************************************************************************)
(****************************************************************************)
(** Call cpp on file and returns the file name containing the result
  Should only be used via the scade gui.

  I need to call cpp since then, i am sure to find all the
  information about variables and types that i need. Then i am also
  sure that thoses definitions are consistent with what is executed.

*)


let (cpp : string -> compiler -> string) =
  fun file compiler ->
    let _ = 
      assert (Filename.check_suffix file ".h");
      assert (Sys.file_exists file)
    in
    let dir = Filename.dirname file in
    let base = 
      try
	Filename.chop_extension  (Filename.basename file)
	  with _ -> assert false
    in
    let temp_file = base ^ ".tmp" in
    let new_file = base ^ "_cpp.h" in
    let str = Util.readfile_rm_crtl_m file in
    let oc = open_out (Filename.concat dir temp_file) in
    let _ =
      output_string oc (Str.global_replace (Str.regexp "bool ") "lurette__boolean " str);
      flush oc;
      close_out oc;
      output_string stderr ("File " ^ temp_file ^ " created in dir "^(Sys.getcwd ()) ^".\n");
      flush stderr
    in
    let make =
      try
	Util.string_to_string_list (Unix.getenv "MAKE")
      with _ ->
	["make"]
    in
    let makefile = 
      match compiler with
	  Scade -> 
	    let lurette_path =
	      try Unix.getenv "LURETTE_PATH"
	      with _ ->
		output_string stdout "Warning: environment var LURETTE_PATH is unset .\n";
		flush stdout;
		""
	    in
	      (Filename.concat (Filename.concat lurette_path "lib") "Makefile.lurette") 
	| ScadeGUI -> (Filename.concat dir "Makefile") 
	| _  -> assert false
    in
      try
	Unix.putenv "USER_TESTING_DIR" dir;
	Unix.putenv "LURETTE_TMP_DIR" (dir);
	ignore (Util.my_create_process 
		  (List.hd make) ((List.tl make)@["-f"; makefile ;  new_file]));
	new_file
      with e -> 
	print_string (Printexc.to_string e);
	flush stdout;
	file
    


(* exported *)
let rec (get_vn_and_ct_list: file -> string -> compiler -> 
	  typedef list * vn_ct list * vn_ct list) =
  fun file0 node compiler ->
    let file = cpp file0 compiler in
      try
	let _ = output_string stderr
	  ("\n parsing " ^ file ^ " (generated by scade) to get var " ^
	     "names and types. \n") ;
	  flush stderr
	in
	let str = Util.readfile_rm_crtl_m file in
	let (sut_vi0, sut_vo0) = get_vn_and_ct_list2 file str node in
	let tdl = get_typedef file in

	  debug_print_vn_ct_list "input" file node sut_vi0;
	  debug_print_vn_ct_list "output" file node sut_vo0;
	  (tdl, sut_vi0, sut_vo0)

      with 
	  Stream.Error "" -> 
	    print_string ("*** Error when parsing header file " ^ file ^ 
			    "(assuming scade convention).\n");
	    flush stdout;
	    exit 2
	| e ->
	    print_string (Printexc.to_string e);
	    print_string ("*** Error when parsing header file " ^ file ^ 
			    "(assuming scade convention).\n");
	    flush stdout;
	    exit 2

