(* Lutin Ocaml API *)

(**********************************************************************)

type input
type ouput

val print_rif_output : output -> unit
val print_rif_input : input -> unit

(**********************************************************************)
type formula
type num

(* formula construction *)

(* First way: using a string parser *)
val formula_parser : string -> formula option

(* Second way: using (abstract) functions *)
make_var : string -> var

val f : var -> formula
val e : var -> num

val pre : var -> var


val f_or : formula -> formula -> formula
val f_xor : formula -> formula -> formula
val f_and : formula -> formula -> formula
val f_not : formula -> formula

val f_inf : num -> num -> formula
val f_sup : num -> num -> formula
(* ... *)

(* 
   Third way: using the internal representation (probably a bad idea to 
   show the internals to users) 
*)
type formula = And(formula, formula) | ...


(**********************************************************************)
type solutions

val solve : formula -> input -> solutions option

val draw : solutions -> output
(* raises Not_found if the set of solutions is empty *)



(**********************************************************************)
type behavior

(* Behavior construction *)

val eps : behavior (* empty behavior *)
val make : formula -> behavior (* a behavior made of one formula *)

val fby : behavior -> behavior -> behavior
val choice : (behavior * int) list -> behavior
val loop : behavior -> behavior
val loop_betwenn : behavior -> int -> int -> behavior

(**********************************************************************)


val step : behavior -> input -> (behavior * solution) option (* Returns None if no transition is possible*)
val step_thick : behavior -> input -> int -> (behavior * solution) list

(* 
   In particular, for all i, i':
      (step eps i) = None
      (step (fst (step (make f) i)) i') = None
*)


(**********************************************************************)

type compiler = Lustre | Scade | C_poc | Stdin (* Default *)
type file = string
type node = string

val plug_sut : file -> node -> compiler -> bool
val plug_oracle : file -> node -> compiler -> bool


(**********************************************************************)

val set_var : var -> value -> behavior -> behavior
(*
   Will replace [var] by [value] in formula appearing in the behavior

   Allow to do all determinitic computations in caml !!

ex: (set_var sqrt_a (sqrt a) behavior)
*)

(**********************************************************************)

val lurette : int -> int -> int -> ... -> unit

(**********************************************************************)

(*
typical main function:

let rec lucky i behavior =
  if i = 0 then () else
    let input = read_input () in 
    let (sol, new_behavior) = step behavior input in
    let output = draw sol in
      print_rif_input input;
      print_rif_output output;
      match new_behavior with
	  None -> ()
        | Some b ->  lucky (i+1) b

*)

(*
examples :

let f1 = formula_parser "a or b and (x>3)"
let b1 = make f1

let b12 = b1 fby b2
let b3 = choice [(f1,3); (f2,1)]

let bl = loop (b12 fby b3)

lucky bl ;;

 *)
