(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License
**-----------------------------------------------------------------------
**
** File: lut4c.ml
** Author: jahier@imag.fr
*)


open Value
let debug = false

let debug_oc = ref stderr
  
let _ =
  if debug then
    debug_oc := open_out "debug_lut4c.log"

(*********************************************************************)
(* Handling Lutin processes *)

(* exported *)
type process_nb = int

(*  *)

type subst_tbl = Value.OfIdent.t
let empty_tbl = Value.OfIdent.empty

type process_info = 
  | Lutin of LutExe.t * LutExe.control_state * LutExe.data_state
  | Empty


type lp_tbl = (int, process_info) Hashtbl.t

let (process_nb_tbl:lp_tbl) = Hashtbl.create 1 

let process_nb_cpt = ref 0
let get_new_process_nb_number () =
  incr process_nb_cpt;
  if debug then (
    output_string !debug_oc (
      "# CAML: get_new_process_nb_number: create a new lucky process (" ^
        (string_of_int !process_nb_cpt) ^ ")\n");
    flush !debug_oc
  );
  !process_nb_cpt


let (add_lp: process_nb -> process_info -> unit) =
  fun lp pi -> 
    Hashtbl.add process_nb_tbl lp pi

let (state_of_lp : process_nb -> process_info) =
  fun lp -> 
    try Hashtbl.find process_nb_tbl lp
    with Not_found -> Empty

let (update_lp : process_nb -> process_info -> unit) =
  fun lp lpi -> 
    Hashtbl.replace process_nb_tbl lp lpi
      
let (remove_lp: process_nb -> unit) =
  fun lp -> 
    Hashtbl.remove process_nb_tbl lp

(* ********************************************************************** *)
(* Types *)
(* ********************************************************************** *)


type subst = Var.subst

(** To indicate whether the point used to perform the step is
    drawn inside, at edges, or at vertices of the convex hull of
    numeric solutions; the step mode is used iff at least one 
    controllable variable is numeric.  
    
    The default step mode is StepInside.
*)
type step_mode = StepInside | StepEdges | StepVertices

(*  ********************************************************************** *)
let lutin_libs = ref []

let (add_lutin_lib : string -> unit) =
  fun f -> 
    lutin_libs := f::!lutin_libs

(* ********************************************************************** *)
(* Make a new Lutin Process *)
(* ********************************************************************** *)

(* exported *)
exception MakeError

let seed_ref = ref None

(* exported *)
let (set_seed: int -> unit) =
  fun seed -> 
    seed_ref := Some seed

(* exported *)
let (make_lutin : string -> string -> process_nb) =
  fun f mnode -> 
    try
      let arg_opt = MainArg.make_opt () in
      let arg_opt = MainArg.set_libs arg_opt !lutin_libs  in

	   (* I defined mine because i need to know the seed that has been drawn 
	      by self_init. *)
      let seed = 
        match !seed_ref with 
          | None -> Random.int 1073741823
          | Some s ->  s
      in
      let _ =
	     (* Initialisation of the random engine *)
        MainArg.set_seed arg_opt (Some seed);
	     Random.init seed ;
	     output_string stderr "#The random engine was initialized with the seed ";
	     output_string stderr ((string_of_int seed) ^ "\n");
	     flush stderr ;
	     
	     Formula_to_bdd.clear_all () ;
	     !Solver.init_snt ()
      in
      let prog = LutExe.make arg_opt [f] mnode in
      let init_ds = { 
        LutExe.ins  = Value.OfIdent.empty; 
        LutExe.outs = Value.OfIdent.empty;
        LutExe.mems = LutExe.get_init_pres prog
      }
      in
      let init_cs = LutExe.get_init_state prog in
      let lp = get_new_process_nb_number () in
	     add_lp lp (Lutin(prog, init_cs, init_ds));
	     lp
    with
	     Failure msg -> 
	       print_string msg;
	       flush stdout;
	       raise MakeError
    
(* exported *)
let (kill_process_nb : process_nb -> unit) =
  fun lp -> 
    remove_lp lp
    

(* ********************************************************************** *)
(* Step function *)
(* ********************************************************************** *)

(* exported *)
let (step: process_nb -> step_mode -> unit) =
  fun lp sm ->
    if debug then (  
      output_string !debug_oc (
        "# CAML: step lp= " ^ (string_of_int lp) ^ " \n" );
      flush !debug_oc;
    );
    match state_of_lp lp with
      | Lutin (prog, cs, ds) ->
          let (cs,ds) = LutExe.step prog cs ds in
	         update_lp lp (Lutin (prog, cs, ds))
              
	           
      | Empty ->
	       failwith "lut4c: Cannot step, no Lutin process found"


(****************************************************************************)

let sol_number = ref Bddd.sol_number
let draw_in_bdd = ref Bddd.draw_in_bdd
let add_snt_entry = ref Bddd.add_snt_entry
let init_snt = ref Bddd.init_snt
let clear_snt = ref Bddd.clear_snt


(*  ********************************************************************** *)
(* Setting the inputs  *)

let (set_input_subst : process_nb -> Var.subst -> unit) =
  fun lp (n,v) -> 
    match state_of_lp lp with
      | Empty -> ()
	   | Lutin(prog, cs, ds) -> 
          let ds = { ds with LutExe.ins = Value.OfIdent.add ds.LutExe.ins (n,v) } in
	         update_lp lp (Lutin (prog, cs, ds))

let (set_input_bool : process_nb -> string -> bool -> unit) =
  fun lp name b ->
    set_input_subst lp (name,Value.B(b))

let (set_input_int : process_nb -> string -> int -> unit) =
  fun lp name i ->
    set_input_subst lp (name,Value.N(Value.I(Num.num_of_int i)))

let (set_input_float : process_nb -> string -> float -> unit) =
  fun lp name f ->
    set_input_subst lp (name,Value.N(Value.F(f)))

(* Getting the outputs *)
let (get_outval_val :  process_nb -> string -> Value.t) =
  fun lp name ->
    match state_of_lp lp with
      | Empty -> failwith 
	       ("lut4c: process " ^ (string_of_int lp) ^ " does not exist")
	   | Lutin(_, _, ds) -> (
          try Value.OfIdent.get ds.LutExe.outs name
	       with Not_found -> 
            let msg = Printf.sprintf "luc4c: '%s' is an unknown output (%s)" name
              (String.concat "," (Value.OfIdent.support ds.LutExe.outs))
            in
	           failwith msg
        )

let (get_output_bool : process_nb -> string -> bool) =
  fun lp name ->
    match get_outval_val lp name with
      | B(b) -> b
      | N(F(f)) -> failwith ("lut4c: "^name^" is not a float")
      | N(I(i)) -> failwith ("lut4c: "^name^" is not an int")

 
let (get_output_int : process_nb -> string -> int) =
  fun lp name ->
    match get_outval_val lp name with
      | B(b) -> failwith ("lut4c: "^name^" is not a bool")
      | N(F(f)) -> failwith ("lut4c: "^name^" is not a float")
      | N(I(i)) -> Util.int_of_num i

 
let (get_output_float : process_nb -> string -> float) =
  fun lp name ->
    if debug then (
      output_string !debug_oc ( 
        "# CAML: get_output_float \"" ^ name ^ "\" (" ^ (string_of_int lp) ^ ") : " ); 
      flush !debug_oc;
    );
    match get_outval_val lp name with
      | B(b) -> failwith ("lut4c: "^name^" is not a bool")
      | N(F(f)) -> 
          if debug then (
 	         output_string !debug_oc (string_of_float f); 
 	         output_string !debug_oc "\n"; 
	         flush !debug_oc
          );
	       f
      | N(I(i)) -> failwith ("lut4c: "^name^" is not an int")

 
    

(*  ********************************************************************** *)

let register () =

  Callback.register "lut4c_make_lutin" make_lutin;
  Callback.register "lut4c_kill_process_nb" kill_process_nb;
  Callback.register "lut4c_step" step;

  Callback.register "lut4c_add_lutin_lib" add_lutin_lib;

  Callback.register "lut4c_set_input_int" set_input_int;
  Callback.register "lut4c_set_input_bool" set_input_bool;
  Callback.register "lut4c_set_input_float" set_input_float;

  Callback.register "lut4c_get_output_int" get_output_int;
  Callback.register "lut4c_get_output_bool" get_output_bool;
  Callback.register "lut4c_get_output_float" get_output_float

  
let _ = register ()
;;
