(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License
**-----------------------------------------------------------------------
**
** File: lustreExp.mli
** Author: jahier@imag.fr
*)

open Type
open Var
open Exp
open Lexeme
open Util

type source_info = Info of int * int

type t =
  | Le_true of  Lexeme.t
  | Le_false of  Lexeme.t
  | Le_infinity of  Lexeme.t
  | Le_iconst of Lexeme.t * int
  | Le_rconst of Lexeme.t * float
  | Le_ident of Lexeme.t * string

  | Le_not of Lexeme.t * t
  | Le_uminus of Lexeme.t * t
  | Le_abs of Lexeme.t * t
  | Le_pre of Lexeme.t * t

  | Le_sum of Lexeme.t * t * t
  | Le_diff of Lexeme.t * t * t
  | Le_prod of Lexeme.t * t * t
  | Le_mod of Lexeme.t * t * t
  | Le_div of Lexeme.t * t * t
  | Le_and of Lexeme.t * t * t
  | Le_or of Lexeme.t * t * t
  | Le_xor of Lexeme.t * t * t
  | Le_impl of Lexeme.t * t * t
  | Le_eq of Lexeme.t * t * t
  | Le_neq of Lexeme.t * t * t
  | Le_sup of Lexeme.t * t * t
  | Le_supeq of Lexeme.t * t * t
  | Le_inf of Lexeme.t * t * t
  | Le_infeq of Lexeme.t * t * t

  | Le_ite of Lexeme.t * t * t * t

  | Le_array of Lexeme.t * t list
  | Le_array_concat of Lexeme.t * t * t
  | Le_array_hat of Lexeme.t * t * t
  | Le_array_access of Lexeme.t * t * t

  | Le_struct of Lexeme.t * (t * t) list
  | Le_struct_access of Lexeme.t * t * t

  | Le_func_call of Lexeme.t * string * t list

type typedef = (string * Type.t)


let (le_to_string : t -> string) =
  fun le ->
    match le with
      | Le_true(lex) -> lex.str
      | Le_false(lex) -> lex.str
      | Le_infinity(lex) -> lex.str
      | Le_iconst(lex, _) -> lex.str
      | Le_rconst(lex, _) -> lex.str
      | Le_ident(lex, _) -> lex.str

      | Le_not(lex , _) -> lex.str
      | Le_uminus(lex , _) -> lex.str
      | Le_abs(lex , _) -> lex.str
      | Le_pre(lex , _) -> lex.str

      | Le_sum(lex , _ , _) -> lex.str
      | Le_diff(lex , _ , _) -> lex.str
      | Le_prod(lex , _ , _) -> lex.str
      | Le_mod(lex , _ , _) -> lex.str
      | Le_div(lex , _ , _) -> lex.str
      | Le_and(lex , _ , _) -> lex.str
      | Le_or(lex , _ , _) -> lex.str
      | Le_xor(lex , _ , _) -> lex.str
      | Le_impl(lex , _ , _) -> lex.str
      | Le_eq(lex , _ , _) -> lex.str
      | Le_neq(lex , _ , _) -> lex.str
      | Le_sup(lex , _ , _) -> lex.str
      | Le_supeq(lex , _ , _) -> lex.str
      | Le_inf(lex , _ , _) -> lex.str
      | Le_infeq(lex , _ , _) -> lex.str

      | Le_ite(lex , _ , _ , _) -> lex.str

      | Le_array(lex , _ ) -> lex.str
      | Le_array_concat(lex , _ , _) -> lex.str
      | Le_array_hat(lex , _ , _) -> lex.str
      | Le_array_access(lex , _ , _) -> lex.str

      | Le_struct(lex , _ ) -> lex.str
      | Le_struct_access(lex , _ , _) -> lex.str
      | Le_func_call (lex,_,_) -> lex.str

(****************************************************************************)

type my_in_channel = { lv_prefix : string ; string : string }

let (type_error : my_in_channel -> Lexeme.t -> string -> unit) =
  fun ic lexeme msg ->
    let (line, col) = Util.from_char_pos_to_line_and_col ic.string lexeme.chstart true in
      print_string (
	"\n*** Type error at token '" ^ lexeme.str ^ "' line " ^
	line ^ " col " ^ col ^ ": " ^ msg ^ ".\n");
      flush stdout

let (syntax_error : my_in_channel -> Lexeme.t -> string -> unit) =
  fun ic lexeme msg ->
    let (line, col) = Util.from_char_pos_to_line_and_col ic.string lexeme.chstart true in
      print_string (
	"\n*** Syntax error at token '" ^ lexeme.str ^ "' line " ^
	line ^ " col " ^ col ^ ": " ^ msg ^ ".\n");
      flush stdout


let (uncompatible_type : my_in_channel -> Lexeme.t -> Type.t -> Type.t
       -> unit) =
  fun ic sinfo t1 t2 ->
    (type_error ic sinfo (
       "types \"" ^ (Type.to_string t1) ^ "\" and \"" ^
       (Type.to_string t2) ^ "\" are not compatible"))

let (not_yet_supported : my_in_channel -> Lexeme.t -> string -> unit) =
  fun ic lexeme str ->
    let (line, col) = Util.from_char_pos_to_line_and_col ic.string lexeme.chstart true in
      print_string (
	"\n*** At token '" ^ lexeme.str ^ "' line " ^
	line ^ " col " ^ col ^ ": " ^ str ^ " is not supported, sorry.\n");
      flush stdout


(****************************************************************************)


let rec (get_position_in_list : 'a -> 'a list -> int) =
  fun x l ->
    match l with
	[] -> assert false
      | y::tail -> if x = y then 0 else 1 + (get_position_in_list x tail)


type vut = (Type.structured * Exp.var_tbl) Util.StringMap.t



let print_tbl tbl =
  print_string ("\n\n");
  (StringMap.iter
     (fun vn opt ->
	print_string (
	  "" ^ vn ^ " -> " ^
	  (match opt with
	     |  (Nu e) -> Exp.num_to_string e
	     |  (Fo f) -> Exp.formula_to_string f
	  ) ^ 
	  "   "))
     tbl
  );
  flush stdout

let print_var_tbl tbl =
  print_string ("\n\n");
  (StringMap.iter
     (fun vn var ->
	print_string (" " ^ vn ^ " -> ");
	Var.print var
     )
     tbl
  );
  flush stdout

let (do_type_identifier : my_in_channel -> Exp.var_tbl -> vut ->
      typedef list -> Exp.ext_func_tbl -> Lexeme.t -> string -> Exp.t * Type.t) =
  fun ic varl vutl tdl ftbl sinfo id ->
    let encode_enum v =
      (*
	     change the type of [v] so that it is an int. Indeed, once the
	     type checking is over, we can forgot the enum original type
	     and work with basic types only, which makes life easier.
      *)
      Var.change_type v IntT
        (* enum type are represented by ints, as usual *)
    in
      try
	     (* (1) Case where [id] is of a simple type, eg, x:int or t.g.d[2]:bool *)
	     let var = 
	       try
	         (Util.mfind id varl) 
	       with Not_found -> 
	         (Util.mfind (ic.lv_prefix ^ id) varl) 
	     in
	       (match (Var.typ var) with
	            BoolT  -> Formu(Bvar(var)), BoolT
	          | IntT   -> Numer(Ivar(var)), IntT
	          | FloatT -> Numer(Fvar(var)), FloatT
	          | UT(EnumT(el)) -> 
		           (Numer(Ivar(encode_enum var))), UT(EnumT(el))		   
		             
	          | UT(ArrayT(_i, _t)) -> raise Not_found
	          | UT(StructT(_fl))   -> raise Not_found 

	       )
      with Not_found ->
	     (*
	       (2) Case where [id] is a user defined type, eg, t:int^3.
	       We flatten the identifiers.
	     *)
	     try
	       let (var_to_f_or_e : string -> Var.name -> Exp.var -> Exp.simple_tbl
		          -> Exp.simple_tbl) =
	         fun prefix vn var acc ->
	           let x =
		          match (Var.typ var) with
		              BoolT  -> Fo(Bvar(var))
		            | IntT   -> Nu(Ivar(var))
		            | FloatT -> Nu(Fvar(var))
		            | UT(EnumT el) -> Nu(Ivar(var))
		            | UT(_) ->
		                (*
			               Variables in [varl] ougth to be basic since
			               LucParse.flatten_user_type recursively flattened
			               user types.
		                *)
		                assert false
	           in
	           let vn' = (Exp.remove_prefix ic.lv_prefix prefix vn) in
		          Exp.add_value vn' x acc
	       in
	       let ut = 
	         try
	           (mfind id vutl)
	         with Not_found ->
	           (mfind (ic.lv_prefix ^ id) vutl)
	       in
	         (match ut with
	            | ArrayT(n, t), id_varl ->
		             (
		               (Liste 
			               (StringMap.fold (var_to_f_or_e id) id_varl 
			                  Exp.empty_simple_tbl)),
		               UT(ArrayT(n, t))
		             )
	            | StructT(l), id_varl ->
		             (
		               (Liste
			               (StringMap.fold (var_to_f_or_e id) id_varl 
			                  Exp.empty_simple_tbl)),
		               UT(StructT(l))
		             )
	            | EnumT(el), id_varl -> 
		             (* id_varl ougth to contain only one element.  We
		                associate it a list (well, a map) of variables
		                to be homogeneous with to the other structured
		                types. Probably it is not a very good idea...
		             *)
		             let var = 
		               try mfind id id_varl 
		               with _ -> 
		                 try mfind (ic.lv_prefix ^ id) id_varl 
		                 with _ -> 
			                assert false
		             in
		               (Numer(Ivar(encode_enum var))), UT(EnumT(el))
	         )
	           
	     with Not_found ->
	       (* (3) Case where [id] is an enum type value *)
	       try
	         let rec search_id_position l =
	           (* search in a vut whether [id] appears as an enum type,
		           and returns its coding (ie, its position in the enum type);
		           raise Not_found otherwise.
	           *)
	           match l with
		            [] -> raise Not_found
		          | (_, UT (EnumT(el)))::tail ->
		              if
		                List.mem id el
		              then
		                (get_position_in_list id el, el)
		              else
		                search_id_position tail
		          | _::tail ->
		              search_id_position tail
	         in
	         let (index, el) = search_id_position tdl in
	           (
		          (Numer (Ival(Num.num_of_int index))),
		          (UT (EnumT el))
	           )
	             

	       with Not_found ->
	         (*  Otherwise, ie, case where [id] is undeclared *)
	         (
	           type_error ic sinfo (id ^ " is used undeclared");
	           (match StringMap.fold (fun _ _ x -> x+1) varl 0 with
		           | 0 -> print_string "*** No variable is declared actually."
		           | 1 -> print_string "*** The only declared variable is:"
		           | _ -> print_string "*** The declared variables are:"
	           );
	           StringMap.iter (fun vn var -> print_string (
				                    "\n\t" ^ vn ^ " of type " ^
				                      (Type.to_string (Var.typ var))))
		          varl;
	           print_string "\n";
	           exit 2
	         )


(****************************************************************************)

let (merge_string_map : 'a Util.StringMap.t -> 'a Util.StringMap.t -> 
       'a Util.StringMap.t) =
  fun map1 map2 -> 
    StringMap.fold (StringMap.add) map1 map2


let rec (do_type : my_in_channel -> Exp.var_tbl -> vut -> typedef list ->
	      Exp.ext_func_tbl -> t -> Exp.var_tbl * Exp.t * Type.t) =
  fun ic varl vutl tdl ftbl le ->
    match le with
      | Le_iconst(sinfo, i) -> varl, Numer(Ival(Num.num_of_int i)), IntT
      | Le_rconst(sinfo, f) -> varl, Numer(Fval(f)), FloatT
      | Le_true(sinfo) -> varl, Formu(True), BoolT
      | Le_false(sinfo) -> varl, Formu(False), BoolT
      | Le_infinity(sinfo) -> varl, Numer(Inf_int), IntT
      | Le_ident(sinfo, id) -> 
	     let e,t = do_type_identifier ic varl vutl tdl ftbl sinfo id in
	     varl, e, t

      | Le_not(sinfo, f) -> (
	     match do_type ic varl vutl tdl ftbl f with
	       | varl, Formu f_not, BoolT -> varl, Formu(Not(f_not)), BoolT
	       | varl, fel, t ->
		      type_error ic sinfo ((Exp.to_string fel) ^ " is not a Boolean");
		      exit 2
	   )
	     
      | Le_uminus(sinfo, num) -> (
	     match do_type ic varl vutl tdl ftbl num with
	       | varl, Numer (Fval f), t -> varl, Numer (Fval (-.f)), t
	       | varl, Numer (Ival i), t -> varl, Numer (Ival (Num.minus_num i)), t
	       | varl, Numer num_minus, t -> varl, Numer (Uminus(num_minus)), t
	       | varl, fel, t ->
		      type_error ic sinfo ((Exp.to_string fel) ^
				                        " is not a numeric expression");
		      exit 2
	   )

      | Le_abs(sinfo, num) -> (
	     match do_type ic varl vutl tdl ftbl num with
	       | varl, Numer num, t       ->
		      let zero = if t = IntT then Ival(Num.num_of_int 0) else Fval(0.0) in
		      let ite = Ite(SupEq(num, zero), num, Uminus(num)) in
		      (varl, Numer ite, t)
	       | varl, fel, t ->
		      type_error ic sinfo ((Exp.to_string fel) ^
				                        " is not a numeric expression");
		      exit 2
	   )
	     
      | Le_pre(sinfo, le) -> (
	     let (make_pre_var : string -> Type.t -> Exp.t option ->
		       Exp.var) =
	       fun n t init ->
            let pvar = Var.make ic.lv_prefix n t Pre in
	             (* only the init field makes sense for pre vars *)
            match init  with
                None  -> pvar
              | Some i ->  set_init pvar i
	     in
	     let (varl2, fel, t) = do_type ic varl vutl tdl ftbl le in
	     match (fel, t) with
	       | Formu (Bvar(var)), t ->
		      let pre_str = Prevar.give_pre_var_name (Var.name var) in
		      let v, varl3 =
		        try ((Util.mfind pre_str varl2), varl2)
		        with Not_found -> 
		          let v = (make_pre_var pre_str (Var.typ var) (Var.init var)) in
		          let varl3 = Util.StringMap.add pre_str v varl2 in
			       v, varl3
		      in
		      varl3, Formu (Bvar v), t
	       | Numer (Fvar(var)), t ->
		      let pre_str = Prevar.give_pre_var_name (Var.name var) in
		      let v, varl3 =
		        try ((Util.mfind pre_str varl2), varl2)
		        with Not_found -> 
		          let v = (make_pre_var pre_str (Var.typ var) (Var.init var)) in
		          let varl3 = Util.StringMap.add pre_str v varl2 in
			       v, varl3
		      in
		      varl3, Numer (Fvar v), t

	       | Numer (Ivar(var)), t ->
		      let pre_str = Prevar.give_pre_var_name (Var.name var) in
		      let v, varl3 =
		        try ((Util.mfind pre_str varl2), varl2)
		        with Not_found -> 
		          let v = (make_pre_var pre_str (Var.typ var) (Var.init var)) in
		          let varl3 = Util.StringMap.add pre_str v varl2 in
			       v, varl3
		      in
		      varl3, Numer (Ivar v), t

	       | Liste val_map, UT(EnumT(el)) ->
		      assert false
              (* 		let val_list =  *)
              (* 		  StringMap.fold (fun vn value acc -> value::acc) val_map []  *)
              (* 		in *)
              (* 		  (match val_list with *)
              (* 		       [Nu(Ivar var)] -> (* that's the way enum are encoded ... *) *)
              (* 			 let pre_str = Prevar.give_pre_var_name (Var.name var) in *)
              (* 			   add_pre_var_names pre_str; *)
              (* 			   Liste  *)
              (* 			     (Exp.add_value  *)
              (* (* 				(remove_prefix (Var.name var)) *) *)
              (* 				(Var.name var) (* XXX *) *)
              (* 				(Nu(Ivar (make_pre_var pre_str  *)
              (* 					    (Var.typ var) (Var.init var)))) *)
              (* 				Exp.empty_simple_tbl), *)
              (* 			   UT(EnumT(el)) *)
              (* 		     | _ ->  *)
              (* 			 (not_yet_supported ic sinfo  *)
              (* 			    "pre over non-atomic expressions"); *)
              (* 			 exit 2 *)
              (* 		  ) *)

	           (* Does not eat any bread ... *)
	       | Numer Inf_int, _ -> varl, fel, t
	       | Numer (Ival i), _ ->  varl, fel, t
	       | Numer (Fval f), _ ->  varl, fel, t
	       | Formu (True), _ -> varl, fel, t
	       | Formu (False), _ -> varl, fel, t


	       | Liste l, t ->
		      not_yet_supported ic sinfo "pre over non-atomic expressions";
		      exit 2
	       | Formu f, _ ->
		      not_yet_supported ic sinfo "pre over non-atomic expressions";
		      exit 2

	       | Numer e, _ ->
		      not_yet_supported ic sinfo "pre over non-atomic expressions";
		      exit 2
	   )
	     
      | Le_sum(sinfo, le1, le2) ->
	     (do_type_num_op2 ic sinfo varl vutl tdl ftbl le1 le2
	        (fun e1 e2 -> Numer(Sum(e1, e2))) true
	        " numeric expression expected")

      | Le_diff(sinfo, le1, le2) ->
	     (do_type_num_op2 ic sinfo varl vutl tdl ftbl le1 le2
	        (fun e1 e2 -> Numer(Diff(e1, e2))) true
	        " numeric expression expected")
	       
      | Le_prod(sinfo, le1, le2) ->
	     (do_type_num_op2 ic sinfo varl vutl tdl ftbl le1 le2
	        (fun e1 e2 -> Numer(Prod(e1, e2))) true
	        " numeric expression expected")

      | Le_mod(sinfo, le1, le2) -> (
	     let (varl2, fel, t) =
	       do_type_num_op2 ic sinfo varl vutl tdl ftbl le1 le2
	         (fun e1 e2 -> Numer(Mod(e1, e2)))  true
	         " integer expression expected"
	     in
	     if
	       t <> IntT
	     then
	       ( uncompatible_type ic sinfo t IntT ; exit 2)
	     else
	       (varl2, fel, t)
	   )

      | Le_div(sinfo, le1, le2) ->
	     (do_type_num_op2 ic sinfo varl vutl tdl ftbl le1 le2
	        (fun e1 e2 -> Numer(Div(e1, e2))) true
	        " numeric expression expected")

      | Le_sup(sinfo, le1, le2) ->
	     (do_type_num_op2 ic sinfo varl vutl tdl ftbl le1 le2
	        (fun e1 e2 -> Formu(Sup(e1, e2))) false
	        " numeric expression expected")

      | Le_supeq (sinfo, le1, le2) ->
	     (do_type_num_op2 ic sinfo varl vutl tdl ftbl le1 le2
	        (fun e1 e2 -> Formu(SupEq(e1, e2))) false
	        " numeric expression expected")

      | Le_inf(sinfo, le1, le2) ->
	     (do_type_num_op2 ic sinfo varl vutl tdl ftbl le1 le2
	        (fun e1 e2 -> Formu(Inf(e1, e2))) false
	        " numeric expression expected")

      | Le_infeq(sinfo, le1, le2) ->
	     (do_type_num_op2 ic sinfo varl vutl tdl ftbl le1 le2
	        (fun e1 e2 -> Formu(InfEq(e1, e2))) false
	        " numeric expression expected")


      | Le_and(sinfo, le1, le2) ->
	     (do_type_bool_op2 ic sinfo varl vutl tdl ftbl le1 le2
	        (fun f1 f2 -> Formu(And(f1, f2)))
	        " Boolean expression expected")

      | Le_or(sinfo, le1, le2) ->
	     do_type_bool_op2 ic sinfo varl vutl tdl ftbl le1 le2
	       (fun f1 f2 -> Formu(Or(f1, f2)))
	       " Boolean expression expected"

      | Le_xor(sinfo, le1, le2) ->
	     do_type_bool_op2 ic sinfo varl vutl tdl ftbl le1 le2
	       (fun f1 f2 -> Formu(Xor(f1, f2)))
	       " Boolean expression expected"

      | Le_impl(sinfo, le1, le2) ->
	     do_type_bool_op2 ic sinfo varl vutl tdl ftbl le1 le2
	       (fun f1 f2 -> Formu(Impl(f1, f2)))
	       " Boolean expression expected"

      | Le_eq(sinfo, le1, le2) ->
	     do_type_bool_polymorph_op2 ic sinfo varl vutl tdl ftbl le1 le2
	       (fun f1 f2 -> (EqB(f1, f2)))
	       (fun e1 e2 -> (Eq(e1, e2)))

      | Le_neq(sinfo, le1, le2) ->
	     do_type_bool_polymorph_op2 ic sinfo varl vutl tdl ftbl le1 le2
	       (fun f1 f2 -> (Not(EqB(f1, f2))))
	       (fun e1 e2 -> (Not (Eq(e1, e2))))
	       
      | Le_ite(sinfo, le1, le2, le3) -> (
	     match (do_type ic varl vutl tdl ftbl le1) with
	       | varl, Formu cond, _ -> (
		      let (varl2, exp2, t2) = do_type ic varl vutl tdl ftbl le2 in
		      let (varl3, exp3, t3) = do_type ic varl2 vutl tdl ftbl le3 in
		      if
		        t2 <> t3
		      then
		        (uncompatible_type ic sinfo t2 t3 ; exit 2)
		      else
		        match (exp2, exp3) with
		          | (Numer num1), (Numer num2) ->
			         varl3, Numer (Ite(cond, num1, num2)), t2

		          | (Formu f1), (Formu f2) ->
			         varl3, Formu (IteB(cond, f1, f2)), BoolT
			           
		          |  (Liste l1), (Liste l2) ->
			         varl3, 
			         (Liste
			            (StringMap.fold
				            (fun vn x1 acc -> 
				              try
				                let x2 = mfind vn l2 in
				                let f = 
					               (match x1, x2 with
					                 | Nu e1, Nu e2 -> Nu(Ite(cond, e1, e2))
					                 | Fo f1, Fo f2 -> Fo(IteB(cond, f1, f2))
					                 | _, _ -> assert false
					               )
				                in
					             Exp.add_value vn f acc
				              with _ -> assert false
				            )
				            l1
				            Exp.empty_simple_tbl
			            )
			         ),
			         t2
			           
		          |  (e1, e2) ->
			         assert false
	       )
	       | _, Numer e , t ->
		      uncompatible_type ic sinfo t BoolT;
		      exit 2
	       | _, Liste _, t ->
		      uncompatible_type ic sinfo t BoolT;
		      exit 2
	   )

      | Le_array(sinfo, le_list) ->
	     let size = List.length le_list in
	     let le1 =
	       try
	         List.hd le_list
	       with _ ->
	         type_error ic sinfo " Empty array"; (* Should it be an error? *)
	         exit 2
	     in
	     let array_type =
	         (* look at the first element to know the type of the array to be
	            able to check that the array is of homogeneous type then *)
	       match (do_type ic varl vutl tdl ftbl le1) with
		        _, Formu f, t -> t
	         | _, Numer e, t -> t
	         | _, Liste l, t -> t
	     in
	     let varl2, l, _ =
	       (List.fold_left
	          (fun (varl_acc,lacc, k) le ->
		         let k_str = ("[" ^ (string_of_int k) ^ "]") in
		         match (do_type ic varl_acc vutl tdl ftbl le) with
			          varl_acc2, Formu f, _ ->
			            if
			              array_type <> BoolT
			            then
			              ( uncompatible_type ic sinfo array_type BoolT;
			                exit 2 )
			            else
			              (varl_acc2, Exp.add_value k_str (Fo f) lacc, k+1)
			                
		           | varl_acc2, Numer e, t ->
			          if
			            array_type <> t
			          then
			            ( uncompatible_type ic sinfo array_type t; exit 2 )
			          else
			            (varl_acc2, Exp.add_value k_str (Nu e) lacc, k+1)
			              
		           | varl_acc2, Liste l, t ->
			          if
			            array_type <> t
			          then
			            (
			              uncompatible_type ic sinfo array_type t;
			              exit 2
			            )
			          else
			            (
			              varl_acc2, 
			              (StringMap.fold 
				              (fun str value acc -> 
				                Exp.add_value (k_str^str) value acc) 
				              l 
				              lacc
			              ),
			              k+1
			            )
	          )
	          (varl, Exp.empty_simple_tbl, 0)
	          le_list
	       )
	     in
	     varl,
	     (Liste l),
	     UT(ArrayT(size, array_type))
	       
      | Le_array_hat(sinfo, elt, n) ->
	     ( match n with
		      Le_iconst(_, i) ->
		        let (varl2, exp, array_type) = 
		          (do_type ic varl vutl tdl ftbl elt) 
		        in
		        let lexp = (Util.unfold (fun x -> x) exp i) in
		        varl2,
		        (Liste (fst
		                  (List.fold_left
			                  (fun (acc, k) exp ->
			                    let k_str = ("[" ^ (string_of_int k) ^ "]") in
			                    match exp with
				                   | Formu f -> (Exp.add_value k_str (Fo f) acc), (k+1)
				                   | Numer e -> (Exp.add_value k_str (Nu e) acc), (k+1)
				                   | Liste l ->
				                     (StringMap.fold
					                     (fun str value acc ->
					                       Exp.add_value (k_str^str) value acc)
					                     l
					                     acc
				                     ),
				                     k+1
			                  )
			                  (Exp.empty_simple_tbl, 0)
			                  lexp
		                  )	  
		         )),
		        UT(ArrayT(i, array_type))
		          
	       | le ->
		      type_error ic sinfo ((le_to_string le) ^ " is not an integer");
		      exit 2
	     )

      | Le_struct(sinfo, le_list) ->
	     let fel_list =
	       (List.map
	          (fun (f, v) ->
		         match f with
		             Le_ident(_,id) ->
			            let _varl2, e, t = (do_type ic varl vutl tdl ftbl v) in
			            (id, (e,t))
		           | _ ->
			          (type_error ic sinfo (
			            (le_to_string f) ^
	                    ", as a struct field, ougth to be an ident"));
			          exit 2
	          )
	          le_list
	       )
	     in
	     let struct_type =
	       (StructT
	          (List.map
		          (fun (field_name, le) -> (field_name, snd le))
		          fel_list ))
	     in
	     varl,
	     (Liste
	        (List.fold_left
		        (fun acc (field_name, le) ->
		          match fst le with
		              Formu f -> Exp.add_value ("." ^ field_name) (Fo f) acc
		            | Numer e -> Exp.add_value ("." ^ field_name) (Nu e) acc
		            | Liste l -> 
			           (StringMap.fold 
			              (fun str value acc2 -> 
			                Exp.add_value 
				               ("." ^ field_name ^ str)
				               value 
				               acc2)
			              l
			              acc
			           )
		        )
		        Exp.empty_simple_tbl
		        fel_list
	        )
	     ),
	     (UT struct_type)

      | Le_array_access(sinfo, e, ei) ->
	     let id = do_type_access ic sinfo le in 
	     let e ,t = do_type_identifier ic varl vutl tdl ftbl sinfo id in
	     varl, e, t
	       
      | Le_struct_access(sinfo, s, ef) ->
	     let id = do_type_access ic sinfo le in
	     let e ,t = do_type_identifier ic varl vutl tdl ftbl sinfo id in
	     varl, e, t
	       
      | Le_array_concat(sinfo, le1, le2) ->
	     let varl2, ex1, ty1 = do_type ic varl vutl tdl ftbl le1 in
	     let varl3, ex2, ty2 = do_type ic varl2 vutl tdl ftbl le2 in
	     ( match ((ex1, ty1), (ex2, ty2)) with
		    | (Liste l1, UT(ArrayT(s1, t1))), (Liste l2, UT(ArrayT(s2, t2))) ->
		      if
		        t1 <> t2
		      then
		        ( uncompatible_type ic sinfo t1 t2 ; exit 2)
		      else
		        let val_map = 
		          (StringMap.fold 
			          (fun str value acc ->
			            try
			              let l = (String.length str) in
			              let i = Str.search_backward (Str.regexp "\\[") str l in
			              let index_str = String.sub str (i+1) (l-i-2) in 
			              let index = int_of_string index_str in
			              let i_str = (string_of_int (index+s1)) in
				           Exp.add_value (i_str^str) value acc 
			            with _  -> 
			              assert false
			          )
			          l2 
			          l1
		          )
		        in
		        (
			       varl3,
			       Liste (val_map), 
			       UT(ArrayT(s1+s2, t1))
		        )
		          
	       | _, _ ->
		      type_error ic sinfo (" only arrays can be concatened");
		      exit 2
	     )

      | Le_func_call(sinfo, id, args) -> 
	     if 
	       List.mem id ["gauss_continue";"gauss_stop";
			              "interval_continue";"interval_stop"] &&
	         List.length args = 3
	     then
	         (*  Pre-defined function call *)
	       match args with
	         | [a1;a2;a3]  -> 
		        let varl1, ex1, ty1 = do_type ic varl vutl tdl ftbl a1 in
		        let varl2, ex2, ty2  = do_type ic varl1 vutl tdl ftbl a2 in
		        let varl3, ex3, ty3  = do_type ic varl2 vutl tdl ftbl a3 in
		        (
		          match ((ex1, ty1),(ex2, ty2),(ex3, ty3)) with
		            | ((Numer exp_a1, IntT), 
		               (Numer exp_a2, IntT), 
		               (Numer exp_a3, IntT)) -> 
			           let num =
			             match id with
			               | "gauss_continue" -> Gcont(exp_a1, exp_a2, exp_a3)
			               | "gauss_stop"-> Gstop(exp_a1, exp_a2, exp_a3)
			               | "interval_continue"  -> Icont(exp_a1, exp_a2, exp_a3)
			               | "interval_stop" -> Istop(exp_a1, exp_a2, exp_a3)
			               | _ -> assert false
			           in 
			           varl3, Numer num, IntT
			             
		            | _,_,_ ->
			           (
			             type_error ic sinfo 
			               (" in the function call to " ^ id ^ 
			                   ", 3 integers are expected");
			             exit 2
			           )
		        )
	         | _ -> 
		        assert false
	     else
	       try
              (* 	      let pr_arg arg = print_string (le_to_string arg) in *)
	         
	         let (func_type, (lib_name, lib)) = Util.mfind id ftbl in
	         let rec split_list acc l = 
		        match l with 
		            [] -> assert false
		          | [a] -> List.rev acc, a
		          | a::tl -> split_list (a::acc) tl
	         in
	         let (type_args, type_res) = split_list [] func_type in
		      
		      if (List.length type_args) <> (List.length args) then
		        (
		          type_error ic sinfo 
		            (" in the function call to " ^ id ^ 
		                ": bad number of arguments (" ^
		                (string_of_int (List.length args)) ^ " whereas " ^
		                (string_of_int (List.length type_args)) ^ " was expected)");
		          exit 2
		        );
		      let varl2, exp_args_rev =
		        List.fold_left2 
		          (fun (varl_acc,acc) arg decl_type_arg -> 
		            let (varl_acc2, exp_arg, type_arg) = 
			           do_type ic varl_acc vutl tdl ftbl arg 
		            in
			         if type_arg <> decl_type_arg then
			           (
			             type_error ic sinfo 
			               (" in the function call to " ^ id ^ ", " ^
			                   (Type.to_string decl_type_arg) ^ 
			                   " was expected, and " ^ (Type.to_string type_arg) ^
			                   " was inferred");
			             exit 2
			           )
			         else
			           varl_acc2, (exp_arg::acc)
		          )
		          (varl,[])
		          args 
		          type_args
		      in
		      let exp_args = List.rev exp_args_rev in
		      
		      let cfunc = 
		        try
                Ezdl.dlsym lib id  
		        with Failure errmsg -> 
		          print_string (errmsg ^ "\n");
		          flush stdout;
		          exit 2
		      in		      
		      ( match type_res with
			       IntT -> 
			         varl2,
			         Numer(IFC(id, cfunc, type_args, lib_name, exp_args)), 
			         type_res
		        | FloatT -> 
			       varl2,
			       Numer(FFC(id, cfunc, type_args, lib_name, exp_args)), 
			       type_res
		        | _  ->  assert false
		      )    
	       with 
            | Not_found ->
	             failwith ("*** Function " ^ id ^ " unknown.\n")
            | e -> 
	           output_string stdout (Printexc.to_string e);
	           flush stdout;
              assert false

	             
(* 	    print_string ("\n"^ id ^ "("); *)
(* 	    pr_arg (List.hd args); *)
(* 	    List.iter (fun arg ->  print_string ","; pr_arg arg) (List.tl args); *)
(* 	    print_string ")\n"; *)
(* 	    flush stdout; *)


	             

and
    (do_type_access : my_in_channel -> Lexeme.t -> t -> string) =
  fun ic sinfo le ->
    (* When an array or a struct access is reached, this function is
       used instead of do_type. like do_type, It "parses" [le], but
       it returns the top-level identifier, e.g., "str.f[3][s.i]"
       instead of a expr. *)
    match le with
      | Le_iconst(sinfo, i) -> string_of_int i
      | Le_rconst(sinfo, f) -> string_of_float f
      | Le_ident(sinfo, id) -> id
      | Le_struct_access(sinfo, s, ef) ->
	     ((do_type_access ic sinfo s) ^ "." ^ (do_type_access ic sinfo ef))
      | Le_array_access(sinfo, e, ei) ->
	     ((do_type_access ic sinfo e) ^ "[" ^ (do_type_access ic sinfo ei) ^ "]")
      | _ ->
	     syntax_error ic sinfo (
	       " such expressions are not supported, sorry." ^
	         " Please use an intermediary variable to access to this element");
	     exit 2
(* This means that i raise an error from expressions such
	as: "([true, false, false])[2]", which is no more than an
	obfuscated way of writting "false"...

	XXX Should i support that kind of access?
*)
and
    (do_type_num_op2 : my_in_channel -> Lexeme.t -> Exp.var_tbl -> vut ->
     typedef list -> Exp.ext_func_tbl -> t -> t -> (Exp.num -> Exp.num -> Exp.t)
     -> bool -> string -> Exp.var_tbl * Exp.t * Type.t) =
  fun ic sinfo varl vutl tdl ftbl le1 le2 op is_num_op msg ->
    let (varl1, ex1, ty1) = do_type ic varl vutl tdl ftbl le1 in
    let (varl2, ex2, ty2) = do_type ic varl1 vutl tdl ftbl le2 in
    match ((ex1, ty1), (ex2, ty2)) with
	   | ((Numer e1, t1), (Numer e2, t2)) ->
	     if
	       t1 <> t2
	     then
	       (uncompatible_type ic sinfo t1 t2 ; exit 2)
	     else
	       varl2, (op e1 e2), (if is_num_op then t1 else BoolT)
	   |  (e1, t1), (_, t2) ->
	     if
	       t1 <> t2
	     then
	       uncompatible_type ic sinfo t1 t2
	     else
	       type_error ic sinfo msg;
	     exit 2
and
    (do_type_bool_op2 : my_in_channel -> Lexeme.t -> Exp.var_tbl -> vut ->
     typedef list -> Exp.ext_func_tbl -> t -> t -> 
     (Exp.formula -> Exp.formula -> Exp.t) -> string ->
     Exp.var_tbl * Exp.t * Type.t) =
  fun ic sinfo varl vutl tdl ftbl le1 le2 op msg ->
    let (varl1, ex1, ty1) = do_type ic varl vutl tdl ftbl le1 in
    let (varl2, ex2, ty2) = do_type ic varl1 vutl tdl ftbl le2 in
    match ((ex1, ty1), (ex2, ty2)) with
	   | ((Formu f1, t1), (Formu f2, t2)) ->
	     if
	       t1 <> BoolT
	     then
	       (
		      uncompatible_type ic sinfo t1 BoolT ;
		      exit 2
	       )
	     else if
	         t2 <> BoolT
	     then
	       (
	         uncompatible_type ic sinfo t2 BoolT ;
		      exit 2
	       )
	     else
	       varl2, (op f1 f2), BoolT
		      
	   |  (e1, t1), (_, t2) ->
	     if
	       t1 <> t2
	     then
	       uncompatible_type ic sinfo t1 t2
	     else
	       type_error ic sinfo msg;
	     exit 2
	       
and
    (do_type_bool_polymorph_op2 : my_in_channel -> Lexeme.t -> Exp.var_tbl ->
     vut -> typedef list -> Exp.ext_func_tbl -> t -> t ->
     (Exp.formula -> Exp.formula -> Exp.formula) ->
     (Exp.num -> Exp.num -> Exp.formula) -> Exp.var_tbl * Exp.t * Type.t) =
  fun ic sinfo varl vutl tdl ftbl le1 le2 opB opN ->
    let (varl1, ex1, ty1) = do_type ic varl vutl tdl ftbl le1 in
    let (varl2, ex2, ty2) = do_type ic varl1 vutl tdl ftbl le2 in
    match ((ex1, ty1), (ex2, ty2)) with
      | ((Numer e1, t1), (Numer e2, t2)) ->
	     if
	       t1 <> t2
	     then
	       (uncompatible_type ic sinfo t1 t2 ; exit 2)
	     else
	       varl2, Formu (opN e1 e2), BoolT
	         
      | ((Formu f1, _), (Formu f2, _)) ->
	     varl2, Formu (opB f1 f2), BoolT
	       
      | (Liste (l1), t1), (Liste (l2), t2) ->
	         (* For lists of the same size, consider the "and" of each elt *)
	     if
	       t1 <> t2
	     then
	       (uncompatible_type ic sinfo t1 t2; exit 2)
	     else
	       let f fe1 fe2 =
	         match fe1, fe2 with
		          (Fo f1, Fo f2) -> (opB f1 f2)
		        | (Nu e1, Nu e2) -> (opN e1 e2)
		        | _ ->
		          (type_error ic sinfo "");
		          exit 2
	       in
	       (try
		       varl2,
		      (Formu
		         (StringMap.fold
		            (fun str fe1 acc ->
			           let fe2 = mfind str l2 in
			           (And(acc, f fe1 fe2))
		            )
		            l1
		            True
		         )
		      ), 
		      BoolT
	        with 
		         _ ->
		           print_tbl l1; 
		           print_tbl l2; 
		           flush stdout;
		           assert false
	       )

      |  (e1, t1), (_, t2) ->
	     uncompatible_type ic sinfo t1 t2 ;
	     exit 2

(****************************************************************************)
(****************************************************************************)
(****************************************************************************)
(****************************************************************************)
(****************************************************************************)
 
(* Infer the type type of a lustreExp.t 
   Work with basic types only.
 *)

(* [infer_type le t] returns also the list of var id with their types *)

(* infer type : bool, int, float numeric or unknown *)
type itype = B | I | F | U | N

let itype_to_type x =
  match x with
      B -> BoolT
    | I -> IntT
    | F -> FloatT 
    | U
    | N -> failwith "Fail to infer the type: please declare your variables"



let itype_to_string x =
  match x with
      B -> "bool"
    | I -> "int"
    | F -> "float" 
    | N -> "num"
    | U -> "unknown"


let type_to_itype t =
  match t with
      BoolT -> B
    | IntT -> I
    | FloatT -> F
    | UT _ -> failwith "Structured type not supported"


let rec compute_less_general_type t1 t2 =
  match (t1, t2) with
      x,y when x = y -> Some x 
    | U,x -> Some x
    | N,F -> Some F
    | N,I -> Some I

    | N,B -> None
    | F,B -> None
    | F,I -> None
    | B,I -> None

    | x, y -> compute_less_general_type y x


(** raises an error if the 2 types are not compatible *)
let (check_type : Lexeme.t -> itype -> itype -> unit) =
  fun sinfo t1 t2 ->
    match compute_less_general_type t1 t2 with
      | Some _  -> ()
      | None -> 
	  failwith ("*** Type error: at lexeme " ^ sinfo.str ^
		    " (char " ^ (string_of_int sinfo.chstart) ^ ") of type " ^
		    (itype_to_string t1) ^ " whereas an expression of type " ^
		    (itype_to_string t2) ^ " was expected.\n")
	

let debug = false

let rec (infer_type_do : t -> itype -> var list -> 
	       Exp.t * itype * var list) =
  fun le typ acc ->
    if debug then
      (
	     print_string ("infer_type_do: "
		                ^ (le_to_string le) ^ " : " 
		                ^ (itype_to_string typ) ^ "\n");
	     flush stdout
      );
    match le with

      | Le_iconst(sinfo, i) -> 
	       check_type sinfo  I typ;
	       Numer(Ival(Num.num_of_int i)), I, acc

      | Le_rconst(sinfo, f) -> 
	       check_type sinfo F typ;
	       Numer(Fval(f)), F, acc

      | Le_true(sinfo) -> 
	       check_type sinfo  B typ;
	       Formu(True), B, acc

      | Le_false(sinfo) -> 
	       check_type sinfo B typ;
	       Formu(False), B, acc

      | Le_infinity(sinfo) -> 
	       assert false

      | Le_func_call(sinfo, f, args) -> 
	       (* cf the do_type rule *)
	       assert false

      | Le_ident(sinfo, id) -> 
	       (   
	         try 
	           let var = List.find (fun var -> (Var.name var) = id) acc in
	           let typ2 = type_to_itype (Var.typ var) in
	     	       (match compute_less_general_type typ typ2 with
		               Some t -> 
		                 (match t with 
			                   B -> Formu(Bvar(var)), B, acc
			                 | I -> Numer(Ivar(var)), I, acc
			                 | F 
			                 | U 
			                 | N -> 
			                     (* By default, we suppose that undeclared 
				                     id are float.*)
			                     Numer(Fvar(var)), F, acc
		                 )
		             | None -> 
		                 check_type sinfo typ typ2;
		                 assert false
		          )
	         with 
		          Not_found -> 
		            (match typ with
		                 B -> 
			                let v = Var.make "" id BoolT Var.Output in
			                  Formu(Bvar(v)), B, v::acc
		               | I -> 
			                let v = Var.make "" id IntT Var.Output in 
			                  Numer(Ivar(v)), I, v::acc
		               | F 
		               | U 
		               | N -> 
			                (* By default, we suppose that undeclared id are float.*)
			                let v = Var.make "" id FloatT Var.Output in
			                  Numer(Fvar(v)), F, v::acc
		            )
	           | e -> 
		            failwith ("*** Type error: " ^ (Printexc.to_string e) ^ "\n")
		              
	       )
      | Le_not(sinfo, le_f) -> 
	       check_type sinfo B typ;
	       let f, _, acc1 = infer_type_do le_f typ acc in
	         (
	           match f with
		            Formu f' -> Formu(Not(f')), B, acc1
		          | _ -> assert false
	         )

      | Le_uminus(sinfo, num) -> 
	       check_type sinfo N typ;
	       let (e,it,acc2) = infer_type_do num typ acc in
	         (match e with
	            | Numer (Fval f)-> Numer (Fval (-.f)), F, acc2
	            | Numer (Ival i)-> Numer (Ival (Num.minus_num i)), I, acc2
	            | Numer num_minus -> Numer (Uminus(num_minus)),it, acc2
	            | _  -> 
		             check_type sinfo N typ;
		             assert false
	         )
	           
      | Le_abs(sinfo, num) ->
	       check_type sinfo N typ;
	       let (e,it,acc2) = infer_type_do num typ acc in
	         (match e with
	            | Numer num -> 
		             let zero = if typ = I then Ival(Num.num_of_int 0) else Fval(0.0) in
		             let ite = Ite(SupEq(num, zero), num, Uminus(num)) in
		               (Numer ite)

	            | _  -> 
		             check_type sinfo N typ;
		             assert false
	         ), it, acc2
	           
      | Le_pre(sinfo, le) -> 
	       failwith "pre is not implemented"
	         
      | Le_sum(sinfo, le1, le2) ->
	       check_type sinfo N typ;
	       let (e1, it1, acc1) = infer_type_do le1 typ acc in
	       let (e2, it2, acc2) = infer_type_do le2 it1 acc1 in
	         (
	           match e1,e2 with
		            Numer e1', Numer e2' -> Numer(Sum(e1', e2')), it2, acc2
		          | _ -> assert false
	         )
	           
      | Le_diff(sinfo, le1, le2) ->
	       check_type sinfo N typ;
	       let (e1, it1, acc1) = infer_type_do le1 typ acc in
	       let (e2, it2, acc2) = infer_type_do le2 it1 acc1 in
	         (
	           match e1,e2 with
		            Numer e1', Numer e2' -> Numer(Diff(e1', e2')), it2, acc2 
		          | _ -> assert false
	         )
	  	        
      | Le_prod(sinfo, le1, le2) ->
	       check_type sinfo N typ;
	       let (e1, it1, acc1) = infer_type_do le1 typ acc in
	       let (e2, it2, acc2) = infer_type_do le2 it1 acc1 in
	         (
	           match e1,e2 with
		            Numer e1', Numer e2' -> Numer(Prod(e1', e2')), it2, acc2 
		          | _ -> assert false
	         )

      | Le_mod(sinfo, le1, le2) ->
	       check_type sinfo I typ;
	       let (e1, _,acc1) = infer_type_do le1 I acc in
	       let (e2, _,acc2) = infer_type_do le2 I acc1 in
	         (
	           match e1,e2 with
		            Numer e1', Numer e2' -> Numer(Mod(e1', e2')), I, acc2 
		          | _ -> assert false
	         )

      | Le_div(sinfo, le1, le2) ->
	       check_type sinfo I typ;
	       let (e1, _, acc1) = infer_type_do le1 I acc in
	       let (e2, _, acc2) = infer_type_do le2 I acc1 in
	         (
	           match e1,e2 with
		            Numer e1', Numer e2' -> Numer(Div(e1', e2')), I, acc2
		          | _ -> assert false
	         )

      | Le_sup(sinfo, le1, le2) ->
	       check_type sinfo B typ;
	       let (e1, it1, acc1) = infer_type_do le1 N acc in
	       let (e2, it2, acc2) = infer_type_do le2 it1 acc1 in
	         check_type sinfo it1 it2;
	         (
	           match e1,e2 with
		            Numer e1', Numer e2' -> Formu(Sup(e1', e2')), B, acc2
		          | _ -> assert false
	         )

      | Le_supeq (sinfo, le1, le2) ->
	       check_type sinfo B typ;
	       let (e1, it1, acc1) = infer_type_do le1 N acc in
	       let (e2, it2, acc2) = infer_type_do le2 it1 acc1 in
	         check_type sinfo it1 it2;
	         (
	           match e1,e2 with
		            Numer e1', Numer e2' -> Formu(SupEq(e1', e2')), B, acc2
		          | _ -> assert false
	         )

      | Le_inf(sinfo, le1, le2) ->
	       check_type sinfo B typ;
	       let (e1, it1, acc1) = infer_type_do le1 N acc in
	       let (e2, it2, acc2) = infer_type_do le2 it1 acc1 in
	         (
	           match e1,e2 with
		            Numer e1', Numer e2' -> Formu(Inf(e1', e2')), B, acc2
		          | _ -> assert false
	         )
	           
      | Le_infeq(sinfo, le1, le2) ->
	       check_type sinfo B typ;
	       let (e1, it1, acc1) = infer_type_do le1 N acc in
	       let (e2, it2, acc2) = infer_type_do le2 it1 acc1 in
	         (
	           match e1,e2 with
		            Numer e1', Numer e2' -> Formu(InfEq(e1', e2')), B, acc2
		          | _ -> assert false
	         )

      | Le_and(sinfo, le1, le2) ->
	       check_type sinfo B typ;
	       let (e1, _, acc1) = infer_type_do le1 B acc in
	       let (e2, _, acc2) = infer_type_do le2 B acc1 in
	         (
	           match e1,e2 with
		            Formu e1', Formu e2' -> Formu(And(e1', e2')), B, acc2
		          | _ -> assert false
	         )

      | Le_or(sinfo, le1, le2) ->
	       check_type sinfo B typ;
	       let (e1, _, acc1) = infer_type_do le1 B acc in
	       let (e2, _, acc2) = infer_type_do le2 B acc1 in
	         (
	           match e1,e2 with
		            Formu e1', Formu e2' -> Formu(Or(e1', e2')), B, acc2
		          | _ -> assert false
	         )

      | Le_xor(sinfo, le1, le2) ->
	       check_type sinfo B typ;
	       let (e1, _, acc1) = infer_type_do le1 B acc in
	       let (e2, _, acc2) = infer_type_do le2 B acc1 in
	         (
	           match e1,e2 with
		            Formu e1', Formu e2' -> Formu(Xor(e1', e2')), B, acc2
		          | _ -> assert false
	         )

      | Le_impl(sinfo, le1, le2) ->
	       check_type sinfo B typ;
	       let (e1, _, acc1) = infer_type_do le1 B acc in
	       let (e2, _, acc2) = infer_type_do le2 B acc1 in
	         (
	           match e1,e2 with
		            Formu e1', Formu e2' -> Formu(Impl(e1', e2')), B, acc2
		          | _ -> assert false
	         )


      | Le_eq(sinfo, le1, le2) ->
	       check_type sinfo B typ;
	       let (e1, it1, acc1) = infer_type_do le1 U acc in
	       let (e2, it2, acc2) = infer_type_do le2 it1 acc1 in
	         (
	           match e1,e2 with
		            Formu e1', Formu e2' -> Formu(EqB(e1', e2')), B, acc2
		          | Numer e1', Numer e2' -> Formu(Eq(e1', e2')), B, acc2
		          | _  -> assert false
	         )
	           
      | Le_neq(sinfo, le1, le2) ->
	       check_type sinfo B typ;
	       let (e1, it1, acc1) = infer_type_do le1 U acc in
	       let (e2, it2, acc2) = infer_type_do le2 it1 acc1 in
	         (
	           match e1,e2 with
		            Formu e1', Formu e2' -> Formu(Not(EqB(e1', e2'))), B, acc2
		          | Numer e1', Numer e2' -> Formu(Not(Eq(e1', e2'))), B, acc2
		          | _  -> assert false
	         )
	           
      | Le_ite(sinfo, le1, le2, le3) -> 
	       let (e1, it1, acc1) = infer_type_do le1 B acc in
	       let (e2, it2, acc2) = infer_type_do le2 typ acc1 in
	       let (e3, it3, acc3) = infer_type_do le3 it2 acc2 in
	         (
	           match e1,e2,e3 with
		            Formu e1', Formu e2', Formu e3' -> Formu(IteB(e1', e2', e3')), B, acc3
		          | Formu e1', Numer e2', Numer e3' -> Numer(Ite(e1', e2', e3')), it2, acc3
		          | _  -> assert false
	         )

      | Le_array(sinfo, le_list) -> failwith "array not supported"
      | Le_array_hat(sinfo, elt, n) -> failwith "array not supported"
      | Le_struct(sinfo, le_list) -> failwith "struct not supported"
      | Le_array_access(sinfo, e, ei) -> failwith "array not supported"
      | Le_struct_access(sinfo, s, ef) -> failwith "struct not supported"
      | Le_array_concat(sinfo, le1, le2) -> failwith "array not supported"


let rec (infer_type : t -> var list -> Exp.t * var list) =
  fun le vars ->
(*     let ivars = List.map (fun (str, t) -> (str, (type_to_itype t))) vars in *)
(*     let (exp, it, ivars') = infer_type_do le B vars in *)
(*       (exp, List.map (fun (str, t) -> (str, (itype_to_type t))) ivars') *)
    let (exp, it, vars') =  infer_type_do le B vars in
      exp, vars'
