(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License
**-----------------------------------------------------------------------
**
** File: luckyDraw.ml
** Author: jahier@imag.fr
*)

open Prog
open Type
open Exp

type value = F of float | I of int | B of bool

type var_name = string
type var_type = Float | Int | Bool
type subst = var_name * value

type solution = subst list

(* Abstract type representing the solution of the formula *)
type solutions_set = {
  bdd  : Bdd.t; 
  comb : Bdd.t; (* could be recomputed from the bdd field *)
  bools_tg : Exp.formula ; (* Boolean var to generate *) 
  nums_tg : Exp.var list; (* Numeric var to generate *)		
  state : Prog.state
}
    

type bool_expr = Exp.formula * var list
type int_expr  = Exp.num * var list
type float_expr  = Exp.num * var list

(** Pretty-print *)
let (bool_expr_to_string : bool_expr -> string) =
  fun f ->
    "\nConstraint: \n\t" ^
    (Exp.formula_to_string (fst f)) ^
    "\nVariables: \n\t" ^
    (List.fold_left  
       (fun acc var ->  
	  (Var.name var) ^ " : " ^ 
	  (Type.to_string (Var.typ var)) ^"\n\t" ^ 
	  acc ) 
       "\n"
       (snd  f)
    ) 
  
let value2num = function
    B(x) -> Value.B(x)
  | I(x) -> Value.N(Value.I(Num.Int x))
  | F(x) -> Value.N(Value.F(x))

(* let (num2value : Value.t -> value) =  *)
(*   fun v ->  *)
(*     match v with *)
(* 	Value.B(x) -> B(x) *)
(*       | Value.N(Value.I(x)) -> I(x) *)
(*       | Value.N(Value.F(x)) -> F(x) *)

let (num2value : Value.num -> value) = 
  fun v -> 
    match v with
      | Value.I(x) -> I(Util.int_of_num x)
      | Value.F(x) -> F(x)

let (bool2bool : Value.t -> value) =
  fun v -> 
    match v with
	     Value.B(x) -> B(x)
      | _ -> assert false

(****************************************************************************)

let sol_number = ref Bddd.sol_number
let draw_in_bdd = ref Bddd.draw_in_bdd
let add_snt_entry = ref Bddd.add_snt_entry
let init_snt = ref Bddd.init_snt
let clear_snt = ref Bddd.clear_snt


let (set_efficient_mode : unit -> unit) =
  fun _ ->
    sol_number := Bddd.sol_number;
    draw_in_bdd := Bddd.draw_in_bdd;
    add_snt_entry := Bddd.add_snt_entry;
    init_snt := Bddd.init_snt;
    clear_snt := Bddd.clear_snt;
    !clear_snt ()

let (set_fair_mode : unit -> unit) =
  fun _ ->
    sol_number := Fair_bddd.sol_number;
    draw_in_bdd := Fair_bddd.draw_in_bdd;
    add_snt_entry := Fair_bddd.add_snt_entry;
    init_snt := Fair_bddd.init_snt;
    clear_snt := Fair_bddd.clear_snt;
    !clear_snt ()


(****************************************************************************)
(* XXX A faire plus proprement ; ca peut dborder  tout moment... *)


let lucky_max_int = max_int / 4

let default_max_float = ref (float_of_int lucky_max_int)
(*   (float_of_int max_int) /. 2.**(float_of_int (!Util.precision + 1)) *)
let default_min_float = ref (-. !default_max_float)
let default_max_int = ref (Num.Int lucky_max_int)
let default_min_int = ref (Num.Int (-lucky_max_int))
let zero  = Num.Int 0


let set_default_max_int i = default_max_int := (Num.Int i)
let set_default_min_int i = default_min_int := (Num.Int i) 
let set_default_max_float f = default_max_float := f
let set_default_min_float f = default_min_float := f

(******************************************************************************)
(** A lustre-like formula parser. *)

let lexer = MyGenlex.make_lexer ["="; "["; "]";"("; ")";"{";"}"; ","; ";"; ".";":";"~";"%"]


let var_type_to_type vt = 
  match vt with
    | Float -> FloatT
    | Int -> IntT
    | Bool -> BoolT

let (make_bool_expr : (var_name * var_type) list -> string -> bool_expr) =
  fun vars0 str ->
    let vars =
      (List.map 
	      (fun (x,t) -> 
	         match t with
		        | Int ->
		            let v = Var.make "" x (var_type_to_type t) Var.Output in 
		            let v1 = Var.set_max v ((Numer(Ival !default_max_int))) in
		              Var.set_min v1 ((Numer(Ival !default_min_int)))
	           | Float ->
		            let v = Var.make "" x (var_type_to_type t) Var.Output in
		            let v1 = Var.set_max v ((Numer(Fval (!default_max_float)))) in
		              Var.set_min v1 ((Numer(Fval (-.(!default_max_float)))))
	           | Bool -> 
		            Var.make "" x (var_type_to_type t) Var.Output 
	      )  
	      vars0
      )
    in
    let ic = Lexing.from_string (str^";") in
    let le = 
      try LucParser.lustreExp LucLexer.lexer ic 
      with e -> 
	     failwith ("*** luckyDraw error: " ^ (Printexc.to_string e) ^ "\n")
    in
      match (LustreExp.infer_type le vars) with
	     | Formu f, vars ->
	         (f, vars)
	     | _ -> failwith "A formula is expected"


(******************************************************************************)
(******************************************************************************)
(******************************************************************************)

(* Boolean constructors *)
let (and_b : bool_expr -> bool_expr -> bool_expr) = 
  fun f1 f2 ->  
    Exp.And(fst f1, fst f2), Util.merge (snd f1) (snd f2) 
 
let or_b   f1 f2 = Exp.Or  (fst f1, fst f2), Util.merge (snd f1) (snd f2) 
let xor_b  f1 f2 = Exp.Xor (fst f1, fst f2), Util.merge (snd f1) (snd f2) 
let impl_b f1 f2 = Exp.Impl(fst f1, fst f2), Util.merge (snd f1) (snd f2) 
let eq_b   f1 f2 = Exp.EqB (fst f1, fst f2), Util.merge (snd f1) (snd f2) 

let not_b f = Exp.Not(fst f), snd f  
let true_b = Exp.True, []  
let false_b = Exp.False, []

let ite_b c f1 f2 = 
  Exp.IteB (fst c, fst f1, fst f2), 
  Util.merge (snd c) (Util.merge (snd f1) (snd f2))

let eq_bool f1 f2  = assert false

let (var_b : string -> bool_expr) =
  fun str -> 
    let v = Var.make "" str BoolT Var.Output in
      (Bvar v), [v]

(* Integer constructors *)

let (var_i : string -> int_expr) =
  fun str -> 
    let v = Var.make "" str IntT Var.Output in
      (Ivar v), [v]

let (val_i : int -> int_expr) =
  fun i -> 
    (Ival (Num.Int i)), []

let eq_i n1 n2 = Exp.Eq (fst n1, fst n2), Util.merge (snd n1) (snd n2) 


let sup_i n1 n2 = Exp.Sup (fst n1, fst n2), Util.merge (snd n1) (snd n2) 
let supeq_i n1 n2 = Exp.SupEq (fst n1, fst n2), Util.merge (snd n1) (snd n2) 
let inf_i n1 n2 = Exp.Inf (fst n1, fst n2), Util.merge (snd n1) (snd n2) 
let infeq_i n1 n2 = Exp.InfEq (fst n1, fst n2), Util.merge (snd n1) (snd n2) 
let ite_i c n1 n2 = 
  Exp.Ite (fst c, fst n1, fst n2), 
  Util.merge (snd c) (Util.merge (snd n1) (snd n2))


let sum_i n1 n2 = Exp.Sum (fst n1, fst n2), Util.merge (snd n1) (snd n2)
let diff_i n1 n2 = Exp.Diff (fst n1, fst n2), Util.merge (snd n1) (snd n2)
let prod_i n1 n2 = Exp.Prod (fst n1, fst n2), Util.merge (snd n1) (snd n2)
let mod_i n1 n2 = Exp.Mod (fst n1, fst n2), Util.merge (snd n1) (snd n2)
let div_i n1 n2 = Exp.Div (fst n1, fst n2), Util.merge (snd n1) (snd n2)
let quot_i = div_i
let uminus_i n1  = Exp.Uminus (fst n1),  (snd n1) 


(* Float constructors *)

let (var_f : string -> float_expr) =
  fun str -> 
    let v = Var.make "" str FloatT Var.Output in
      (Ivar v), [v]

let (val_f : float -> float_expr) =
  fun f -> 
    (Fval f), []

let eq_f n1 n2 = Exp.Eq (fst n1, fst n2), Util.merge (snd n1) (snd n2) 


let sup_f n1 n2 = Exp.Sup (fst n1, fst n2), Util.merge (snd n1) (snd n2) 
let supeq_f n1 n2 = Exp.SupEq (fst n1, fst n2), Util.merge (snd n1) (snd n2) 
let inf_f n1 n2 = Exp.Inf (fst n1, fst n2), Util.merge (snd n1) (snd n2) 
let infeq_f n1 n2 = Exp.InfEq (fst n1, fst n2), Util.merge (snd n1) (snd n2) 
let ite_f c n1 n2 = 
  Exp.Ite (fst c, fst n1, fst n2), 
  Util.merge (snd c) (Util.merge (snd n1) (snd n2))


let sum_f n1 n2 = Exp.Sum (fst n1, fst n2), Util.merge (snd n1) (snd n2)
let diff_f n1 n2 = Exp.Diff (fst n1, fst n2), Util.merge (snd n1) (snd n2)
let prod_f n1 n2 = Exp.Prod (fst n1, fst n2), Util.merge (snd n1) (snd n2)
let div_f n1 n2 = Exp.Div (fst n1, fst n2), Util.merge (snd n1) (snd n2)
let quot_f = div_f
let uminus_f n1  = Exp.Uminus (fst n1),  (snd n1) 


open Prog
(******************************************************************************)
let (solve : bool_expr -> ?verbose:(int) -> solutions_set) =
  fun (f, vars) ?(verbose = 0) -> 
    let (bvars, nvars) = List.partition (fun var -> (Var.typ var) = BoolT) vars in
    let bool_vars_to_gen_f = if bvars = [] then True else
	   List.fold_left 
	     (fun acc vn -> (And(Bvar(vn), acc)))
	     (Bvar(List.hd bvars)) 
	     (List.tl bvars) 
    in
    let input = Value.OfIdent.empty in
    let state = 
      {
	     d =  {
	       memory = Value.OfIdent.empty;
	       ctrl_state = [] ;
	       last_input = Value.OfIdent.empty ;
	       last_output = Value.OfIdent.empty ;
	       last_local = Value.OfIdent.empty ;
	       verbose = verbose
	     } ;
	     s = {
	       reactive = false ;
	       ext_func_tbl = Util.StringMap.empty;
	       bool_vars_to_gen = [bvars] ;
	       num_vars_to_gen  = [nvars];
	       in_vars =  [] ;
	       out_vars = vars ;
	       loc_vars = [] ;
	       memories_names   = [] ;
          initial_ctrl_state = [];
	       output_var_names = [List.map Var.name vars] ;
          get_wtl = (fun _ _ -> assert false);
          is_final = (fun _ -> assert false);
          gen_dot = (fun _ _ _ -> assert false);
	     }
      }
    in
    let ctx_msg = Prog.ctrl_state_to_string_long state.d.ctrl_state in 
    let bdd = Formula_to_bdd.f input state.d.memory ctx_msg state.d.verbose f in
    let comb0 = (Formula_to_bdd.f input state.d.memory ctx_msg state.d.verbose bool_vars_to_gen_f) in
    let comb = Bdd.dand (Bdd.support bdd) comb0 in

      (*     let rec print_li li = *)
      (*       List.iter  *)
      (* 	(fun i -> *)
      (* 	   print_string ( *)
      (* 	     Constraint.to_string( *)
      (* 	       Formula_to_bdd.index_to_linear_constraint i)) ;  *)
      (* 	   print_string "\n\t") li; *)
      (*       print_string"\n" *)
      (*     in *)
      (*     let _=  *)
      (*       print_string "----------------------------------\ncomb:\n\t"; *)
      (*       print_li (Bdd.list_of_support comb) ; *)
      (*       print_string "bdd: \n\t"; *)
      (*       print_li (Bdd.list_of_support bdd) ; *)
      (*       flush stdout; *)
      (*     in *)

      {
	     bdd  = bdd; 
	     comb = comb; 
	     bools_tg = bool_vars_to_gen_f;
	     nums_tg = nvars;
	     state = state
      }


(******************************************************************************)
(** To indicate whether the solution is to be drawn inside, at edges, or at
  vertices of the convex hull of numeric solutions. *)
type draw_mode = int * int * int 


(* -> ?precision:(int)  *)
let (draw : ?mode:(draw_mode) -> ?verbose:(int) -> solutions_set -> solution list) =
  fun ?(mode = (1,0,0)) ?(verbose = 0) ss -> 

(*  ?(precision = 2) *)

    if Bdd.is_false ss.bdd then [] else
      try
	let (k1, k2, k3) = mode in
        let msg = Prog.ctrl_state_to_string_long ss.state.d.ctrl_state in
	let (bool_subst_tab, store_range, store_poly) =
	  !draw_in_bdd Value.OfIdent.empty ss.state.d.memory ss.state.d.verbose msg ss.nums_tg ss.bdd ss.comb
	in
	let bool_subst_l = Value.OfIdent.content bool_subst_tab in
	let num_subst_ll0 =
	  if
	    ss.nums_tg = []
	  then
	    []
	  else
	    let sll1 = Draw.inside verbose store_range store_poly k1 [] 
	    and sll2 = Draw.edges verbose store_range store_poly k2 [] 
	    and sll3 = Draw.vertices verbose store_range store_poly k3 []
	    in
	      List.rev_append sll1 (List.rev_append sll2 sll3)
	in
	let num_subst_ll =
	  List.map (List.map (fun (vn, num) -> (vn, num2value num))) num_subst_ll0
	in
	let bool_subst_l2 = List.map (fun (x,y) -> (x,bool2bool y)) bool_subst_l in
	let subst_ll =
	  if
	    num_subst_ll = []
	  then
	    [bool_subst_l2]
	  else
	    List.map
	      (fun num_subst_l -> List.append bool_subst_l2 num_subst_l)
	      num_subst_ll
	in
	  subst_ll
      with Store.No_numeric_solution -> []


(** nb: the default step mode is StepInside 

  ?precision:(int)

*)


(******************************************************************************)
let (is_empty : solutions_set -> bool) =
  fun ss -> 
    assert false
(* 
   Bdd.is_false ss.bdd     
    -> va dire si c'est vide wrt les bool, mais ne dit rien sur les numriques

   La seule solution pour savoir si c'est vraiment vide, cest de faire 
   un tirage; du coup, cette fonction n'est pas tres utile...

*)


(******************************************************************************)

(* I defined mine because i need to know the seed that has been drawn
   by self_init. *)
let random_seed _ =
  let () = Random.self_init () in
  let seed = Random.int 1073741823 in
    Random.init seed ;
    print_string (
      "The random engine is initialized with the seed "^
      (string_of_int seed)^"\n");
    flush stdout

let (set_seed: int -> unit) =
  fun seed -> 
  Printf.fprintf stderr "The random engine seed is set to %i\n" seed;
  flush stderr;
  Random.init seed

let string_of_value = function
    B(b) -> if b then "t" else "f"
  | I(i) -> string_of_int i
  | F(f) -> string_of_float f

(******************************************************************************)
(******************************************************************************)

let print_subst (vn, value) =
      print_string (vn ^ "=" ^ (string_of_value value) ^ " ")

let (print_solution : solution -> unit) = 
  fun sol -> 
    List.iter print_subst sol;
    print_string "\n"; 
    flush stdout


let _ = 
  !init_snt ();
  random_seed ()

(******************************************************************************)
(******************************************************************************)

(* Non-regression test *)

(* let _ =  *)
(*   let e =   *)
(*     make_bool_expr   *)
(*       [("x",Int);("y",Int);"b",Bool]   *)
(*       "(if b then x+y>=0 else x-y<=0) and x > y and x < y"   *)
(*   in  *)
(*   let e2 = make_bool_expr ["b",Bool] "b and (not b)" in *)
(*     draw ~mode:(1,0,0) (solve e);  *)
(*     draw (solve e2);; *)



