{

  (*
    la gestion des no de ligne est faite dans Lexeme
    pour viter les dpendances boucles entre Lexer et Parser
  *)
open Lexeme
open LucParser
open Lexing
open Hashtbl

(* rcupration d'erreur avec correction line/col *)
exception Lexical_error of string * int * int

let handle_lexical_error fn lexbuf = (
  let lxm = Lexeme.make (lexbuf ) in
    try
      fn lexbuf
    with Lexical_error(msg, _, _) ->
      raise(Lexical_error(msg, lxm.line, lxm.cstart))
)
				
(* table des mots-cl *)
let keywords = Hashtbl.create 50 ;;

Hashtbl.add keywords "and"        (function x -> TK_AND x) ;;
Hashtbl.add keywords "div"        (function x -> TK_DIV x) ;;
Hashtbl.add keywords "else"       (function x -> TK_ELSE x) ;;
Hashtbl.add keywords "enum"       (function x -> TK_ENUM x) ;;
Hashtbl.add keywords "false"      (function x -> TK_FALSE x) ;;
Hashtbl.add keywords "if"         (function x -> TK_IF x) ;;
Hashtbl.add keywords "mod"        (function x -> TK_MOD x) ;;
Hashtbl.add keywords "not"        (function x -> TK_NOT x) ;;
Hashtbl.add keywords "or"         (function x -> TK_OR x) ;;
Hashtbl.add keywords "nor"        (function x -> TK_NOR x) ;;
Hashtbl.add keywords "pre"        (function x -> TK_PRE x) ;;
Hashtbl.add keywords "then"       (function x -> TK_THEN x) ;;
Hashtbl.add keywords "true"       (function x -> TK_TRUE x) ;;
Hashtbl.add keywords "xor"        (function x -> TK_XOR x) ;;

Hashtbl.add keywords "abs"        (function x -> TK_ABS x) ;;
Hashtbl.add keywords "infinity"   (function x -> TK_INFINITY x) ;;

let is_a_keyword ( s: string ) = (
  try
    let res = Hashtbl.find keywords s in
      (Some res)
  with
      Not_found -> None
)

let token_code tk = (
  match tk with
      TK_EOF           -> ("TK_EOF" , Lexeme.dummy )
    | TK_ERROR     lxm -> ("TK_ERROR" , lxm)
    | TK_AND       lxm -> ("TK_AND" , lxm)
    | TK_ABS       lxm -> ("TK_ABS" , lxm)
    | TK_BAR       lxm -> ("TK_BAR" , lxm)
(*     | TK_BOOL      lxm -> ("TK_BOOL" , lxm) *)
    | TK_CDOTS     lxm -> ("TK_CDOTS" , lxm)
    | TK_CLOSE_BRACKET lxm -> ("TK_CLOSE_BRACKET" , lxm)
    | TK_CLOSE_BRACE   lxm -> ("TK_CLOSE_BRACE" , lxm)
    | TK_CLOSE_PAR     lxm -> ("TK_CLOSE_PAR" , lxm)
(*     | TK_COLON     lxm -> ("TK_COLON" , lxm) *)
    | TK_COMA      lxm -> ("TK_COMA" , lxm)
    | TK_DIV       lxm -> ("TK_DIV" , lxm)
    | TK_DIESE     lxm -> ("TK_DIESE" , lxm)
    | TK_DOT       lxm -> ("TK_DOT" , lxm)
    | TK_ELSE      lxm -> ("TK_ELSE" , lxm)
    | TK_EQ        lxm -> ("TK_EQ" , lxm)
    | TK_ENUM      lxm -> ("TK_ENUM" , lxm)
    | TK_FALSE     lxm -> ("TK_FALSE" , lxm)
(*     | TK_FIELD     lxm -> ("TK_FIELD" , lxm) *)
    | TK_GT        lxm -> ("TK_GT" , lxm)
    | TK_GTE       lxm -> ("TK_GTE" , lxm)
    | TK_HAT       lxm -> ("TK_HAT" , lxm)
    | TK_ICONST    lxm -> ("TK_ICONT" , lxm)
    | TK_IDENT     lxm -> ("TK_IDENT" , lxm)
    | TK_IF        lxm -> ("TK_IF" , lxm)
    | TK_IMPL      lxm -> ("TK_IMPL" , lxm)
    | TK_INT       lxm -> ("TK_INT" , lxm)
    | TK_LT        lxm -> ("TK_LT" , lxm)
    | TK_LTE       lxm -> ("TK_LTE" , lxm)
    | TK_MINUS     lxm -> ("TK_MINUS" , lxm)
    | TK_TILDE     lxm -> ("TK_TILDE" , lxm)
    | TK_MOD       lxm -> ("TK_MOD" , lxm)
    | TK_NEQ       lxm -> ("TK_NEQ" , lxm)
    | TK_NOR       lxm -> ("TK_NOR" , lxm)
    | TK_NOT       lxm -> ("TK_NOT" , lxm)
    | TK_OPEN_BRACKET lxm -> ("TK_OPEN_BRACKET" , lxm)
    | TK_OPEN_BRACE   lxm -> ("TK_OPEN_BRACE" , lxm)
    | TK_OPEN_PAR     lxm -> ("TK_OPEN_PAR" , lxm)
    | TK_OR        lxm -> ("TK_OR" , lxm)
    | TK_PCENT     lxm -> ("TK_PCENT" , lxm)
    | TK_PLUS      lxm -> ("TK_PLUS" , lxm)
    | TK_POWER     lxm -> ("TK_POWER" , lxm)
    | TK_PRE       lxm -> ("TK_PRE" , lxm)
    | TK_RCONST    lxm -> ("TK_RCONST" , lxm)
    | TK_SEMICOL   lxm -> ("TK_SEMICOL" , lxm)
    | TK_STAR      lxm -> ("TK_STAR" , lxm)
    | TK_SLASH     lxm -> ("TK_SLASH" , lxm)
    | TK_THEN      lxm -> ("TK_THEN" , lxm)
    | TK_TRUE      lxm -> ("TK_TRUE" , lxm)
    | TK_XOR       lxm -> ("TK_XOR" , lxm)
    | TK_INFINITY  lxm -> ("TK_INFINITY" , lxm)
)

}

(* Pour simplifier les rgles des constantes numriques *)

let chiffres = ['0'-'9']
let exposant = ( 'e' | 'E' ) ( '+' | '-' )? ( chiffres) +

	       
rule lexer = parse
    eof
      { TK_EOF }
      (* saute les blancs *)
  | [' ' '\013' '\009' '\012'] +
    { lexer lexbuf }
      (* retour  la ligne *)
  | '\n'
      {
	Lexeme.new_line ( lexbuf );
	lexer lexbuf	
      }
      
  (* commentaire parenths *)
  | "(*"
      {
	handle_lexical_error comment_par lexbuf;
	lexer lexbuf
      }
      
  (* commentaire en ligne *)
  | "--"
      {
	handle_lexical_error comment_line lexbuf;
	lexer lexbuf
      }
      (* constantes entires et relles *)
  | (chiffres)+  { TK_ICONST (Lexeme.make lexbuf ) }
      
  | (chiffres)+ (exposant) { TK_RCONST (Lexeme.make lexbuf ) }
      
  | (chiffres)+ '.' (chiffres)+ (exposant)? { TK_RCONST (Lexeme.make lexbuf ) }
      
  | '.' (chiffres)+ (exposant)? { TK_RCONST (Lexeme.make lexbuf ) }
      
  (* mots-cl dbutant par un sparateurs (prioritaires) *)
  | "=>" { TK_IMPL ( Lexeme.make lexbuf ) }
  | "<=" { TK_LTE ( Lexeme.make lexbuf ) }
  | "<>" { TK_NEQ ( Lexeme.make lexbuf ) }
  | ">=" { TK_GTE ( Lexeme.make lexbuf ) }
(* 		 | ".%" { TK_FIELD ( Lexeme.make lexbuf ) } *)
  | ".." { TK_CDOTS ( Lexeme.make lexbuf ) }
  | "**" { TK_POWER ( Lexeme.make lexbuf ) }
      (* sparateurs simples *)
  | "+"  { TK_PLUS ( Lexeme.make lexbuf ) }
  | "^"  { TK_HAT ( Lexeme.make lexbuf ) }
  | "#"  { TK_DIESE ( Lexeme.make lexbuf ) }
  | "-"  { TK_MINUS ( Lexeme.make lexbuf ) }
  | "~"  { TK_TILDE ( Lexeme.make lexbuf ) }
  | "/"  { TK_SLASH ( Lexeme.make lexbuf ) }
  | "%"  { TK_PCENT ( Lexeme.make lexbuf ) }
  | "*"  { TK_STAR ( Lexeme.make lexbuf ) }
  | "|"  { TK_BAR ( Lexeme.make lexbuf ) }
  | "="  { TK_EQ ( Lexeme.make lexbuf ) }
      
  | "."  { TK_DOT ( Lexeme.make lexbuf ) }
  | ","  { TK_COMA ( Lexeme.make lexbuf ) }
  | ";"  { TK_SEMICOL ( Lexeme.make lexbuf ) }
      (* 		 | ":"  { TK_COLON ( Lexeme.make lexbuf ) } *)
  | "("  { TK_OPEN_PAR ( Lexeme.make lexbuf ) }
  | ")"  { TK_CLOSE_PAR ( Lexeme.make lexbuf ) }
  | "{"  { TK_OPEN_BRACE ( Lexeme.make lexbuf ) }
  | "}"  { TK_CLOSE_BRACE ( Lexeme.make lexbuf ) }
  | "["  { TK_OPEN_BRACKET ( Lexeme.make lexbuf ) }
  | "]"  { TK_CLOSE_BRACKET ( Lexeme.make lexbuf ) }
  | "<"  { TK_LT ( Lexeme.make lexbuf ) }
  | ">"  { TK_GT ( Lexeme.make lexbuf ) }
      
  (* mot-cl ou identificateur *)
  | ['A'-'Z' 'a'-'z' '_'] ['A'-'Z' 'a'-'z' '\'' '_' '0'-'9'] *
    {
      let lxm = Lexeme.make lexbuf in
      let x = is_a_keyword ( lxm.str ) in
	match x with
	    None -> TK_IDENT ( lxm )
	  | Some keyw -> keyw ( lxm )
    }
  | _    { TK_ERROR ( Lexeme.make lexbuf ) }
		
and comment_par = parse
    "*)"
    { }
  | "\n"
      {
	Lexeme.new_line ( lexbuf );
	comment_par lexbuf	
      }
  | eof
      {
	raise(Lexical_error("unterminated comment", 0, 0))
      }
  |	_
      { comment_par lexbuf }

and comment_line = parse
    '\n'
      {
	Lexeme.new_line ( lexbuf );
      }
  | eof
      { }
  |	_
      { comment_line lexbuf }

