(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** 
**-----------------------------------------------------------------------
**
** File: graphUtil.ml
** Author: jahier@imag.fr
**
*)

module Tbl = Util.StringMap

type color = Grey | Black
type succ_table = string list Tbl.t
type color_table = color Tbl.t

exception Cycle of string * string * color_table

let rec (visit : succ_table -> color_table -> string -> color_table) =
  fun succ_t color_t n ->
    if not (Tbl.mem n succ_t) then Tbl.add n Black color_t else
      let color_t' =
	     List.fold_left 
	       (fun color_t nt -> 
	         try
	           match Tbl.find nt color_t with
		          | Grey -> raise (Cycle (n, nt, color_t))
		          | Black -> color_t
	         with 
		 (* The node [nt] is white *)
		          Not_found -> visit succ_t color_t nt
	       ) 
	       (Tbl.add n Grey color_t)
	       (Tbl.find n succ_t)
      in
	   Tbl.add n Black color_t'

(* exported *)
let (has_cycle : string -> succ_table -> string option) =
  fun init succ_table -> 
    try ignore(visit succ_table Tbl.empty init); None
    with Cycle(_n, nt, _color_table) -> Some(nt)


(********************************************************************************)
(* Topologically sort nodes of a graph that contains no cycle *)

let (top_sort : string list -> succ_table -> string list) =
  fun l succ_tbl -> 
    let mtbl_init = 
      List.fold_left (fun acc str -> Tbl.add str false acc) Tbl.empty l 
    in
    let rec f acc l stbl mtbl =
      match l with
	       [] -> List.rev acc
	     | x::tail -> 
	       if
	         try
		        Tbl.find x mtbl 
	         with Not_found -> assert false
	           then 
	             f acc tail stbl mtbl 
	           else if (Tbl.mem x stbl) then 
	             f acc ((Tbl.find x stbl) @ l) (Tbl.remove x stbl) mtbl
	           else
	             f (x::acc) tail stbl (Tbl.add x true mtbl)
            in
          f [] l succ_tbl mtbl_init

(********************************************************************************)

(**/**)

(* Non regression unit tests *)

let build_succ_table el =
  List.fold_left
    (fun acc (nf,nt) ->
       try Tbl.add nf (nt::(Tbl.find nf acc)) acc
       with Not_found -> Tbl.add  nf [nt] acc
    )
    Tbl.empty
    el

(* nb: could be exported *)
type edge = string * string
let (has_cycle2 : string -> edge list -> string option) =
  fun init el ->
    let succ_table = build_succ_table el in
      try ignore(visit succ_table Tbl.empty init); None
      with Cycle(_n, nt, _color_table) -> Some(nt)

let g1 = [("a","b");("b","a")]
let g2 = [("a","b");("a","c"); ("a","d");("a","e"); ("c","e")]
let g3 = [("a","b");("a","c"); ("a","d");("a","e"); ("c","e");("e","a")]
let g4 = [("init","a"); ("init","b");("a","c");("a","d"); ("b","c"); ("b","d"); 
	  ("d","e");  ("c","e")]
let g5 = [("init","a"); ("init","b");("a","c");("a","d"); ("b","c"); ("b","d"); 
	  ("d","e");  ("c","e"); ("d","a")]

let _ = assert ((has_cycle2 "a" g1) <> None)
let _ = assert ((has_cycle2 "a" g2) = None)
let _ = assert ((has_cycle2 "a" g3) <> None)
let _ = assert ((has_cycle2 "init" g4) = None)
let _ = assert ((has_cycle2 "init" g5) <> None)

(********************************************************************************)

let list = ["init";"a";"b";"c";"d";"e"]
let succ_tbl = build_succ_table g4

let _ = assert ((top_sort list succ_tbl) = [ "e"; "d";"c"; "b"; "a";"init"])

(********************************************************************************)
(* used for debugging *)

let print_table t2str =
  Tbl.iter (fun key value -> print_string ("\t" ^ key ^ " -> "^(t2str value) ^"\n"))

let color_to_string  = function Grey  -> "grey" | Black -> "black"
