(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License
**-----------------------------------------------------------------------
**
** File: graph.mli
** Author: jahier@imag.fr
*)

(**  *)

(** A graph data structure as well as various functions that operate
  over it. This ADT is optimized for get_ functions (hence, add_trans
  is more expensive).

  ps: it is side-effect free.
  pps: It is based on hashtbl which is not, but i perform copies when necessary.
*)


type ('a, 'b) t
(** Graph type, where 'a is the type of nodes, and 'b the type of
  arcs. *)

val create: unit -> ('a, 'b) t
  (** [create ()] creates an empty graph. *)


val contain_trans : ('a, 'b) t -> 'a -> 'b -> 'a -> bool
  (** [contain_trans g node_from arc_info node_to] returns true iff the
    transition from [node_from] to [node_to] with arc label [arc_info] is
    in the graph [g].  *)


val contain_node : ('a, 'b) t -> 'a -> bool
  (** [contain_node g node] returns true iff the node [node] exists in
    the graph [g].  *)

val add_trans: ('a, 'b) t -> 'a -> 'b -> 'a -> ('a, 'b) t
  (**
    [add_trans g node_from arc_info node_to] adds a transition
    from [node_from] to [node_to] with arc label [arc_info] to the
    graph [g].
    *)

val remove_trans : ('a, 'b) t -> 'a * 'b * 'a -> ('a, 'b) t
  (** [remove_trans g (nf, arc, nt)] removes the transition [(nf, arc, nt)]
    from the graph [g]
  *)


val get_list_of_target_nodes: ('n, 'arc) t -> 'n -> ('n * 'arc) list
  (**
    [get_list_of_target_nodes g node] returns the list of target
    nodes starting from [node] in the graph [g]. Actually, it does
    not only return a list of nodes but a list of pairs [(node_to,
    arc_label)] where [arc_label] is the arc label of the arc from
    [node] to [node_to].
  *)

val get_list_of_father_nodes: ('a, 'b) t -> 'a -> ('a * 'b) list
  (**
    [get_list_of_father_nodes g node] returns the list of
    nodes targetting [node] in the graph [g].
  *)


val get_all_nodes: ('a, 'b) t -> 'a list
  (**
    [get_all_nodes g] returns the set of nodes of [g].
  *)

val get_all_trans: ('a, 'b) t -> ('a * 'b * 'a) list
  (**
    [get_all_trans g] returns the list transitions of the graph [g]
  *)

val size : ('a, 'b) t -> int * int * int
  (** Return the number of nodes and the number of transitions of the graph. *)


