(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License
**-----------------------------------------------------------------------
**
** File: graph.ml
** Author: jahier@imag.fr
*)

(*------------------------------------------------------------------------*)

open Hashtbl


(* exported *)
type ('a, 'b) t = {
  mutable table :  ('a, ('a * 'b) list) Hashtbl.t;
  mutable father : ('a, ('a * 'b) list) Hashtbl.t;
  mutable nodes : 'a list;
  mutable trans : ('a * 'b * 'a) list
}

let (copy : ('a, 'b) t -> ('a, 'b) t) =
  fun g -> 
    {
      table = (Hashtbl.copy g.table) ;
      father = (Hashtbl.copy g.father) ;
      nodes = g.nodes ;
      trans = g.trans
    }

(* exported *)
let (create: unit -> ('a, 'b) t) =
  fun () ->
    {
      table = (Hashtbl.create 1) ;
      father = (Hashtbl.create 1) ;
      nodes = [] ;
      trans = []
    }


(* exported *)
let (get_list_of_target_nodes: ('a, 'b) t -> 'a -> ('a * 'b) list) =
  fun g node ->
    try Util.hfind g.table node
    with Not_found -> []

(* exported *)
let (get_list_of_father_nodes: ('a, 'b) t -> 'a -> ('a * 'b) list) =
  fun g node ->
    try Util.hfind g.father node
    with Not_found -> []

(* exported *)
let (contain_trans : ('a, 'b) t -> 'a -> 'b -> 'a -> bool) =
  fun g node_from arc_info node_to ->
    List.mem (node_from, arc_info, node_to) g.trans

(* exported *)
let (contain_node : ('a, 'b) t -> 'a -> bool) =
  fun g node ->
    List.mem node g.nodes


(* exported *)
let (add_trans: ('a, 'b) t -> 'a -> 'b -> 'a -> ('a, 'b) t) =
  fun g0 node_from arc_info node_to ->
    let g = copy g0 in
      if
	List.mem (node_from, arc_info, node_to) g.trans
      then
	g
      else
	let l = g.nodes in
	let l1 = if (List.mem node_from l) then l else node_from::l in
	let l2 = if (List.mem node_to l1) then l1 else node_to::l1 in
	  
	  if (Hashtbl.mem g.table node_from) then
	    let list = ((node_to, arc_info)::(Util.hfind g.table node_from)) in
	      Hashtbl.replace g.table node_from list
	  else
	    Hashtbl.add g.table node_from [(node_to, arc_info)] ;
	  
	  if (Hashtbl.mem g.father node_to) then
	    let list2 = ((node_from, arc_info)::(Util.hfind g.father node_to)) in
	      Hashtbl.replace g.father node_to list2
	  else
	    Hashtbl.add g.father node_to [(node_from, arc_info)] ;
	  
	  g.nodes <- l2 ;
	  g.trans <- (node_from, arc_info, node_to)::g.trans ;
	  assert (
	    try
	      List.for_all
		(fun (nf, arc, nt) ->
		   List.mem (nt, arc) (Util.hfind g.table nf) &&
		   List.mem (nf, arc) (Util.hfind g.father nt)
		)
		g.trans
	    with _ -> false
	  );
	  g



let (remove_nodes : ('a, 'b) t -> 'a -> ('a, 'b) t) =
  fun g0 n ->
    let g = copy g0 in
      if
	((not (Hashtbl.mem g.father n)) || (Util.hfind g.father n) = []) &&
	((not (Hashtbl.mem g.table n))  || (Util.hfind g.table n) = [])
      then
	g.nodes <- Util.rm n g.nodes
      else
	();
      g

(* exported *)
let (remove_trans : ('a, 'b) t -> 'a * 'b * 'a -> ('a, 'b) t) =
  fun g0 (nf, arc, nt) ->
    let g = copy g0 in
      g.trans <- Util.rm (nf, arc, nt) g.trans ;
      Hashtbl.replace g.table  nf (Util.rm (nt,arc) (Util.hfind g.table nf));
      Hashtbl.replace g.father nt (Util.rm (nf,arc) (Util.hfind g.father nt));

      (* I should clean up nodes too? *)
      remove_nodes (remove_nodes g nf) nt


(* exported *)
let (get_all_nodes: ('a, 'b) t -> 'a list) =
  fun g -> g.nodes


(* exported *)
let (get_all_trans:  ('a, 'b) t -> ('a * 'b * 'a) list ) =
  fun g -> g.trans

(* exported *)
let (size : ('a, 'b) t -> int * int * int) =
  fun g ->
    (Util.hashtbl_size g.table, List.length g.nodes, List.length g.trans)

  
