(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License
**-----------------------------------------------------------------------
**
** File: formula_to_bdd.ml
** Main author: jahier@imag.fr
*)

open Value
open Exp
open Constraint


let output_msg msg =
  output_string stderr msg;
  flush stderr


(** [lookup input pre vn] looks up the value of vn in [pre] and [input] *)
let rec (lookup: Var.env_in -> Var.env -> string -> int -> Exp.var -> Value.t option) =
  fun input memory ctx_msg vl var ->
    match (Var.mode var) with
      | Var.Output -> None
      | Var.Local -> None
      | Var.Input ->
	       ( try
	           Some(Var.get_val_env_in input (Var.name var))
	         with _ ->
              output_msg ctx_msg;
		        exit 2
	       )
      | Var.Pre ->
	       (
	         try
	           (* Some(List.assoc (Var.name var) memory) *)
	           Some(Value.OfIdent.get memory (Var.name var))
	         with
		          (* if not in memory, look at a possible init value *)
		          Not_found ->
		            match (Var.init var) with
		              | Some (Numer (Ival i)) -> Some (N (I i))
		              | Some (Numer (Fval f)) -> Some (N (F f))
		              | Some (Numer (Ivar vi)) -> lookup input memory ctx_msg vl vi
		              | Some (Numer (Fvar vf)) -> lookup input memory ctx_msg vl vf
		              | Some (Formu  True) -> Some (B true)
		              | Some (Formu  False) -> Some (B false)
		              | Some (Liste _) -> 
			               assert false (* XXX not sure ...*)
                          (* 		    | Some (Liste [Nu (Ival i)]) ->  *)
                          (* 			Some (N (I i)) *)
			                 
		              | Some _ ->
			               (* Structured expression ougth to have been
			                  flattened before *)
			               output_msg (
			                 "*** Error: the initial value of variable " 
			                 ^ (Prevar.get_root_var_name (Var.name var)) ^
			                   "  is not valid.\n");
			               exit 2
		              | None ->
			               output_msg (
			                 "*** The expression " ^ (Prevar.format (Var.name var))
			                 ^ " is undefined at current step " ^ ctx_msg
			                 ^ "*** You need to rewrite your environment"
			                 ^ " spec in such a way that no pre is used"
			                 ^ " for that variable at that step\n    "
			                 ^ " or to give it an initial value.\n"^ ctx_msg);
			               exit 2
	       )


(****************************************************************************)
(****************************************************************************)

(* Transforming a formula into a bdd is expensive; this is why we
   store the result of this transformation in (internal) tables.

   Moreover, formula that does not depend on pre nor input vars are
   stored in a different table that is not cleared at each new step
   (cf [bdd_global] and [set_bdd_global] versus [bdd] and [set_bdd]).
    Ditto for the linear_constraint <-> bdd index tables.

   The reason for that is that, formula that depends on pre or input
   are not likely to appear twice if they contains numeric values.
   For Boolean, well, I could leave them in the global table.
*)

(* This counter is used to provide fresh var indexes *)
let index_cpt = ref 0

(* List of currently unused bdd var indexes. When we need an index,
   we first look in this list before creating a new index. This is to avoid
   the creation of useless bdd index, which considerably slow down the
   Bdd package.
*)
let free_index_list = ref []

(* returns a free index and updates [free_index_list] *)
let (get_an_index : unit -> int) =
  fun _ ->
    let i = 
      match !free_index_list with
	| [] ->
(* 	  print_string ("new index ("^ (string_of_int (!index_cpt + 1)) ^ ") !\n"); *)
	    index_cpt := !index_cpt + 1;
	    !index_cpt
	| i::tail ->
	    free_index_list := tail;
	    assert (not (List.mem i tail));
	    i
    in
      i
	
let (free_indexes : int list -> unit) =
  fun il ->
    (* We sort the list because the initial bdd order is generally the best *)
    free_index_list := List.sort (-) (List.rev_append il !free_index_list)


let init_manager () =
(*   let man =  Manager.make 100 0 0 10 cudd_maxmem in *)
(*     Manager.disable_autodyn man; *)
(*     man *)
  Bdd.init ~pagesize:50000 ~verbose:true ()
  
let bdd_manager = ref (init_manager ()) 

let lc2i_tbl = Hashtbl.create 0
let i2lc_tbl = Hashtbl.create 0
let global_lc2i_tbl = Hashtbl.create 0
let global_i2lc_tbl = Hashtbl.create 0

let bdd_tbl = Hashtbl.create 0
let bdd_tbl_global = Hashtbl.create 0

(* Returns the bdd of a formula if it is available in the cache,
  raises [Not_found] otherwise. *)
let (bdd : Exp.formula  -> Bdd.t) =
  fun f -> Util.hfind bdd_tbl f


let (bdd_global : Exp.formula  -> Bdd.t) =
  fun f -> Util.hfind bdd_tbl_global f

(** Stores the correspondance between a formula and a bdd. *)
let (set_bdd : Exp.formula  -> Bdd.t -> unit) =
  fun f bdd ->
    Hashtbl.replace bdd_tbl f bdd

let (set_bdd_global : Exp.formula  -> Bdd.t -> unit) =
  fun f bdd -> 
    Hashtbl.replace bdd_tbl_global f bdd


(* exported *)
let (get_index_from_linear_constraint : Constraint.t -> int) =
  fun f -> 
    try Util.hfind lc2i_tbl f
    with Not_found ->
      try Util.hfind global_lc2i_tbl f
      with Not_found ->
	-1

let (linear_constraint_to_index : Constraint.t -> bool -> int) =
  fun f depend_on_input -> 
    let i =
      if
        depend_on_input
      then
        (
	       try Util.hfind lc2i_tbl f
	       with Not_found ->
	         let index = get_an_index () in
	         Hashtbl.add lc2i_tbl f index;
	         Hashtbl.add i2lc_tbl index f;
	         index
        )
      else
        (
	       try Util.hfind global_lc2i_tbl f
	       with Not_found ->
	         let index = get_an_index () in
	         Hashtbl.add global_lc2i_tbl f index;
	         Hashtbl.add global_i2lc_tbl index f;
	         index
        )
    in
      (*       output_msg ((Constraint.to_string_verbose f) ^ " -> " ^ (string_of_int i) *)
      (* 		  ^ (if depend_on_input then " (depends on inputs or memory)" *)
      (* 		     else " (does not depend on inputs nor memory)") ^ "\n"); *)
    i


(* exported *)
let (index_to_linear_constraint : int -> Constraint.t) =
  fun i ->
    try Util.hfind global_i2lc_tbl i
    with Not_found ->
	 try Util.hfind i2lc_tbl i with _ ->
    failwith (Printf.sprintf "Error: can not find index %i in Formula_to_bdd tables\n" i)


(****************************************************************************)
(* Clearing table procedures *)

let (clear_linear_constraint_index : unit -> unit) =
  fun _ ->
    let index_to_free =
      Hashtbl.fold
	     (fun index _ acc -> index::acc)
	     i2lc_tbl
	     [];
    in
    Hashtbl.clear i2lc_tbl ;
    Hashtbl.clear lc2i_tbl ;
    free_indexes index_to_free

let (clear_global_linear_constraint_index : unit -> unit) =
  fun _ ->
    Hashtbl.clear i2lc_tbl ;
    Hashtbl.clear lc2i_tbl ;
    Hashtbl.clear global_i2lc_tbl ;
    Hashtbl.clear global_lc2i_tbl ;
    index_cpt := 0;
    free_index_list := []



(* Exported *)
let  (clear_step : unit -> unit) =
  fun _ ->
    Hashtbl.clear bdd_tbl ;
    Hashtbl.clear bdd_tbl_global ;
    clear_linear_constraint_index ()


(* Exported *)
let (clear_all : unit -> unit) =
  fun _ ->
    Hashtbl.clear bdd_tbl ;
    Hashtbl.clear bdd_tbl_global ;
    clear_global_linear_constraint_index ()


(****************************************************************************)
(****************************************************************************)

type comp = SupZero | SupEqZero | EqZero

(* 
  The bool associated the bdd says whether or not the result 
   of the translation should be tabulated or not. 

   The heuristic is to avoid the tabulation of expressions that
   depends on 
   - float
   - memory
   - input 

   true means : do not store

   it's just an heuristic anyway
*)


(* exported *)
let rec (f : Var.env_in -> Var.env -> string -> int -> Exp.formula -> Bdd.t) =
  fun input memory ctx_msg vl f ->
    fst (translate_do input memory ctx_msg vl f)
and (translate_do : Var.env_in -> Var.env -> string -> int -> Exp.formula ->
      Bdd.t * bool) =
  fun input memory ctx_msg vl f ->
    try (bdd f, true)
    with Not_found ->
      try (bdd_global f, false)
      with Not_found ->
	     let (bdd, dep) =
 	       match f with
	           Not(f1) ->
		          let (bdd_not, dep) =  (translate_do input memory ctx_msg vl f1) in
		            (Bdd.dnot bdd_not, dep)

	         | Or(f1, f2) ->
		          let (bdd1, dep1) = (translate_do input memory ctx_msg vl f1)
		          and (bdd2, dep2) = (translate_do input memory ctx_msg vl f2)
		          in
		            ((Bdd.dor bdd1 bdd2), dep1 || dep2)

	         | And(f1, f2) ->
		          let (bdd1, dep1) = (translate_do input memory ctx_msg vl f1)
		          and (bdd2, dep2) = (translate_do input memory ctx_msg vl f2)
		          in
		            (Bdd.dand bdd1 bdd2, dep1 || dep2)

	         | Impl(f1, f2) ->
		          let (bdd1, dep1) = (translate_do input memory ctx_msg vl f1)
		          and (bdd2, dep2) = (translate_do input memory ctx_msg vl f2)
		          in
		            (Bdd.dor (Bdd.dnot bdd1) bdd2, dep1 || dep2)

	         | Xor(f1, f2) ->
		          let (bdd1, dep1) = (translate_do input memory ctx_msg vl f1)
		          and (bdd2, dep2) = (translate_do input memory ctx_msg vl f2)
		          in
		            (Bdd.xor bdd1 bdd2, dep1 || dep2)

	         | IteB(f1, f2, f3) ->
		          let (bdd1, dep1) = (translate_do input memory ctx_msg vl f1)
		          and (bdd2, dep2) = (translate_do input memory ctx_msg vl f2)
		          and (bdd3, dep3) = (translate_do input memory ctx_msg vl f3)
		          in
		            ((Bdd.ite bdd1 bdd2 bdd3), dep1 || dep2 || dep3 )
		              
	         | True ->  (Bdd.dtrue  !bdd_manager, false)
	         | False -> (Bdd.dfalse !bdd_manager, false)
	         | Bvar(v) ->
		          ( match (lookup input memory ctx_msg vl v) with
		                Some(B(bool)) ->
			               if bool
			               then (Bdd.dtrue  !bdd_manager, true)
			               else (Bdd.dfalse !bdd_manager, true)
		              | Some(x) ->
			               output_msg (
			                 "\n*** Type error: " ^  (Value.to_string x) ^
			                   "(" ^ (formula_to_string f) ^ ")" ^
			                   " ougth to be a Boolean.\n");
			               exit 2
		              | None ->
			               (match (Var.alias v) with
			                    Some (Formu(fa)) -> translate_do input memory ctx_msg vl fa
			                  | Some (Numer(e)) -> assert false
			                  | Some (Liste l)  -> assert false
			                  | None ->
			                      assert ((Var.mode v) <> Var.Input);
			                      (Bdd.ithvar
				                      (linear_constraint_to_index (Bv(v)) false),
				                    false)
			               )
		          )

	         | EqB(f1, f2) ->
		          let (bdd1, dep1) = (translate_do input memory ctx_msg vl f1)
		          and (bdd2, dep2) = (translate_do input memory ctx_msg vl f2)
		          in
		            (Bdd.eq bdd1 bdd2, dep1 || dep2)

	         | Eq(e1, e2) ->
		          let gne = expr_to_gne (Diff(e1, e2)) input memory ctx_msg vl in
		            (gne_to_bdd gne EqZero)
		              
	         | SupEq(e1, e2) ->
		          let gne = expr_to_gne (Diff(e1, e2)) input memory ctx_msg vl in
		            (gne_to_bdd gne SupEqZero)
		              
	         | Sup(e1, e2)   ->
		          let gne = expr_to_gne (Diff(e1, e2)) input memory ctx_msg vl in
		            (gne_to_bdd gne SupZero)
		              
	         | InfEq(e1, e2) ->
		          let gne = expr_to_gne (Diff(e2, e1)) input memory ctx_msg vl in
		            (gne_to_bdd gne SupEqZero)
		              
	         | Inf(e1, e2)   ->
		          let gne =  expr_to_gne (Diff(e2, e1)) input memory ctx_msg vl in
		            (gne_to_bdd gne SupZero)
	     in
          if vl > 1 then (
            Printf.eprintf "%s" (" >>> " ^ (formula_to_string f) ^  " " ^  
                                   (if Bdd.is_false bdd then " is false " else "") ^
                                   (if Bdd.is_true bdd then " is true " else "") ^"\n") ; 
            flush stderr
          );
	       if
	         dep
	       then
	         ( set_bdd f bdd;	
	           match f with
		            Not(nf) -> () (* Already in the tbl thanks to the rec call *)
		          | _  -> set_bdd (Not(f)) (Bdd.dnot bdd)
	         )
	       else
	         (* [f] translate_does not depend on pre nor input memory ctx_msg vl vars *)
	         ( set_bdd_global f bdd ;	
	           match f with
		            Not(nf) -> () (* Already in the table thanks to the rec call *)
		          | _  ->
		              set_bdd_global (Not(f)) (Bdd.dnot bdd)
	         );

	       (bdd, dep)
and
    (num_to_gne: Var.env_in -> Var.env -> string -> int -> Exp.num -> Gne.t) =
  fun input memory ctx_msg vl e ->
    expr_to_gne e input memory ctx_msg vl
and
    (expr_to_gne: Exp.num -> Var.env_in -> Var.env -> string -> int -> Gne.t) =
  fun e input memory ctx_msg vl ->
    (** Evaluates pre and input vars appearing in [e] and tranlates
        it into a so-called garded normal form.  *)
    let gne =
      match e with
	       Sum(e1, e2) ->
	         let gne1 = (expr_to_gne e1 input memory ctx_msg vl)
	         and gne2 = (expr_to_gne e2 input memory ctx_msg vl)
	         in
	           Gne.add gne1 gne2
	     | Uminus e ->
	         Gne.opposite (expr_to_gne e input memory ctx_msg vl)

	     | Inf_int ->
	         assert false (* should not occur since only weigth can be infinite
			                   and this checked at parsing ... *)

	     | Diff(e1, e2) ->
	         let gne1 = (expr_to_gne e1 input memory ctx_msg vl)
	         and gne2 = (expr_to_gne e2 input memory ctx_msg vl)
	         in
	           Gne.diff gne1 gne2

	     | Prod(e1, e2) ->
	         let gne1 = (expr_to_gne e1 input memory ctx_msg vl)
	         and gne2 = (expr_to_gne e2 input memory ctx_msg vl)
	         in
	           Gne.mult gne1 gne2

	     | Quot(e1, e2) ->
	         let gne1 = (expr_to_gne e1 input memory ctx_msg vl)
	         and gne2 = (expr_to_gne e2 input memory ctx_msg vl)
	         in
	           Gne.div gne1 gne2

	     | Mod(e1, e2)  ->
	         let gne1 = (expr_to_gne e1 input memory ctx_msg vl)
	         and gne2 = (expr_to_gne e2 input memory ctx_msg vl)
	         in
	           Gne.modulo gne1 gne2

	     | Div(e1, e2) ->
	         let gne1 = (expr_to_gne e1 input memory ctx_msg vl)
	         and gne2 = (expr_to_gne e2 input memory ctx_msg vl)
	         in
	           Gne.div gne1 gne2

	     | Ivar(v) ->
	         let str = (Var.name v) in
	           ( match (lookup input memory ctx_msg vl v) with
		              Some(N(I(i))) ->
		                (Gne.make
		                   (Ne.make "" (I i))
		                   true
		                )
		            | None ->
		                (match (Var.alias v) with
			                  Some (Numer(ea)) ->
			                    expr_to_gne ea input memory ctx_msg vl
		                   | Some (Formu(f)) ->
			                    output_msg "\n*** Type error.\n*** ";
			                    output_msg (formula_to_string f) ;
			                    output_msg " ougth to be a numeric expression.\n";
			                    exit 2
		                   | Some (Liste l) ->
			                    assert false
		                   | None ->
			                    (Gne.make
			                       (Ne.make str (I(Num.num_of_int 1)))
			                       false
			                    )
		                )
		            | Some(N(F(f))) ->
		                output_msg "\n*** Warning : type error, ";
		                output_msg ((string_of_float f)
				                      ^ " is a float, but an int is expected.\n");
		                let i = int_of_float f in
                        (Gne.make
		                     (Ne.make "" (I (Num.num_of_int i)))
		                     true
		                  ) 
		            | Some(B(f)) ->
		                output_msg "\n*** Warning : type error, ";
		                output_msg ((string_of_bool f)
				                      ^ " is a bool, but an int is expected. " ^
                                  "Continuing with 0.\n");
		                (Gne.make
		                   (Ne.make "" (I (Num.num_of_int 0)))
		                   true
		                )
	           )

	     | Fvar(v) ->
	         let str = (Var.name v) in
	           ( match (lookup input memory ctx_msg vl v) with
		              Some(N(F(f))) ->
		                ( Gne.make
			                 (Ne.make "" (F(f)))
			                 true
		                )
		            | None ->
		                (match (Var.alias v) with
			                  Some (Numer(ea)) -> expr_to_gne ea input memory ctx_msg vl
		                   | Some (Formu(f)) ->
			                    output_msg "\n*** Type error.\n*** ";
			                    output_msg (formula_to_string f) ;
			                    output_msg " ougth to be a numeric expression.\n";
			                    assert false
		                   | Some (Liste l) ->
			                    assert false
		                   | None ->
			                    (Gne.make
			                       (Ne.make str (F(1.)))
			                       false
			                    )
		                )
		            | Some(N(I(i))) ->
                      let f = Num.float_of_num i in
		                  output_msg "\n*** Type error, ";
		                  output_msg ((Num.string_of_num i)
				                        ^ " is an int, but a float is expected. I convert it to '"^
                                    (string_of_float f)^"'\n");
                        assert false

		            | Some(B(f)) ->
                      let ff = if f then 0.0 else 1.0 in
		                  output_msg "\n*** Type error, ";
		                  output_msg ((string_of_bool f)
				                        ^ " is a bool, but a float is expected. I convert it to '"^
                                    (string_of_float ff)^"'\n");
                        assert false
	           )

	     | Ival(i) ->
	         (Gne.make
	            (Ne.make "" (I(i)))
	            false
	         )

	     | Fval(f) ->
	         ( Gne.make
		          (Ne.make "" (F(f)))
		          false
	         )

	     | FFC(id, cfunc, type_args, lib_name, args) ->
	         let evaluated_args = eval_ezdl_args input memory ctx_msg vl id args in
	         let _ = if vl > 1 then (
	           print_string ("\t" ^ id ^ " "^ 
			                     (List.fold_left 
			                        (fun acc t  -> acc ^ " " ^(Ezdl.carg_to_string t)) 
			                        "" evaluated_args));
	           flush stdout;
	         )
	         in
	         let res_call = Ezdl.cargs2f cfunc evaluated_args in
	         let _ = if vl > 1 then (
	           print_string (" = " ^ (string_of_float res_call) ^ "\n");
	           flush stdout;
	         )
	         in
	           ( Gne.make
		            (Ne.make "" (F(res_call)))
		            true (* As C functions may perform side-effects, we need 
			                 to recompute the result at each step !
                          not only an heuristic here...
                       *)
	           )

	     | IFC(id, cfunc, type_args, lib_name, args) ->
	         let evaluated_args = eval_ezdl_args input memory ctx_msg vl id args in
	         let _ = if vl > 1 then (
	           print_string ("\t" ^ id ^ " "^ 
			                     (List.fold_left 
			                        (fun acc t  -> acc ^ " " ^(Ezdl.carg_to_string t)) 
			                        "" 
			                        evaluated_args));
	           flush stdout;
	         )
	         in
	         let res_call = Ezdl.cargs2i cfunc evaluated_args in
	         let _ = if vl > 1 then (
	           print_string (" = " ^ (string_of_int res_call) ^ "\n");
	           flush stdout;
	         )
	         in
	           (Gne.make
		           (Ne.make "" (I(Num.num_of_int res_call)))
		           true
	           )

	     (* Those one are hard-coded to make things simpler and faster for Lutin. *)
	     | Gcont(a1,a2,a3) -> 
	         let (a1') = eval_int_arg input memory ctx_msg vl "gauss_continue" a1
	         and (a2') = eval_int_arg input memory ctx_msg vl "gauss_continue" a2
	         and (a3') = eval_int_arg input memory ctx_msg vl "gauss_continue" a3
	         in
	         let i = LutinUtils.gauss_continue a1' a2' a3' in
	           (Gne.make
		           (Ne.make "" (I(Num.num_of_int i)))
		           true
	           )
	             
	     | Gstop(a1,a2,a3) ->
	         let (a1') = eval_int_arg input memory ctx_msg vl "gauss_stop" a1
	         and (a2') = eval_int_arg input memory ctx_msg vl "gauss_stop" a2
	         and (a3') = eval_int_arg input memory ctx_msg vl "gauss_stop" a3
	         in
	         let i = LutinUtils.gauss_stop a1' a2' a3' in
	           (Gne.make
		           (Ne.make "" (I(Num.num_of_int i)))
		           true
	           )

	     | Icont(a1,a2,a3) ->
	         let (a1') = eval_int_arg input memory ctx_msg vl "interval_continue" a1
	         and (a2') = eval_int_arg input memory ctx_msg vl "interval_continue" a2
	         and (a3') = eval_int_arg input memory ctx_msg vl "interval_continue" a3
	         in
	         let i = LutinUtils.interval_continue a1' a2' a3' in
	           (Gne.make
		           (Ne.make "" (I(Num.num_of_int i)))
		           true
	           )

	     | Istop(a1,a2,a3) ->
	         let (a1') = eval_int_arg input memory ctx_msg vl "interval_stop" a1
	         and (a2') = eval_int_arg input memory ctx_msg vl "interval_stop" a2
	         and (a3') = eval_int_arg input memory ctx_msg vl "interval_stop" a3
	         in
	         let i = LutinUtils.interval_stop a1' a2' a3' in
	           (Gne.make
		           (Ne.make "" (I(Num.num_of_int i)))
		           true
	           )
	     | Ite(f, e1, e2) ->
	         let (bdd, depf) = translate_do input memory ctx_msg vl f 
	         and gne_t = (expr_to_gne e1 input memory ctx_msg vl)
	         and gne_e = (expr_to_gne e2 input memory ctx_msg vl) in
              Gne.of_ite bdd depf gne_t gne_e
    in
      gne
	     
and (eval_ezdl_args : Var.env_in -> Var.env -> string -> int -> string -> Exp.t list
      -> Ezdl.carg list) =
  fun input memory ctx_msg vl id args -> 
    List.map
      (fun arg -> 
         match arg with
	          Formu bool_expr -> assert false
	        | Numer num_expr -> 
	            let gne = num_to_gne input memory ctx_msg vl num_expr in
	              (match Gne.get_constant gne with
		              | None ->
		                  let errmsg = 
			                 "*** Error when calling " ^ id ^ ". " ^
			                   (Exp.num_to_string num_expr) ^ 
			                   " should be bound (i.e., it should depend " ^ 
			                   "only one inputs and memories).\n"
		                  in
			                 failwith errmsg
		              | Some (I i) -> Ezdl.Int_carg (Util.int_of_num i)
		              | Some (F f) -> Ezdl.Double_carg f
		           )
	        | Liste _ -> assert false
	   )
	   args 

and (eval_int_arg : Var.env_in -> Var.env -> string -> int -> string -> Exp.num -> int) =
  fun input memory ctx_msg vl id num_expr -> 
    Util.int_of_num (eval_num_arg input memory ctx_msg vl id num_expr)

and (eval_num_arg : Var.env_in -> Var.env -> string -> int -> string -> Exp.num -> Num.num) =
  fun input memory ctx_msg vl id num_expr ->
    	 let gne = num_to_gne input memory ctx_msg vl num_expr in
	   (match Gne.get_constant gne with
	      | Some (I i) ->  i
	      | Some (F f) -> 
		       let errmsg = 
		         "*** Error when calling " ^ id ^ ". " ^
		           (Exp.num_to_string num_expr) ^ " should be an integer.\n"
		       in
		         failwith errmsg
	      | _ ->
		       let errmsg = 
		         "*** Error when calling " ^ id ^ ". " ^
		           (Exp.num_to_string num_expr) ^ 
		           " should be bound (i.e., it should depend " ^ 
		           "only one inputs and memories).\n"
		       in
		         failwith errmsg
	   )

and
    (gne_to_bdd : Gne.t -> comp -> Bdd.t * bool) =
  fun gne cmp ->
    (** Use [cmp] to compare [gne] with 0 and returns the
        corresponding formula.  E.g., if [gne] is bounded to
        [e1 -> c1; e2 -> c2], then [gne_to_bdd gne SupZero] returns
        (the bdd corresponding to) the formula [(c1 and (e1 > 0)) or
        (c2 and (e2 > 0))] 

        Also returns a flag that is true iff [e] depends on pre or
        input vars.
    *)
    match cmp with
	     SupZero ->
	       ( Gne.fold
	           (fun nexpr (c, dep) (acc, dep_acc) ->
		           let new_dep = dep || dep_acc
		           and bdd =
		             if
		               Ne.is_a_constant nexpr
		             then
		               let cst = Ne.find "" nexpr in
		                 match cst with
			                  Some(I(i)) ->
			                    if Num.ge_num i (Num.num_of_int 0)
			                    then (Bdd.dtrue !bdd_manager)
			                    else (Bdd.dfalse !bdd_manager)
			                | Some(F(f)) ->
			                    if f > 0.
			                    then (Bdd.dtrue !bdd_manager)
			                    else (Bdd.dfalse !bdd_manager)
			                | None ->
			                    (Bdd.dfalse !bdd_manager)
		             else
		               Bdd.ithvar
                       (* 		       !bdd_manager *)
		                 (linear_constraint_to_index (Ineq(GZ(nexpr)))
			                 dep)
		           in
		             (Bdd.dor (Bdd.dand c bdd) acc, new_dep)
	           )
	           gne
	           ((Bdd.dfalse !bdd_manager), false)
	       )
      | SupEqZero ->
	       ( Gne.fold
	           (fun nexpr (c, dep) (acc, dep_acc) ->
		           let new_dep = dep || dep_acc
		           and bdd =
		             if Ne.is_a_constant nexpr
		             then
		               let cst = Ne.find "" nexpr in
		                 match cst with
			                  Some(I(i)) ->
			                    if Num.ge_num i (Num.num_of_int 0) 
			                    then (Bdd.dtrue !bdd_manager)
			                    else (Bdd.dfalse !bdd_manager)
			                | Some(F(f)) ->
			                    if f >= 0.
			                    then (Bdd.dtrue !bdd_manager)
			                    else (Bdd.dfalse !bdd_manager)
			                | None ->
			                    (Bdd.dtrue !bdd_manager)

		             else
		               Bdd.ithvar
		                 (linear_constraint_to_index (Ineq(GeqZ(nexpr)))
			                 dep)
		           in
		             (Bdd.dor (Bdd.dand c bdd) acc, new_dep)
	           )
	           gne
	           ((Bdd.dfalse !bdd_manager), false)
	       )
      | EqZero ->
	       ( Gne.fold
	           (fun nexpr (c, dep) (acc, dep_acc) ->
		           let new_dep = dep || dep_acc
		           and bdd =
		             if Ne.is_a_constant nexpr
		             then
		               let cst = Ne.find "" nexpr in
		                 match cst with
			                  Some(I(i)) ->
			                    if Num.eq_num i (Num.num_of_int 0) 
			                    then (Bdd.dtrue !bdd_manager)
			                    else (Bdd.dfalse !bdd_manager)
			                | Some(F(f)) ->
			                    if f = 0.
			                    then (Bdd.dtrue !bdd_manager)
			                    else (Bdd.dfalse !bdd_manager)
			                | None ->
			                    (Bdd.dtrue !bdd_manager)

		             else
		               Bdd.ithvar
		                 (linear_constraint_to_index (EqZ(nexpr)) dep)
		           in
		             (Bdd.dor (Bdd.dand c bdd) acc, new_dep)
	           )
	           gne
	           ((Bdd.dfalse !bdd_manager), false)
	       )

(***********************************************************************)
