(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License
**-----------------------------------------------------------------------
**
** File: fair_bddd.mli
** Main author: jahier@imag.fr
*)

(** Same inteface as Bddd module, but with a different
  implementation; here, the emphasis is put on the fairness of the
  draw, which is not the case of Bddd, as the volume of polyhedron
  and ranges is not taken into account. The most problematic example
  of Bddd is this one:

       0 < x < 1000 and x <> 2

  With Bddd, 1 is draw with a probability of 0.5 ...  On the
  contrary, in this module, we will compute the polyhedron volume.
  This is expensive for 2 reasons:

   - compute polyhedron volume is expensive (even with the
     monte-carlo inspired method we use to find a rough approximation
     of it).  

   - we loose somehow the shareness of bdd, as we need to
     keep track of where we come from at any point of the bdd, in
     order to know to what polyhedron correspond the current path.

  Let me reword this last argument. Because of shareness, any node in
  a bdd may be targetted by different nodes; hence, the t and e
  branches of that node associate different polyhedra, depending on
  which nodes points to it. Therefore, we will store in the snt the
  list of father nodes (via a list of variable indexes) which somehow
  a way of loosing shareness. The reason we keep bdd for that
  purpose is that it makes the implementation simpler as several
  functions can be reused.

*)

open Sol_nb
	
val draw_in_bdd : Var.env_in -> Var.env -> int -> string -> 
  Exp.var list -> Bdd.t -> Bdd.t -> Var.env * Store.t' * Store.p
  (** [draw_in_bdd inputs memory verbose_level msg vars bdd comb]
      returns a draw of the Boolean variables as well as a range based
      and a polyhedron based representation of numeric constraints.

      Raises [No_numeric_solution].
  *)

val sol_number : Bdd.t -> sol_nb * sol_nb
(** Returns an entry of the table *)


(**/**)

val add_snt_entry : Bdd.t -> sol_nb * sol_nb -> unit
(** mofifies an entry. usefull whenever we realize latter that a bdd
 represents a formula that is unsatisfiable for numerical reasons.

  XXX ougth to be dead code for that module as we build polyhedra
  during the table construction, not during the draw, which means
  empty polyhedron are detected then.
*)


val init_snt : unit -> unit
val clear_snt : unit -> unit



