(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License
**-----------------------------------------------------------------------
**
** File: bddd.ml
** Main author: jahier@imag.fr
*)


(** In the following, we call a comb the bdd of a conjunction of
 litterals (var). They provide the ordering in which litterals
 appear in the bdds we manipulate.
*)

open Util
open Exp
open Value
open Constraint
open Store

(****************************************************************************)
open Sol_nb

(*
type sol_nb =  float
let mult_sol_nb n m = n *. m
let zero_sol = 0.
let one_sol = 1.
let eq_sol_nb = (=)
let two_power_of n = 2. ** (float_of_int n)
let string_of_sol_nb = string_of_float
*)

type one_or_two = | One of sol_nb | Two of sol_nb * sol_nb

let (add_one_or_two : sol_nb -> one_or_two -> sol_nb) =
  fun n1 n2 -> 
    match n2 with
      | One m -> add_sol_nb n1 m
      | Two(m1, m2) -> add_sol_nb n1 (add_sol_nb m1 m2)


(** snt Associates to a bdd the (absolute) number of solutions
  contained in its lhs (then) and rhs (else) branches. Note that
  adding those two numbers only give a number of solution that is
  relative to which bdd points to it.

  Note it may happen that the then part is splitted in 2 when we have
  added an equality (ne = a): in that case, it contains the number of
  solutions corresponding to the "ne < a" and to the "ne > a" part.
*)

type snt = (Bdd.t * Store.t, sol_nb * one_or_two) Hashtbl.t

let snt_build = ref false

(* let print_snt snt = *)
(*   Hashtbl.iter *)
(*     (fun (bdd, store) (n,m) ->  *)
(*        Bdd._print bdd; *)
(*        Format.print_string ( *)
(* 	 " -> (" ^ (string_of_float n) ^ ", " ^ (string_of_float n) ^ ")\n") *)
(*     ) *)
(*     snt *)

let snt_ref = ref (Hashtbl.create 1)
exception Solution of sol_nb * one_or_two

(*
   nb : in bddd.ml, it was ok to look at not(bdd) in the snt
   and then switch the sol nb of both branches.*

   Here, it is not the case. Indeed, obtaining the <<negation>>
   of the store (namely, the store obtained by adding the negation of the
   constraints that were added to obtain the store) is not that easy
   as switching the "then" and the "else" branches.
*)
(* exported *)
let (sol_number : Bdd.t -> sol_nb * sol_nb) =
  fun bdd ->
    if      Bdd.is_true bdd then (one_sol, zero_sol)
    else if Bdd.is_false bdd then (zero_sol, one_sol)
    else
      try
	     Hashtbl.iter
	       (fun (bdd0, store) (sol_nb_t, sol_nb_e) ->
	          if
	            bdd = bdd0
	          then
	            raise (Solution(sol_nb_t, sol_nb_e))
	          else
	            ()
	       )
	       !snt_ref;
	     raise Not_found
      with
	       Solution(sol_nb_t, sol_nb_e) ->
	         match sol_nb_e with
	           | One m -> (sol_nb_t, m)
	           | Two (m1,m2) -> (sol_nb_t, (add_sol_nb  m1 m2))
	    
	

let (add_snt_entry : Bdd.t -> sol_nb * sol_nb -> unit) =
  fun bdd (n,m) ->
    assert ((n,m) = (sol_number bdd));
    () (* we already know that in this mode. *)

let (init_snt : unit -> unit) =
  fun _ -> ()

let (clear_snt: unit -> unit) =
  fun _ ->
    Hashtbl.clear !snt_ref;
    snt_build := false;
    init_snt ()

let (sol_number_exists : Bdd.t -> Store.t -> snt -> bool) =
  fun bdd store snt ->
    (* check if the snt has been computed for [bdd] *)
    if      Bdd.is_true bdd then true
    else if Bdd.is_false bdd then false
    else
      Hashtbl.mem snt (bdd, store)

(****************************************************************************)
(****************************************************************************)

type var_idx = int

let rec (build_sol_nb_table_rec: Bdd.t -> Bdd.t -> Store.t -> snt -> sol_nb) =
  fun bdd comb store snt ->
    (** Returns the relative (to which bbd points to it) number of
        solutions of [bdd]. Also udpates the solution number table for [bdd],
        and recursively for all its sub-bdds.

        [comb] is a positive cube that ougth to contain the indexes of
        boolean vars that are still to be generated, and the numerical
        indexes that appears in [bdd].
    *)
    let _ = assert (
      not (Bdd.is_cst bdd) && (Bdd.topvar comb) = (Bdd.topvar bdd) )
    in
    let sol_nb =
      try
	     (* either it has already been computed ... *)
	     let (nt, ne) = hfind snt (bdd, store) in	
	       (* solutions numbers in the table are absolute *)
	       (add_one_or_two nt ne)

      with Not_found ->
	     (* ... or not. *)

	     let bdd_t = (Bdd.dthen bdd)
	     and bdd_e = (Bdd.delse bdd)
	     and cstr = Formula_to_bdd.index_to_linear_constraint (Bdd.topvar bdd)
	     in
	     let (nt, ne) =
	       match cstr with
	           Bv _ ->
		          let nt = compute_absolute_sol_nb bdd_t comb store snt
		          and ne = compute_absolute_sol_nb bdd_e comb store snt
		          in
		            Hashtbl.replace snt (bdd, store) (nt, One ne);
		            (nt, One ne)

	         | Ineq ineq ->
		          let nt =
		            try
		              let store_t = Store.add_constraint store cstr in
		                compute_absolute_sol_nb bdd_t comb store_t snt
		            with
		                Store.No_numeric_solution -> zero_sol
		          in
		          let ne =
		            try
		              let store_e =
		                Store.add_constraint store (Ineq (neg_ineq ineq))
		              in
		                compute_absolute_sol_nb bdd_e comb store_e snt
		            with
		                Store.No_numeric_solution -> zero_sol
		          in
		            Hashtbl.replace snt (bdd, store) (nt, One ne);
		            (nt, One ne)
		              
	         | EqZ ne ->
		          let ne_sup = (Ineq (GZ(ne)))
		          and ne_inf = (Ineq (GZ(Ne.neg_nexpr ne)))
		          in
		          let nt =
		            try
		              let store_eq  = Store.add_constraint store cstr in
		                compute_absolute_sol_nb bdd_t comb store_eq snt
		            with
		                Store.No_numeric_solution -> zero_sol
		          in
		            
		          let n_inf =
		            try
		              let store_inf = Store.add_constraint store ne_inf in
		                compute_absolute_sol_nb bdd_e comb store_inf snt
		            with
		                Store.No_numeric_solution -> zero_sol
		          in

		          let n_sup =
		            try
		              let store_sup = Store.add_constraint store ne_sup in
		                compute_absolute_sol_nb bdd_e comb store_sup snt
		            with
		                Store.No_numeric_solution -> zero_sol

		          in
		            Hashtbl.replace snt (bdd, store) (nt, Two (n_inf, n_sup));
		            (nt, Two(n_inf, n_sup))
	     in
	       (add_one_or_two nt ne)
    in
      sol_nb
and
    (compute_absolute_sol_nb: Bdd.t -> Bdd.t -> Store.t -> snt -> sol_nb) =
  fun sub_bdd comb store snt ->
    (*
      Returns the absolute number of solutions of [sub_bdd] w.r.t. [comb],
      where [comb] is the comb of the father of [sub_bdd].

      The [comb] is used to know which output Boolean variables are
      unconstraint along a path in the bdd. Indeed, the comb is made
      of all the Boolean output var indexes plus the num contraints
      indexes that appears in the bdd; hence, if the topvar of the
      bdd is different from the topvar of the comb, it means that
      the topvar of the comb is unsconstraint and we need to
      multiply the number of solution of the branch by 2 (for each
      missing var).
    *)
    if
      Bdd.is_false sub_bdd
    then
      zero_sol
    else if
      Bdd.is_true sub_bdd
    then
      let bool_sol_nb =
	     if
	       Bdd.is_true comb
	     then
	       one_sol
	     else
	       (*
	         Here, the idea is that the number of solution is the
	         volume of the store multiplied by 2^n, where n is the
	         number of Boolean var that has not been constrainted
	         along the bdd path.
	         
	         Note that Numeric index which do not appear in the bdd path
	         must not  be counted. Indeed, the absence of the constraint
	         do not mean that the variable the contraint is holding on is
	         free.
	       *)
	       
	       let missing_indexes = (Bdd.list_of_support (Bdd.dthen comb)) in
	       let missing_boolean_indexes =
	         (fst
	            (List.partition
		            (fun idx ->
		               match Formula_to_bdd.index_to_linear_constraint idx with
			                Bv _  -> true
		                 | _ -> false
		            )
		            missing_indexes
	            )
	         )
	       in
	         two_power_of (List.length (missing_boolean_indexes))
      in
        mult_sol_nb (sol_nb_of_float (Store.compute_volume 0 store)) bool_sol_nb
	       
    else
      let topvar = Bdd.topvar sub_bdd in
      let rec
	       (count_missing_vars: Bdd.t -> var_idx -> int -> Bdd.t * int) =
	     fun comb var_idx cpt ->
	       (*
	         Returns [cpt] + the number of variables occurring in [comb]
	         before reaching [var_idx] ([var_idx] excluded). Also returns the comb
	         which topvar is [var_idx].
	       *)
	       let _ = assert (not (Bdd.is_cst comb)) in
	       let combvar = Bdd.topvar comb in
	         if
	           var_idx = combvar
	         then
	           (comb, cpt)
	         else
	           let new_cpt =
		          match Formula_to_bdd.index_to_linear_constraint var_idx with
		              Bv _  -> cpt+1
		            | _     -> cpt
	           in
		          count_missing_vars (Bdd.dthen comb) var_idx new_cpt
      in
      let (sub_comb, missing_vars_nb) =
	     count_missing_vars (Bdd.dthen comb) topvar 0
      in
      let n0 = build_sol_nb_table_rec sub_bdd sub_comb store snt in
      let factor = (two_power_of missing_vars_nb) in
	     (mult_sol_nb n0 factor)


(**
  [build_sol_nb_table bdd comb] build a table that associate each to
  a [bdd] its solution number in its 2 branches. [comb] is a bdd made
  of the the conjunctions of variables of [bdd] union the Boolean
  variable to be generated (which migth not appear in [bdd]).
*)	
let (build_sol_nb_table: var list -> Bdd.t -> Bdd.t -> unit) =
  fun num_vars bdd comb ->
    snt_build := true;
    let store0 = Store.create num_vars in
      if
	     (sol_number_exists bdd store0 !snt_ref)
      then
	     ()
      else
	     let rec skip_unconstraint_bool_var_at_top comb v =
	       (* [build_sol_nb_table_rec] supposes that the bdd and its comb
	          have the same top var.
	       *)
	       if Bdd.is_true comb then comb else
	         if
	           v = Bdd.topvar comb
	         then
	           comb
	         else
	           skip_unconstraint_bool_var_at_top (Bdd.dthen comb) v
	     in

	     let comb2 = skip_unconstraint_bool_var_at_top comb (Bdd.topvar bdd) in
	     let _ =
	       build_sol_nb_table_rec bdd comb2 store0 (!snt_ref)
	     in
	       ()


(****************************************************************************)
(****************************************************************************)


let (toss_up_one_var: Var.name -> Var.subst) =
  fun var ->
    let ran = Random.float 1. in
      if (ran < 0.5) then (var, B(true)) else (var, B(false))


let (toss_up_one_var_index: Var.env_in -> Var.env -> int -> string -> var_idx ->
       Var.subst option) =
  fun input memory vl ctx_msg var_idx ->
    (* if [var_idx] is a index that corresponds to a boolean variable,
       this fonction performs a toss and returns a substitution for
       the corresponding boolean variable. It returns [None]
       otherwise (ie if it is a num).

       Indeed, if it happens that a numerical constraint does not
       appear along a path, we'll simply ignore it and hence it will not
       be added to the store.
    *)
    let cstr = Formula_to_bdd.index_to_linear_constraint var_idx in
      match cstr with
          Bv(v) -> (
	    match (Var.default v) with
		Some (Formu f) ->
		  let bdd = Formula_to_bdd.f input memory ctx_msg vl f in
		    if Bdd.is_false bdd then Some((Var.name v), B false)
		    else if Bdd.is_true bdd then Some((Var.name v), B true)
		    else
		      (
			print_string (
			  "*** Default values should not depend on " ^
			  "controllable variables, \nwhich is the case of " ^
			  (formula_to_string f) ^ ", the default value of " ^
			  (Var.name v) ^ "\n");
			exit 2
		      )
	      | Some (Numer _) -> assert false
	      | Some (Liste l) -> assert false
	      | _ -> Some(toss_up_one_var (Var.name v))
	  )
	| _  -> None


let rec (draw_in_bdd_rec: Var.env_in -> Var.env -> int -> string ->
	       Var.env * Store.t -> Bdd.t -> Bdd.t ->
	       Var.env * Store.t' * Store.p) =
  fun input memory vl ctx_msg (sl, store) bdd comb ->
    (** Returns [sl] appended to a draw of all the boolean variables
        bigger than the topvar of [bdd] according to the ordering
        induced by the comb [comb]. Also returns the (non empty) store
        obtained by adding to [store] all the numeric constraints that
        were encountered during this draw.

        Raises the [No_numeric_solution] exception whenever no valid
        path in [bdd] leads to a satisfiable set of numeric
        constraints.
    *)

    let snt = !snt_ref in
      if
        Bdd.is_true bdd
      then
        (* Toss the remaining bool vars. *)
        let (store_int, store_poly) =
	       Store.switch_to_polyhedron_representation vl store
        in
	       (
	         (* (List.append sl *)
	         (Value.OfIdent.add_list sl
	            (Util.list_map_option
		            (toss_up_one_var_index input memory vl ctx_msg)
		            (Bdd.list_of_support comb))
	         ),
	         store_int,
	         store_poly
	       )
      else
        let _ = assert (not (Bdd.is_false bdd)) in
        let _ = assert (sol_number_exists bdd store snt) in
        let bddvar = Bdd.topvar bdd in
        let cstr = (Formula_to_bdd.index_to_linear_constraint bddvar) in
	       match cstr with
	         | Bv(var) ->
	             draw_in_bdd_rec_bool input memory vl ctx_msg var (sl, store) bdd comb snt
	         | EqZ(e) ->
	             draw_in_bdd_rec_eq input memory vl ctx_msg e (sl, store) bdd comb snt
	         |  Ineq(ineq) ->
	              draw_in_bdd_rec_ineq input memory vl ctx_msg ineq (sl, store) bdd comb snt


and (draw_in_bdd_rec_bool: Var.env_in -> Var.env -> int -> string -> var ->
      Var.env * Store.t -> Bdd.t -> Bdd.t -> snt ->
	   Var.env * Store.t' * Store.p) =
  fun input memory vl ctx_msg var (sl, store) bdd comb snt ->
    let bddvar = Bdd.topvar bdd in
    let topvar_comb  = Bdd.topvar comb in

      if
	     bddvar <> topvar_comb
      then
	     (* that condition means that topvar_comb is an unconstraint
	        boolean var; hence we toss it up, or we compute its default
	        value (cf "~default" flag). *)
	     let new_sl =
	       match toss_up_one_var_index input memory vl ctx_msg topvar_comb with
	           (* Some(s) -> s::sl *)
	           Some(s) -> Value.OfIdent.add sl s
	         | None -> sl
	     in
	       draw_in_bdd_rec input memory vl ctx_msg (new_sl, store) bdd (Bdd.dthen comb)
      else
	     (* bddvar = combvar *)
	     let (n, m) = 
	       match hfind snt (bdd, store) with
	           (n, One m) -> n, m
	         | (_, Two _) -> assert false
	     in
	     let _ =
	       if ((eq_sol_nb n zero_sol) && (eq_sol_nb m zero_sol))
	       then raise No_numeric_solution ;
	     in
	     let (
	       bdd1, bool1, sol_nb1,
	       bdd2, bool2, sol_nb2
	     ) =
	       let ran = Random.float 1.
	       and sol_nb_ratio = float_of_sol_nb
	         (div_sol_nb n (add_one_or_two n (One m)))
	       in
	         if
	           ran < sol_nb_ratio
		          (*
		            Depending on the result of a toss (based on the number
		            of solution in each branch), we try the [then] or the
		            [else] branch first.
		          *)
	         then
	           ((Bdd.dthen bdd), true, n,
	            (Bdd.delse bdd), false, m )
	         else
	           ((Bdd.delse bdd), false, m,
	            (Bdd.dthen bdd), true, n )
	     in
	     let (sl1, sl2, new_comb) = (
	       (* (((Var.name var), B(bool1))::sl), *)
	       (* (((Var.name var), B(bool2))::sl), *)
	       Value.OfIdent.add sl (Var.name var, B bool1),
	       Value.OfIdent.add sl (Var.name var, B bool2),
	       (if Bdd.is_true comb then comb else Bdd.dthen comb)
	     )	
	     in
	       (*
	         A solution will be found in this branch iff there exists
	         at least one path in the bdd that leads to a satisfiable
	         set of numeric constraints. If it is not the case,
	         [res_opt] is bound to [None].
	       *)
	       try
	         draw_in_bdd_rec input memory vl ctx_msg (sl1, store) bdd1 new_comb
	       with
	           No_numeric_solution ->
		          if vl >= 2 then
		            print_string "\n..................BACKTRACK!\n";
		          if not (eq_sol_nb sol_nb2 zero_sol)
		          then draw_in_bdd_rec input memory vl ctx_msg (sl2, store) bdd2 new_comb
		            (*
		              The second branch is now tried because no path in
		              the first bdd leaded to a satisfiable set of
		              numeric constraints.
		            *)
		          else raise No_numeric_solution
	         | e ->
		          print_string ((Printexc.to_string e) ^ "\n");
		          flush stdout;
		          assert false
		            
and (draw_in_bdd_rec_ineq: Var.env_in -> Var.env -> int -> string -> Constraint.ineq ->
      Var.env * Store.t -> Bdd.t -> Bdd.t -> snt ->
	   Var.env * Store.t' * Store.p) =
  fun input memory vl ctx_msg cstr (sl, store) bdd comb snt ->
    (*
      When we add to the store an inequality constraint GZ(ne) or
      GeqZ(ne) ([split_store]), 2 stores are returned. The first is a
      store where the inequality has been added; the second is a
      store where its negation has been added.

      Whether we choose the first one or the second depends on a toss
      made according the (boolean) solution number in both branches
      of the bdd. If no solution is found into that branch, the other
      branch with the other store is tried.
    *)
    let (n, m) = 
      match hfind snt (bdd, store) with
	       (n, One m) -> n, m
	     | (_, Two _) -> assert false
    in
    let _ =
      if ((eq_sol_nb n zero_sol) && (eq_sol_nb m zero_sol))
      then raise No_numeric_solution ;
    in
    let ran = Random.float 1. in
    let cstr_neg = Constraint.neg_ineq cstr in
    let (cstr1, bdd1, sol_nb1, cstr2, bdd2, sol_nb2) =
      if
	     ran < float_of_sol_nb (div_sol_nb (n) ((add_one_or_two n (One m))))
      then
	     (cstr, (Bdd.dthen bdd), n, cstr_neg, (Bdd.delse bdd), m)
      else
	     (cstr_neg, (Bdd.delse bdd), m, cstr, (Bdd.dthen bdd), n)
    in
    let store1 = add_constraint store (Ineq cstr1) in
      (* A solution will be found in this branch iff there exists
	      at least one path in the bdd that leads to a satisfiable
	      set of numeric constraints. If it is not the case,
	      [res_opt] is bound to [None]. *)

      try
	     if not (is_store_satisfiable vl store1) then raise No_numeric_solution;
	     draw_in_bdd_rec input memory vl ctx_msg (sl, store1) bdd1 comb
      with
	       No_numeric_solution ->
	         let store2 = add_constraint store (Ineq cstr2) in
	           (*
		          The second branch is tried if no path in the first
		          bdd leaded to a satisfiable set of numeric
		          constraints.
	           *)
	           if
		          (not (eq_sol_nb sol_nb2 zero_sol))
		          && is_store_satisfiable vl store2
	           then
		          draw_in_bdd_rec input memory vl ctx_msg (sl, store2) bdd2 comb
	           else
		          raise No_numeric_solution
	     | e ->
	         print_string ((Printexc.to_string e) ^ "\n");
	         flush stdout;
	         assert false

and (draw_in_bdd_rec_eq: Var.env_in -> Var.env -> int -> string -> Ne.t ->
      Var.env * Store.t -> Bdd.t -> Bdd.t -> snt ->
	   Var.env * Store.t' * Store.p) =
  fun input memory vl ctx_msg ne (sl, store) bdd comb snt ->
    (*
      We consider 3 stores:
      - store + [ne = 0]
      - store + [ne > 0]
      - store + [ne < 0]

      Whether we choose the first one or not depends on toss made
      according the (boolean) solution number in both branches of the
      bdd. If the else branch is choosen (if it is chosen in the
      first place, or if backtracking occurs because no solutions is
      found in the then branch) whether we try the second or the
      third store first is (fairly) tossed up.
    *)
    let (n1, n2, n3) = 
      match (hfind snt (bdd, store)) with
	     | (ne, One _) -> assert false
	     | (ne, Two(nt1, nt2)) -> ne, nt1, nt2
    in
    let _ =
      if (eq_sol_nb (add_sol_nb n1 (add_sol_nb n2 n3)) zero_sol) 
      then raise No_numeric_solution 
    in
    let ran  = Random.float 1.
    and cstr1 = (EqZ ne)
    and cstr2 = (Ineq (GZ (Ne.neg_nexpr ne)))
    and cstr3 = (Ineq (GZ ne))
    in
    let p12 = float_of_sol_nb (div_sol_nb n1 (add_sol_nb n1 n2))
    and p21 = float_of_sol_nb (div_sol_nb n2 (add_sol_nb n1 n2))
    and p23 = float_of_sol_nb (div_sol_nb n2 (add_sol_nb n3 n2))
    and p32 = float_of_sol_nb (div_sol_nb n3 (add_sol_nb n3 n2))
    and p13 = float_of_sol_nb (div_sol_nb n1 (add_sol_nb n1 n3))
    and p31 = float_of_sol_nb (div_sol_nb n3 (add_sol_nb n1 n3))
    in
    let p123 = p12 *. p23
    and p132 = p13 *. p32
    and p213 = p21 *. p13
    and p231 = p23 *. p32
    and p312 = p31 *. p21
      (*     and p321 = p32 *. p21 *)
    in
    let bddt = Bdd.dthen bdd
    and bdde = Bdd.delse bdd 
    in
    let c123 = (
      cstr1, bddt, n1,
      cstr2, bdde, n2,
      cstr3, bdde, n3
    )
    and c132 = ( 
      cstr1, bddt, n1,
      cstr3, bdde, n3,
      cstr2, bdde, n2
    )
    and c213 = (
      cstr2, bdde, n2,
      cstr1, bddt, n1,
      cstr3, bdde, n3
    )
    and c231 = (
      cstr2, bdde, n2,
      cstr3, bdde, n3,
      cstr1, bddt, n1
    )
    and c312 = (
      cstr3, bdde, n3,
      cstr1, bddt, n1,
      cstr2, bdde, n2
    )
    and c321 = (
      cstr3, bdde, n3,
      cstr2, bdde, n2,
      cstr1, bddt, n1
    )
    in
    let (
      cstr1', bdd1, sol_nb1,
      cstr2', bdd2, sol_nb2,
      cstr3', bdd3, sol_nb3) =
      if
	     (ran < p123)
      then
	     c123
      else if
	     (ran < p132 +. p123)
      then
	     c132
      else if
	     (ran < p213 +. p132 +. p123)
      then
	     c213
      else if
	     (ran < p231 +. p213 +. p132 +. p123)
      then
	     c231
      else if
	     (ran < p312 +. p231 +. p213 +. p132 +. p123)
      then
	     c312
      else 
	     c321
    in
    let store1 = add_constraint store cstr1' in
      (*
	     A solution will be found in this branch iff there exists
	     at least one path in the bdd that leads to a satisfiable
	     set of numeric constraints.
      *)
      try
	     if
	       (not (is_store_satisfiable vl store1)) || (sol_nb1 = zero_sol)
	     then
	       raise No_numeric_solution;
	     draw_in_bdd_rec input memory vl ctx_msg (sl, store1) bdd1 comb
      with
	       No_numeric_solution -> 
	         (*
	           The second branch is tried if no path in the first bdd
	           leaded to a satisfiable set of numeric constraints.
	         *)
	         let store2 = add_constraint store cstr2' in
	           try
		          if
		            (not (is_store_satisfiable vl store2)) || (sol_nb2 = zero_sol)
		          then
		            raise No_numeric_solution;
		          draw_in_bdd_rec input memory vl ctx_msg (sl, store2) bdd2 comb
	           with
		            No_numeric_solution ->
		              let store3 = add_constraint store cstr3' in
		                if
			               (not (is_store_satisfiable vl store3)) || (sol_nb3 = zero_sol)
		                then
			               raise No_numeric_solution;
		                draw_in_bdd_rec input memory vl ctx_msg (sl, store3) bdd3 comb
		          | e ->
		              print_string ((Printexc.to_string e) ^ "\n");
		              flush stdout;
		              assert false


(* exported *)
let rec (draw_in_bdd: Var.env_in -> Var.env -> int -> string -> var list -> Bdd.t ->
	   Bdd.t -> Var.env * Store.t' * Store.p) =
  fun input memory vl ctx_msg num_vars bdd comb ->
    build_sol_nb_table num_vars bdd comb;
    let (g,d) = sol_number bdd in
      if 
	(abs_float(float_of_sol_nb g) < !(Util.eps)) 
	&& (abs_float(float_of_sol_nb d) < !(Util.eps)) 
      then
	raise No_numeric_solution
      else
	draw_in_bdd_rec input memory vl ctx_msg (Value.OfIdent.empty, Store.create num_vars) bdd comb
