(** EXPANSION : main

------------------------------------------------------------

L'expansion consiste essentiellement  construire 3 tables globales
indexes par des ident target :

- Table des variables (support) : elle contient les entres/sorties
  et les locales (cf. exist) remontes au top via un nommage unique.

- 2010/04/15 : pour faciliter le traitement des pre
        (en particulier pour les exist avec init), on cree systematiquement
   un ident pour les pre :
        - Input/Output: (pre in/pre out) on cre un alias (e.g. prein = pre in)
        NON (CA MARCHE PAS) - Local : (pre x) on cre une var support prex,
     avec comme default "pre x"
        - Local : il faut aussi un alias, mais plus compliqu car le pre
        est potentiellement rinitialis dans le support_scope.
   L'ide est qu'on associe une variable spciale boolenne associe
   au support_scope soit enter_support_scope.
        si la variable x a une valeur init (e), l'alias est :
      pre_x = if enter_support_scope then e else pre x
        sinon, c'est pas la peine, on peut avoir simplement : 
      pre_x = pre x
        La variable enter_support_scope est une variable support avec
        valeur par dfaut "false", et qui est force  vrai
        sur entre dans le support_scope.

        Problme : il faut se dbrouiller pour ne crer le enter_support_scope
        que si besoin !

        ATTENTION : ne pas confondre "sscope" avec les scopes de l'expansion
          
- Table des alias : associe  des idents cible des expressions algbriques.
    Les idents cible correpondent au instances de macros utilises
    dans le corps du node.

- Table des traces :  chaque trace nomme est associe une expression
  de trace expanse.

----------------------------------------------------------*)

open Hashtbl;;
open Lexeme;;
open LutErrors;;
open Printf;;
open Syntaxe;;
open CkTypeEff;;
open CkIdentInfo;;
open CheckEnv;;
open CoIdent;;
open CoAlgExp;;
open CoTraceExp;;


let dbg = Verbose.get_flag "Expand"

(**************************************************************
Type des infos associes aux variables support, aux alias et
aux traces.
**************************************************************)

(**
L'info associe  une variable support contient sa nature,
l'expression algbrique associe (celle qui permet de faire
rfrence  la variable dans les expressions algbriques),
ainsi que la "pile" permettant le retour au source.

2010/04, rajout :
- ventuelle rfrence au pre
- ventuel support_scope 
- ventuelle valeur initiale 
- ventuelle valeur par dfaut 
*)
type support_nature =
   Input
|  Output
|  LocalIn   (* hidden input *)
|  LocalOut  (* hidden output *)
and support_info = {
	si_ident : CoIdent.t ;
   si_nature : support_nature ;
   si_type : CkTypeEff.t ;
   si_ref_exp : CoAlgExp.t ;
   si_src : CoIdent.src_stack;
   (* on ne la cre qu' la demande *)
   mutable si_pre_ref_exp : CoAlgExp.t option ;
   si_default : CoAlgExp.t option ;
   si_scope : support_scope option ;
   si_init : CoAlgExp.t option ;
   si_range : (CoAlgExp.t * CoAlgExp.t) option;
}

(**************************************************************
LOCAL SCOPE HANDLING (see EXIST_n)
---------------------------------------------------------------
A scope contains a pre (re)init context 
**************************************************************)

and support_scope = {
        sco_src : CoIdent.src_stack;
        mutable sco_escope : CoTraceExp.escope;
}

let new_support_scope (src : CoIdent.src_stack) = (
  let res = {
    sco_src = src;
    (* made on demand *)
    sco_escope = CoTraceExp.new_escope () ;
  } in
    res
)

(**
Alias : L'info contient le type, l'expression  utiliser pour faire
rfrence  l'alias, et celle qui correspond  sa dfinition
*)
type alias_info = {
  ai_type : CkTypeEff.t;
  ai_def_exp : CoAlgExp.t;
  ai_ref_exp : CoAlgExp.t;
  ai_src : CoIdent.src_stack
}

(**
Trace : L'info contient juste l'expression associe
 la dfinition.
*)
type trace_info = {
  ti_def_exp : CoTraceExp.t;
  ti_src : CoIdent.src_stack ;
}

(**
Exception :  pas grand chose  stocker : c'est juste un ident ...
*)
type except_info = {
  ei_src : CoIdent.src_stack ;
}

(**
Externe : il s'agit surtout de garder l'info pour utilisation
    ulterieure
*)
type extern_info = {
  xi_decl : Syntaxe.let_info;
  xi_prof : CkTypeEff.profile;
  xi_src : Lexeme.t ;
}

(**************************************************************
On utilise de plus une table ``dynamique'' qui fait l'association entre
ident source et ident target. Les associations varient au cours de
l'expansion, mais la table est tout de mme globale~:
on se contente d'craser au fur et  mesure les associations, sans
se soucier d'avoir une cohrence globale de la table (le binding a dj t
fait, donc tout est correct).

---------------------------------------------------------------------
Notes sur les instances de macros
---------------------------------------------------------------------
Macros sans paramtres :
------------------------
        Pour les macros sans paramtres, le corps est expans une fois pour toute
        et le mme alias/trace est utilis pour toutes
        les rfrences.
Macros avec paramtres :
------------------------
        Pour les macros avec paramtres, on ne fait pas d'quivalence
        syntaxique. Par exemple, pour~:

                let f (x,y,z : ...) = E in
                ...
                ... f(a,b,c) ... -- instance f1 
                ...
                ... f(a,b,c) ... -- instance f2 
                ...

        La macro f va tre expanse 2 fois, c'est--dire qu'on aurra
        les def. d'alias ou de trace suivantes~:

                f1 = expand(E/x <- f1_x, y <- f1_y, z <- f1_z
                avec f1_x = expand(a), f1_y = expand(b), f1_z = expand(c).

                et idem pour [[f2]].

**************************************************************)


(**************************************************************
Table d'association source/target
***************************************************************
On utilise une table globale. Les associations sont crases
au grs de l'volution du scope, sans aucune vrification
particulire puisqu'on sait que le programme est correct
cot binding.

N.B. la cl (unique) utilise pour dsigner les identificateurs
source correspond  l'instance de la dclaration de l'ident.
Pour savoir  quelle instance de dclaration est associe
une INSTANCE DE RFRENCE, il faut IMPRATIVEMENT
passer par la fonction CheckEnv.binding
**************************************************************)

type target_nat =
   TN_support
|  TN_alias
|  TN_trace
|  TN_except


(* MAIN STRUCTURE (no more global var) *)
type t = {
        nme : string;
        (* unique ident space *)
        idspace : CoIdent.space;
        (* lists, in the order of declaration *)
        mutable alist : CoIdent.t list;
        mutable inlist : CoIdent.t list;
        mutable outlist : CoIdent.t list;
        mutable locinlist : CoIdent.t list;
        mutable locoutlist : CoIdent.t list;
        mutable prelist : (CoIdent.t * support_info) list;
        (* info tabs *)
        stab : (CoIdent.t, support_info) Hashtbl.t;
        atab : (CoIdent.t, alias_info) Hashtbl.t;
        ttab : (CoIdent.t, trace_info) Hashtbl.t;
        etab : (string, extern_info) Hashtbl.t;
        xtab : (CoIdent.t, except_info) Hashtbl.t;
        (* run instance table *)
        mutable runcpt : int;
        runtab: (CoIdent.t, t)  Hashtbl.t;
        (* src -> target tab *) 
        src2target : (Syntaxe.ident, (target_nat * CoIdent.t)) Hashtbl.t ;
        mtrace : CoIdent.t;

        (* global pack info, should be in some global struct shared by several t *)
        gxlist : Syntaxe.except_info list;
}

(* get a run def *)
let get_run_expanded_code it rid = Util.hfind it.runtab rid

(*
Initialize with global infos
only gxlist for the time being ?
*)
let new_empty_t nme mainid gxlist = {
   nme = nme;
	idspace = CoIdent.new_space ();
   alist = [];
   inlist = [];
   outlist = [];
   locinlist = [];
   locoutlist = [];
   prelist = [];
   stab = Hashtbl.create 50;
   atab = Hashtbl.create 50; 
   ttab = Hashtbl.create 50; 
   etab = Hashtbl.create 50; 
   gxlist = gxlist;
   xtab = Hashtbl.create 50;
	runcpt = 0;
	runtab = Hashtbl.create 50;
	src2target = Hashtbl.create 50;
   mtrace = mainid;
}

(* On accde  la table  imprativemnent via les ``mthodes'' suivante *)

(* gestion des instances de run *)
let add_run_instance: t -> string -> t -> string = 
    fun               x     nnme      e ->
(
	x.runcpt <- x.runcpt + 1;
	let inme = Printf.sprintf "%s_%02d" nnme x.runcpt in
	Hashtbl.add x.runtab inme e;
	inme
)

let src_decl_to_target_info (zeres:t) id = (
        Util.hfind zeres.src2target id
)

let src_decl_to_target_id (zeres:t) id = (
        snd (Util.hfind zeres.src2target id)
)

(* L'utilitaire suivant enchane les tapes, 
    N'UTILISER QUE SI ON EST SUR QUE TOUT EST CORRECT ! *)

let src_ref_to_target_id (zeres:t) (env : CheckEnv.t) (id : Syntaxe.ident) = (
        let info = CheckEnv.get_binding env id in
        let id_decl = CkIdentInfo.def_ident info in
        src_decl_to_target_id zeres id_decl
)

let add_support_target_id (zeres:t) (src: Syntaxe.ident) (tgt: CoIdent.t) = (
   Hashtbl.add zeres.src2target src (TN_support, tgt)
)

let add_alias_target_id (zeres:t) (src: Syntaxe.ident) (tgt: CoIdent.t) = (
Verbose.put ~flag:dbg " Add alias %s <- %s\n" (src.it) (CoIdent.to_string tgt);
   Hashtbl.add zeres.src2target src (TN_alias, tgt)
)

let add_trace_target_id (zeres:t) (src: Syntaxe.ident) (tgt: CoIdent.t) = (
Verbose.put ~flag:dbg " Add trace %s <- %s\n" (src.it) (CoIdent.to_string tgt);
   Hashtbl.add zeres.src2target src (TN_trace, tgt)
)

let add_except_target_id (zeres:t) (src: Syntaxe.ident) (tgt: CoIdent.t) = (
   Hashtbl.add zeres.src2target src (TN_except, tgt)
)

(* "expansion" des parametres ref :
    on associe au pa_src le tgtid
    qui DOIT ETRE deja associe a l'arg
*)
let set_ref_param
	(zeres: t)
   (env : CheckEnv.t)
   (pa_src: Syntaxe.ident)
   (arg: Syntaxe.val_exp) =
(
        match (arg.it) with     
         IDENT_n id -> (
                let info = CheckEnv.get_binding env id in
                let id_decl = CkIdentInfo.def_ident info in
                let tgtid = match (src_decl_to_target_info zeres id_decl) with
                        (TN_support, ti) -> ti | _ -> assert false
                in
                add_support_target_id zeres pa_src tgtid
        ) | _ -> assert false
)


(**************************************************************
Table des exceptions 
---------------------------------------------------------------
Ce sont des idents particuliers qui n'apparassent
que dans des constructions particulires (catch, raise)
Inutile (voire dangereux ?) de les mlanger avec les
autres idents.
**************************************************************)

let new_global_except
	(zeres: t)
	(ex: Syntaxe.ident) =
(
        (* on garde l'ident comme target *)     
   let tgtid = CoIdent.get ex.it in
   let src = CoIdent.base_stack () in
        Hashtbl.add zeres.xtab tgtid {
                ei_src = src
        };
        add_except_target_id zeres ex tgtid
)

let new_local_except
	(zeres:t)
	(ex: Syntaxe.ident)
	(sstack: CoIdent.scope_stack) =
(
  (* on cre un ident target ``frais'' *) 
  let tgtid = CoIdent.get_fresh zeres.idspace ex.it in
  let src = CoIdent.get_src_stack ex.src sstack None in 
    Hashtbl.add zeres.xtab tgtid {
      ei_src = src
    };
    add_except_target_id zeres ex tgtid
)

(**************************************************************
Table des variables support
---------------------------------------------------------------
**************************************************************)

(* Utilitaires : renvoient une CoAlgExp.t adquate *)
let alg_exp_of_support_ref (zeres:t) tgtid = (
        (Util.hfind zeres.stab tgtid).si_ref_exp
)

(** Dclaration d'une entre *)
(* input (de node) => on garde l'ident, incontrlable *)
let new_support_input 
        (zeres: t)
        (* info optionnelles *)
        (init: CoAlgExp.t option)
        (i : Syntaxe.ident)
        (te : CkTypeEff.t)
        (sstack : CoIdent.scope_stack) =
(
   let tgtid = CoIdent.get i.it in
	let _ = try (
		let oldinfo = Util.hfind zeres.stab tgtid in
		let oldlxm = CoIdent.head_of_src_stack oldinfo.si_src in
		let msg = "input ident already used at " ^
			(lexeme_line_col oldlxm)
		in
		raise (Compile_error (i.src, msg))
	) with Not_found -> ()
	in
   let src = CoIdent.get_src_stack i.src sstack None in 
   Hashtbl.add zeres.stab tgtid {
		si_ident = tgtid ;
		si_nature = Input ;
		si_type = te;
		si_ref_exp = CoAlgExp.of_support tgtid te false;
		(* a priori pas utilis *)
		(* si_pre_ref_exp = CoAlgExp.of_pre tgtid te ; *)
		si_pre_ref_exp = None ;

		si_src = src;
		si_default = None;
		si_scope = None;
		si_init  = init;
      si_range = None;
	};
	(* on garde la liste pour les sortir dans l'ordre *)
	zeres.inlist <- (tgtid) :: zeres.inlist ;
   add_support_target_id zeres i tgtid
)

(** Dclaration d'une sortie *)
(* output (de node) => on garde l'ident, contrlable *)
let new_support_output
        (zeres: t)
        (* info optionnelles *)
        (init: CoAlgExp.t option)
        (range_opt: (CoAlgExp.t * CoAlgExp.t) option)
        (i : Syntaxe.ident)
        (te : CkTypeEff.t)
        (sstack : CoIdent.scope_stack) =
(
   let tgtid = CoIdent.get i.it in
	let _ = try (
		let oldinfo = Util.hfind zeres.stab tgtid in
		let oldlxm = CoIdent.head_of_src_stack oldinfo.si_src in
		let msg = "output ident already used at " ^
			(lexeme_line_col oldlxm)
		in
		raise (Compile_error (i.src, msg))
	) with Not_found -> ()
	in
   let src = get_src_stack i.src sstack None in
   Hashtbl.add zeres.stab tgtid {
		si_ident = tgtid ;
		si_nature = Output ;
		si_type = te;
		si_ref_exp = CoAlgExp.of_support tgtid te true;
		
		(* a priori pas utilis *)
		(* si_pre_ref_exp = CoAlgExp.of_pre tgtid te ; *)
		si_pre_ref_exp = None ;
		
		si_src = src;
		si_default = None;
		si_scope = None;
		si_init = init;
		si_range = range_opt;
	};
	(* on garde la liste pour les sortir dans l'ordre *)
	zeres.outlist <- (tgtid) :: zeres.outlist ;
	add_support_target_id zeres i tgtid
)

(** Dclaration d'une locale *)
(* local (de node) => on garde l'ident, contrlable *)
let new_support_local
        (zeres: t)
        (* in or out ? *)
        (nat: support_nature) 
        (* info optionnelles *)
        (scope: support_scope option)
        (init: CoAlgExp.t option)
        (range_opt: (CoAlgExp.t * CoAlgExp.t) option) 
        (* info gnrales *)
        (i : Syntaxe.ident)
        (te : CkTypeEff.t)
        (sstack : CoIdent.scope_stack) =
(
	let isout = (nat = LocalOut) in
   let tgtid = CoIdent.get_fresh zeres.idspace i.it in
   let src = get_src_stack i.src sstack None in
	Hashtbl.add zeres.stab tgtid {
		si_ident = tgtid; 
		si_nature = nat; 
		si_type = te;
		(* controlable if LocalOut *)
		si_ref_exp = CoAlgExp.of_support tgtid te isout;

		(* a priori pas utilis *)
		(* si_pre_ref_exp = CoAlgExp.of_pre tgtid te ; *)
		si_pre_ref_exp = None ;

		si_src = src;
		si_default = None;
		si_scope = scope;
		si_init = init;
		si_range = range_opt;
	};
   (* on garde la liste pour les sortir dans l'ordre *)
	let _ = if isout then 
   	zeres.locoutlist <- (tgtid) :: zeres.locoutlist
	else
   	zeres.locinlist <- (tgtid) :: zeres.locinlist
	in
	(* adds the def to the exist-scope if any *)
	let _ = match scope with
	|	None -> ()
	|	Some ctx -> ctx.sco_escope <- CoTraceExp.add_escope ctx.sco_escope (tgtid, init) 
	in
   add_support_target_id zeres i tgtid;
	tgtid
)

(**************************************************************
Table des alias
---------------------------------------------------------------

**************************************************************)

(* Utilitaire, renvoie une CoAlgExp.t adquate *)
let alg_exp_of_alias_ref zeres tgtid = (
        (Util.hfind zeres.atab tgtid).ai_ref_exp
)

(* Unicit des idents target d'alias *)
let fresh_alias_id zeres s = ( CoIdent.get_fresh zeres.idspace s )
let fixed_alias_id _ s = ( CoIdent.get s )

(* creation d'un alias :
   - creation instance->def
   - association source->instance
ATTENTION:
   new_alias enchane les deux !
*) 
let create_alias
  (zeres: t)
  (i : Syntaxe.ident)
  (tgtid : CoIdent.t )
  (edef : CoAlgExp.t)
  (te : CkTypeEff.t )
  (sstack : CoIdent.scope_stack) = (
Verbose.exe ~flag:dbg (fun () -> Printf.printf " Create alias %s = %s\n   type: %s\n"
		(CoIdent.to_string tgtid)
		(CoAlgExp.lus_dumps edef)
		(CkTypeEff.to_string te))
;
   assert (CkTypeEff.is_data te);
   let src = get_src_stack i.src sstack None in
   let ctrl = CoAlgExp.is_controlable edef in
   let eref = CoAlgExp.of_alias tgtid te ctrl in
   Hashtbl.add zeres.atab tgtid {
      ai_type = te ;
      ai_def_exp = edef ;
      ai_ref_exp = eref ;
      ai_src = src ;
   };
   zeres.alist <- tgtid::(zeres.alist)
)
let new_alias
  (zeres: t)
  (i : Syntaxe.ident)
  (tgtid : CoIdent.t )
  (edef : CoAlgExp.t)
  (te : CkTypeEff.t )
  (sstack : CoIdent.scope_stack) = (
      create_alias zeres i tgtid edef te sstack;
      add_alias_target_id zeres i tgtid
)

(*------------------------------------------
Treatment of "pre X":
--------------------------------------------
- In all case:
  an ident preX is created

- then, if X does not belong to a local scope:
  preX is aliased to "pre X"
  (N.B. the possible init val is handled by the declaration)

- else, if X belongs to a local scope, but has no init:
  preX is aliased to "pre X"

- else, if X belongs to a local scope, and has init:
  preX is DECLARED as support 
  "preX = init" is added to the support init constraint
  "preX = pre X" is added to the support other constraint

*)
let new_pre_handler 
        (id : CoIdent.t)
        (id_info : support_info) 
= (
        let te = id_info.si_type in
        let pre_X = CoAlgExp.of_pre id te in
        pre_X
)

let alg_exp_of_support_pre_ref (zeres: t) tgtid = (
        let zevar = Util.hfind zeres.stab tgtid in
        match zevar.si_pre_ref_exp with
        |       Some pe -> pe
        |       None -> (
                (* nouveau pre *)
                let pe = new_pre_handler tgtid zevar in
                zevar.si_pre_ref_exp <- Some pe;
                pe
        )
)

(**************************************************************
Table des traces
---------------------------------------------------------------
**************************************************************)

(* Unicit des idents target de traces *)
let fresh_trace_id zeres s = ( CoIdent.get_fresh zeres.idspace s )
let fixed_trace_id s = ( CoIdent.get s )

(* creation d'une trace :
   - creation instance->def
   - association source->instance
ATTENTION:
   new_trace enchane les deux !
*) 
let create_trace
        (zeres: t)
        (i : Syntaxe.ident)
        (tgtid : CoIdent.t )
        (edef : CoTraceExp.t)
        (sstack : CoIdent.scope_stack) =
(
   let src = get_src_stack i.src sstack None in
   Hashtbl.add zeres.ttab tgtid {
                ti_def_exp = edef ;
                ti_src = src
        };
)
let new_trace
        (zeres: t)
        (i : Syntaxe.ident)
        (tgtid : CoIdent.t )
        (edef : CoTraceExp.t)
        (sstack : CoIdent.scope_stack) =
(
   create_trace zeres i tgtid edef sstack;
   add_trace_target_id zeres i tgtid
)
(**************************************************************
Table des externes (utilisees)
---------------------------------------------------------------
**************************************************************)

let add_used_extern
        (zeres: t)
        (info : CkIdentInfo.t) =
(
        let id = (CkIdentInfo.def_ident info) in
        if (Hashtbl.mem zeres.etab id.it) then ()
        else (
                match CkIdentInfo.get_nature info with
(* HERE: because of (new) handling of external libs
  surely something to modify here ??? *)
                        External_func (Some leti, _, p) -> (
                                let ei = {
                                        xi_decl = leti;
                                        xi_prof = p;
                                        xi_src = id.src
                                } in
                        Hashtbl.add zeres.etab id.it ei; ()
                ) | _ -> assert false
        )
)


(**************************************************************
DUMP
---------------------------------------------------------------
Pour debug etc ...
**************************************************************)
let (dump_src_ctx : t -> string) = 
  fun x -> 
    (String.concat "\n"
       (List.map 
          (fun id -> Lexeme.to_string id.src)
          x.gxlist
          ))


let rec dump (x:t) = (
  let dump_extern_info (tgt : string) (xi : extern_info) = (
    printf "extern %s : %s" tgt 
      (CkTypeEff.prof_to_string xi.xi_prof) ;
    printf "\n" ;
  )
  in
  let dump_support_info (tgt : CoIdent.t) (si : support_info) = (
    let ns = match si.si_nature with
      | Input -> "Input"
      | Output -> "Output"
      | LocalIn -> "LocalIn"
      | LocalOut -> "LocalOut"
    in
      printf "%s (%s):\n" (CoIdent.to_string tgt) ns ;
      printf "  type : %s\n" (CkTypeEff.to_string si.si_type);
      printf "  ref  : "; CoAlgExp.dump si.si_ref_exp ;
      printf "\n  pre  : ";
      let _ = match si.si_pre_ref_exp with
        | None   -> printf "(never used)"
        | Some e -> CoAlgExp.dump e
      in
        printf "\n  dflt : ";
        let _ = match si.si_default with
          | None   -> printf "(not defined)"
          | Some e -> CoAlgExp.dump e
        in
          printf "\n  init : ";
          let _ = match si.si_init with
            | None   -> printf "(not defined)"
            | Some e -> CoAlgExp.dump e
          in
            printf "\n  range : ";
            let _ = match si.si_range with
              | None   -> printf "(not defined)"
              | Some (e1,e2) -> (
                  printf "[ ";
                  CoAlgExp.dump e1;
                  printf " ; ";
                  CoAlgExp.dump e2;
                  printf "]";
                )
            in
              printf "\n  src  :\n" ;
              CoIdent.print_src_stack si.si_src ;
              printf "\n" ;
  )
  in
  let dump_alias_info (tgt : CoIdent.t) (ai : alias_info) = (
    printf "%s : %s = "
      (CoIdent.to_string tgt) (CkTypeEff.to_string ai.ai_type);
    CoAlgExp.dump ai.ai_def_exp ;
    printf "\n";
    CoIdent.print_src_stack ai.ai_src
  )
  in
  let dump_trace_info (tgt : CoIdent.t) (ti : trace_info) = (
    printf "%s = " (CoIdent.to_string tgt);
    CoTraceExp.dump ti.ti_def_exp ;
    printf "\n";
    CoIdent.print_src_stack ti.ti_src
  ) 
in
    printf "$$$$$$$$$ INFO FOR %s $$$$$$$$$$\n" x.nme;
    printf "--- EXTERN TABLE of %s\n" x.nme  ;
    Hashtbl.iter dump_extern_info x.etab;
    printf "--- SUPPORT TABLE of %s\n" x.nme ;
    Hashtbl.iter dump_support_info x.stab;
    printf "--- ALIAS TABLE of %s\n" x.nme ;
    Hashtbl.iter dump_alias_info x.atab;
    printf "--- TRACE TABLE of %s\n" x.nme ;
    Hashtbl.iter dump_trace_info x.ttab;
    let dump_run_info id ex = (
      printf "\n--- RUN TABLE of %s CONTAINS %s\n" x.nme id;
      dump ex
    ) in
      Hashtbl.iter dump_run_info x.runtab;

)

(**************************************************************
EXPANSION (main)
**************************************************************)

(** Expansion des identificateurs dans une expression algbriques

        On 2 (+1) cas :

  1) variable support ou paramtre formel :

  Les deux cas sont similaires car ils correspondent aux
        "feuilles" (on n'a rien  expanser).
        Dans les deux cas, l'ident ``cible'' a t cr
        un bonne fois pour toute  la dclaration (cf.
        [[let]], [[exist]] etc).
        On n'a rien  faire~: on renvoie l'expression
        "cible" de rfrence dj cre.

        2) identificateur dfini :

        il s'agit d'un identificateur dfini en amont
        par un [[let x = e ...]]. Un tel identificateur
        est expans  une seule fois, quand on rencontre
        sa premire rfrence. On traite (expanse)
        alors rcursivement sa dfinition, on cre
        une entre cible (alias ou trace suivant le type).
        Si on rencontre une autre rfrence, on rend le rsultat
        dj calcul.

        3??) identificateur prdfini :

        Pour l'instant il n'y a rien de tel dans le langage

*)
let rec treat_exp_ident_ref 
    (zeres: t)
    (env : CheckEnv.t)
    (sstack : scope_stack)
    (id : Syntaxe.ident) 
    = (
      (* info associe  id au cours du check *)
      let info = CheckEnv.get_binding env id in
        (* predef ? *)
        if (CkIdentInfo.is_predef info) then (
          (* pas grand chose  faire .. *)
          CoAlgExp.of_const id.it (CkIdentInfo.get_type info)
        ) else (
          (* dj trait ? *)
          let id_decl = CkIdentInfo.def_ident info in
            try (
               match (src_decl_to_target_info zeres id_decl) with
               | (TN_support, tgtid) -> (
                  alg_exp_of_support_ref zeres tgtid
               )
               | (TN_alias, tgtid) -> (
                  alg_exp_of_alias_ref zeres tgtid
               )
               | (TN_except, tgtid) -> raise (
                  Internal_error ("Expand.treat_exp_ident_ref",
                     "unexpected exception ident"^tgtid)
               )
               
               | (TN_trace, tgtid) -> raise (
                  Internal_error ("Expand.treat_exp_ident_ref",
                     "unexpected trace ident"^tgtid)
               )
            ) with Not_found -> (
              (* pas encore vu ... *)
              match (CkIdentInfo.get_nature info) with
                  Def_ident li -> (
                    (* forcement defini ! *)
                    let def = match li.lti_def with
                        Some d -> d
                      |  _ -> assert false
                    in
                    let newscope = get_scope id_decl.src
                      (get_src_stack id.src sstack None) in
                    let edef = treat_exp zeres env newscope def in
                    let tgtid = fresh_alias_id zeres id.it in
                    let te = (CkIdentInfo.get_type info) in 
                      new_alias zeres id_decl tgtid edef te sstack ;
                      alg_exp_of_alias_ref zeres tgtid
                  )|  _ -> (assert false)
            )
        )
    )
and
    (** Expansion des identificateurs dans une expression trace

        Le traitement est similaire au cas algbrique,  part que
        les cas "impossibles" ne sont pas les mmes et qu'il faut
        expliciter un ventuel "cast" bool -> trace

        N.B. la notion de trace "predef" n'existe pas
        pour l'instant, mais on prvoie quand mme le cas ...
    *)
    treat_trace_ident_ref 
      (zeres: t)
      (env : CheckEnv.t)
      (sstack : scope_stack)
      (id : Syntaxe.ident) 
      (e:Syntaxe.val_exp)
    = (
      (* info associe  id au cours du check *)
      let info = CheckEnv.get_binding env id in
Verbose.put ~flag:dbg "treat_trace_ident_ref %s\n" (CkIdentInfo.to_string info);
        (* predef ? *)
        if (CkIdentInfo.is_predef info) then (
          (* rien  faire *)
          assert false
        ) else (
          (* dj trait ? *)
          let id_decl = CkIdentInfo.def_ident info in
            try (
               match (src_decl_to_target_info zeres id_decl) with
               | (TN_trace, tgtid) -> (
                  CoTraceExp.of_ref tgtid
               )
               | (TN_support, tgtid) -> (
                  (* cast *)
                  CoTraceExp.of_constraint (alg_exp_of_support_ref zeres tgtid)
                     (id.src, get_src_stack id.src sstack (Some e))
               )
               | (TN_alias, tgtid) -> (
                  (* cast *)
                  CoTraceExp.of_constraint (alg_exp_of_alias_ref zeres tgtid)
                    (id.src, get_src_stack id.src sstack (Some e))
               )
               | (TN_except, tgtid) -> (
                  (* erreur interne ! on n'a rien  faire l !!! *)
                  assert false
               )
            ) with Not_found -> (
              (* pas encore vu ... *)
              match (CkIdentInfo.get_nature info) with
					| Def_ident li -> (
                    (* Forcement defini ! *)
                    let def = match li.lti_def with
                        Some d -> d
                      |  _ -> assert false
                    in
                    let newscope = get_scope id_decl.src
                      (get_src_stack id.src sstack (Some e)) in
                    let edef = treat_trace zeres env newscope def in
                    let tgtid = fresh_trace_id zeres id.it in
                      new_trace zeres id_decl tgtid edef sstack ;
                      CoTraceExp.of_ref tgtid
					)
					| _ -> (
						Printf.fprintf stderr "unexpected ident:\n %s\n" (CkIdentInfo.to_string info);	
						assert false
					)
            )
        )
    )
  (** Expansion des appels  (Gnralits)
      ------------------------------------------------------

      Le traitement est assez similaire, que l'on soit en train
      d'expanser une trace ou une expression algbrique.

      Dans les deux cas, on expanse chaque instance, sans chercher
       tenir compte d'une ventuelle quivalence
      (c'est donc diffrent du traitement des identificateurs
      simples qui sont expanss une seule fois).

      Le principe est le suivant, on est sur une instance de la forme :
      idref (e1, e2, e3, ...)
      On retrouve (via CheckEnv) que cette instance est lie :
      -  un oprateur prdfini, auquel ca on ne fait rien,
      -  une macro dfinie de la forme : iddef (x1, x2, x3, ...) = e

      On commence par traiter les arguments~; pour chaque xi :
      - on expanse [[ei]] dans l'environnement courant : [[expand-ei]],
      - on cre un ident target uniq-xi (trace/alias),
      - on associe [[expand-ei]]  [[uniq-xi]] dans la table
      correspondante (trace/alias),

      Puis on expanse [[e]]~:
      - dans un nouvel environnement ou chaque ident (de dclaration)
      [[xi]] est associ  l'ident cible [[uniq-xi]] correspondant,
      - on expanse [[e]],
      - on cre un ident target [[uniq-idref]], qu'on associe
       [[expand-e]] dans la table correspondante (trace/alias),
      - on renvoie un rfrence  [[uniq-idref]] (trace ou alias suivant le cas).

  *)
and treat_exp_call
    (zeres: t)
    (env : CheckEnv.t)
    (e : Syntaxe.val_exp) (* only for retrieving the right type *)
    (sstack : scope_stack)
    (id : Syntaxe.ident) 
    (args : Syntaxe.val_exp list) 
    (e : Syntaxe.val_exp) 
= (
   (** Expansion des appels  (cas des expression algbriques)
       On n'a pas de soucis de cast implicite bool/trace : on
       est assur de ne manipuler que des expressions algbriques.
   *)
   (* info associe  id au cours du check *)
   let info = CheckEnv.get_binding env id in
Verbose.put ~flag:dbg "treat_exp_call %s (%s)\n" (Lexeme.to_string id.src)
 (CkIdentInfo.to_string info);
   (* predef/extern => on n'a rien  expanser !
      - les extern sont mises dans la table
      - les arguments d'externes doivent tre non-contrlables     
      - TODO: les arguments de modulo, div devrait etre jet ici aussi !
   *)
   if (CkIdentInfo.is_predef info) then (
      let aelist = List.map (treat_exp zeres env sstack) args in
      CoAlgExp.of_call
           (CoIdent.from_string id.it)
           (CheckEnv.get_exp_type env e)
           aelist
   ) else if (CkIdentInfo.is_extern info) then (
      let aelist = List.map (treat_exp zeres env sstack) args in
      Verbose.put ~flag:dbg "Expand.treat_exp_call: external func \"%s\"\n" info.ii_name;
      add_used_extern zeres info ;
      (* non contrlable ? *)
      let check_unctl esrc eexp = (
         if (CoAlgExp.is_controlable eexp) then (
            Verbose.exe ~flag:dbg (fun () -> CoAlgExp.dumpf stderr eexp);
            raise ( Compile_error (esrc.src,
                 "extern. func. arguments must be uncontrollable"))
         ) else ()
      ) in
        List.iter2 check_unctl args aelist;
        let ii = CheckEnv.get_ident_info env id in 
        match ii.ii_nature with
        | External_func (_, Some ei, prof) -> 
           CoAlgExp.of_external_call
              (CoIdent.from_string id.it)
              ei
              prof
              (CheckEnv.get_exp_type env e)
              aelist
        | External_func (_, None, prof) ->
              raise ( Compile_error (id.src,
                "don't know where to find external function, use -L option"
              ))
        | _ -> assert false
     ) else (
        (* ident de dclaration : utile ? *)
        let iddecl = CkIdentInfo.def_ident info in
          (* info est une Macro_ident avec forcement 
             une def (le cas extern est deja traite) *)
        let (params, def) = (
           match (CkIdentInfo.get_nature info) with
           | Macro_ident (Some li, _) -> (
               match li.lti_inputs with
               | Some tilst -> (
                  match li.lti_def with
                    |  Some d ->
                         (* (List.map fst tilst, d) *)
                         (tilst, d)
                    | None -> (assert false)
               )
               | _ -> (assert false)
           ) | _ -> assert false   
        ) in 
            (* Il y a forcement une def !
               l'instance de call induit
               un nouveau "scope" dans lequel on va pouvoir
               situer les rfrences aux paramtres formels,
               sinon, similaire a un appel predef 
            *)
        let newscope = get_scope iddecl.src (get_src_stack id.src sstack  (Some e)) in
        (*
           Passage par valeur :
           on cre/associe les alias correspondants,
           pour chaque couple (pa_decl, arg) :
           - on expanse pa_arg -> pa_val
           - on cre un ident cible pa_target
           - on cre une dfinition d'alias
           N.B. ON POSTPONE l'association d'alias 
               pour ne pas interfrer avec l'valuation
               des param suivants ! 
           pa_target = pa_val
           Passage par reference :
           la dclaration du param est DIRECTEMENT lie au tgtid pass en parametre
         *)
         let postponed_aliases = ref [] in
         let treat_param (par : Syntaxe.ident * Syntaxe.type_exp) (arg : Syntaxe.val_exp) = (
            let pa_decl = fst par in
Verbose.put ~flag:dbg "  treat_param %s\n" pa_decl.it;
            let pa_type = snd par in
            let te = CkTypeEff.of_texp pa_type in
            if (CkTypeEff.is_ref te) then (
               set_ref_param zeres env pa_decl arg
            ) else (
               (* passage par valeur *)
               let pa_val = treat_exp zeres env sstack arg in
               let pa_target = fresh_alias_id zeres pa_decl.it in    
               postponed_aliases := (pa_decl,pa_target)::!postponed_aliases;
               create_alias zeres pa_decl pa_target pa_val te newscope 
            )
         ) in
         List.iter2 treat_param params args ;
         (* on traite les alias postpon *)
         List.iter (fun (s,t) -> add_alias_target_id zeres s t) !postponed_aliases;
         (* on expanse l'expression de id dans ce nouveau contexte *)
         let edef = treat_exp zeres env newscope def in
         let tgtid = fresh_alias_id zeres id.it in
         let te = (CkIdentInfo.get_type info) in
         new_alias zeres id tgtid edef te sstack;
         alg_exp_of_alias_ref zeres tgtid
     )
)
and
    (** Expansion des appels  (cas des expression de trace)

        Le cas est trs similaire  celui des algbriques, si ce n'est
        qu'on doit faire attention au cast implicite entre bool et trace :

        - si le call lui-mme est un bool, on passe par [[treat_exp_call]]
        et on fait un cast explicite du rsultat. On n'a pas besoin (?) de
        se soucier du type
        des arguments, vu qu'une trace ne peut pas participer au calcul
        d'une valeur algbrique !

        - si le call est une trace, il faut encore passer en revue le type
        DCLAR de chaque paramtre pour lui associer la bonne information
        (alias ou trace).

        En fait, comme on sait que le programme est correct, on se contente
        de se poser la question du cast :
        - pour chaque param: suivant son type on l'expanse comme une trace
        ou une expression algbrique, et on met le rsultat dans la table
        des traces ou des alias,
        - pour le rsultat : pareil, suivant son type on l'expanse comme une
        trace ou une exp. alg., on le met dans la bonne table, et finalement
        on caste en trace si ce n'est en est pas une.
        
        REMARQUE : il y a beaucoup de duplication entre les deux cas,
        en particulier toute la partie traitement des arguments
        pourrait tre partage~; en effet le cas "exp" n'est qu'un
        cas simplifi du cas "trace"
    *)
treat_trace_call
   (zeres: t)
   (env : CheckEnv.t)
	(e : Syntaxe.val_exp) (* only for retrieving the right type *)
   (sstack : scope_stack)
   (id : Syntaxe.ident) 
   (args : Syntaxe.val_exp list) 
   (e : Syntaxe.val_exp)
= (
   (* info associe  id au cours du check *)
   let info = CheckEnv.get_binding env id in
Verbose.put ~flag:dbg "treat_trace_call %s (%s)\n" (Lexeme.to_string id.src) (CkIdentInfo.to_string info);
   (* predef/extern => c'est forcement un cast ! *)
     if ( (CkIdentInfo.is_predef info) or (CkIdentInfo.is_extern info)) then (
       let aelist = List.map (treat_exp zeres env sstack) args in
         if (CkIdentInfo.is_extern info) then (
           add_used_extern zeres info ;
           (* non contrlable ? *)
           let check_unctl esrc eexp = (
             if (CoAlgExp.is_controlable eexp) then (
               raise ( Compile_error (esrc.src,
                                      "extern. func. arguments must be uncontrollable"))
             ) else ()
           ) in
             List.iter2 check_unctl args aelist
         ) else ();
         let ae = CoAlgExp.of_call
           (CoIdent.from_string id.it)
           (CheckEnv.get_exp_type env e)
           (* (CkIdentInfo.get_type info) *)
          aelist
      in CoTraceExp.of_constraint ae (id.src, get_src_stack id.src sstack (Some e))
   ) else (
      (* ident de dclaration : utile ? *)
      let iddecl = CkIdentInfo.def_ident info in
      (* info est une Macro_ident avec forcement 
         une def (le cas extern est deja traite) *)
      let (params, def) = (
         match (CkIdentInfo.get_nature info) with
         | Macro_ident (Some li, _) -> (
             match li.lti_inputs with
             | Some tilst -> (
                match li.lti_def with
                | Some d -> (tilst, d)
                (* (List.map fst tilst, d) *)
                | None -> (assert false)
             )
             | _ -> (assert false)
         ) | _ -> assert false   
      ) in 
      (* l'instance de call induit un nouveau "scope"
         dans lequel on va pouvoir situer les rfrences
         aux paramtres formels
      *)
      let newscope = get_scope iddecl.src (get_src_stack id.src sstack (Some e)) in
      (* on cre/associe les alias correspondants,
         pour chaque couple (pa_decl, arg) :
         - on expanse pa_arg -> pa_val (suivant le type)
         - on cre un ident cible pa_target
         - on cre une dfinition (alias ou trace)
           NB. on postpone l'insertion (alias ou trace)
           pa_target = pa_val
      *)
      let postponed_aliases = ref [] in
      let postponed_traces = ref [] in
      let treat_param 
          (pa : Syntaxe.ident * Syntaxe.type_exp)
          (arg : Syntaxe.val_exp)
          = (
            let pa_decl = fst pa in
            let pa_type = snd pa in
            let te = CkTypeEff.of_texp pa_type in
              if (CkTypeEff.is_ref te) then (
                (* c'est forcement un data ... *)
                set_ref_param zeres env pa_decl arg
              ) else (
                (* trace ou data ... *)
                if (te = CkTypeEff.trace) then (
                  let pa_val = treat_trace zeres env sstack arg in
                  let pa_target = fresh_trace_id zeres pa_decl.it in        
                    postponed_traces := (pa_decl,pa_target)::!postponed_traces;
                    new_trace zeres pa_decl pa_target pa_val newscope 
                ) else (
                  let pa_val = treat_exp zeres env sstack arg in
                  let pa_target = fresh_alias_id zeres pa_decl.it in        
                    postponed_aliases := (pa_decl,pa_target)::!postponed_aliases;
                    create_alias zeres pa_decl pa_target pa_val te newscope 
                )
              )
          ) 
      in
      List.iter2 treat_param params args ;
      (* on traite les alias/traces postpons *)
      List.iter (fun (s,t) -> add_alias_target_id zeres  s t) !postponed_aliases;
      List.iter (fun (s,t) -> add_trace_target_id zeres  s t) !postponed_traces;

      (* on expanse l'expression de id dans ce nouveau contexte *)
      let te = (CkIdentInfo.get_type info) in
        if (te =  CkTypeEff.trace) then (
          let edef = treat_trace zeres env newscope def in
          let tgtid = fresh_trace_id zeres id.it in
            new_trace zeres id tgtid edef (e.src, get_src_stack e.src sstack (Some e));
            CoTraceExp.of_ref tgtid
        ) else (
          let edef = treat_exp zeres env newscope def in
          let tgtid = fresh_alias_id zeres id.it in
            new_alias zeres id tgtid edef te sstack;
            CoTraceExp.of_constraint(alg_exp_of_alias_ref zeres tgtid) newscope
        )
   )
)
and
    (** Expansion des expressions algbrique (cas gnral)

        C'est essentiellement une fonction val_exp -> CoAlgExp.t
        Les autres paramtres sont:
        - env : pour rcuprer les infos relatives aux idents source,
        - sstack : la "pile" des infos source pour localiser
        la trace en cours de traitement (messages d'erreur).

        Le typage ayant russi, on est sur de ne pas avoir affaire
         une expression de trace.
    *)
    treat_exp
      (zeres: t)
      (env : CheckEnv.t)
      (sstack : scope_stack) 
      (e : Syntaxe.val_exp)
    = (
      match e.it with   
          (* constantes *)
          TRUE_n -> CoAlgExp.of_true
        |  FALSE_n -> CoAlgExp.of_false
             (*
               |  ICONST_n id -> (CoAlgExp.of_iconst id)
               |  RCONST_n id -> (CoAlgExp.of_rconst id)
             *)
        |  ICONST_n id -> (CoAlgExp.of_iconst id.it)
        |  RCONST_n id -> (CoAlgExp.of_rconst id.it)
             (* pre => forcment un support
             *)
        |  PRE_n id -> (
             let info = CheckEnv.get_binding env id in
             let id_decl = CkIdentInfo.def_ident info in
             let tgtid = src_decl_to_target_id zeres id_decl in 
               alg_exp_of_support_pre_ref zeres tgtid
           )
             (* ident/pre *)
        |       IDENT_n id -> (
                  treat_exp_ident_ref zeres env sstack id
                )
                  (* nouvelle macro :
                     rien  faire, on expanse " la demande"
                  *)
        |       LET_n (_, e) -> (
                  treat_exp zeres env sstack e
                )
                  (* oprateur *)
        |  CALL_n (id, args) -> (
             treat_exp_call zeres env e sstack id args e
           )
             (* traces => impossible *)
        |  EXIST_n _ 
        |  RUN_n  _ 
        |  ERUN_n  _ 
        |  EXCEPT_n _ 
        |  FBY_n _
        |  CHOICE_n _
        |  PRIO_n _
        |  PARA_n _
        |  LOOP_n _
        |  LOOPI_n _
        |  LOOPA_n _
        |  RAISE_n _
        |  TRY_n _
        |  CATCH_n _
        |  TRAP_n _
        |  ASSERT_n _ -> raise (
             Internal_error ("Compile.treat_exp","unexpected case")
           ) 
    )
and
    (** Expansion des expressions de trace (cas gnral)

        C'est essentiellement une fonction val_exp -> CoTraceExp.t
        Les autres paramtres sont :
        - env : pour rcuprer les infos relatives aux idents source,
        - sstack : la "pile" des infos source pour localiser
        la trace en cours de traitement (messages d'erreur).

    *)
    (* utile : value l'ventuelle valeur initiale dans
       une dclaration d'ident
    *)
    eval_init_in_var_decl
      (zeres: t)
      (env : CheckEnv.t)
      (sstack : scope_stack) 
      (i, t, vopt, range_opt) =
  (
    let range_exp =
      (match range_opt with
         | None -> None
         | Some (low, high) -> 
             let low_exp  = treat_exp zeres env sstack low
             and high_exp = treat_exp zeres env sstack high in
               
            if (CoAlgExp.is_controlable low_exp) then (
              raise ( Compile_error (i.src, "range values should not be controllable"))
            ) else 
              if (CoAlgExp.is_controlable high_exp) then (
                raise ( Compile_error (i.src, "range values should not be controllable"))
            ) else Some(low_exp, high_exp)
      )
    in
    let iexp = match vopt       with
      | None -> (
          None
        )
      | Some e -> (
          let res = treat_exp zeres env sstack e in
            (*
              Verbose.exe ~level:2 (fun () ->
              Verbose.put " oui : " ;
              SyntaxeDump.set_err ();
              SyntaxeDump.dump_exp e;
              Verbose.put  " -> " ;
              CoAlgExp.dumpf stderr res;
              Verbose.put  "\n" ;
              );
            *)
            if (CoAlgExp.is_controlable res) then (
              raise ( Compile_error (i.src, "init values should not be controllable"))
            ) else Some res
        ) in
   	let te = CkTypeEff.of_texp t in
      (i, te, iexp, range_exp)
  )
and
(* similar to eval_init_in_var_decl except that :
	- no range
	- types are given as a tuple
*)
    eval_init_in_run_decl
      (zeres: t)
      (env : CheckEnv.t)
      (sstack : scope_stack) 
      (i, topt, vopt)
		te
	=
  (
    let iexp = match vopt       with
      | None -> ( None)
      | Some e -> (
          let res = treat_exp zeres env sstack e in
            (*
              Verbose.exe ~level:2 (fun () ->
              Verbose.put " oui : " ;
              SyntaxeDump.set_err ();
              SyntaxeDump.dump_exp e;
              Verbose.put  " -> " ;
              CoAlgExp.dumpf stderr res;
              Verbose.put  "\n" ;
              );
            *)
            if (CoAlgExp.is_controlable res) then (
              raise ( Compile_error (i.src, "init values should not be controllable"))
            ) else Some res
        ) in
      (i, te, iexp, None)
  )
and
    treat_trace
      (zeres: t)
      (env : CheckEnv.t)
      (sstack : scope_stack) 
      (e : Syntaxe.val_exp) 
    = (
      match e.it with   
          (* certainement cast implicite bool -> trace *)
          TRUE_n
        |  FALSE_n
        |  PRE_n _ -> (
             (* FAUX POUR CE QUI EST DU CALL !!!! *)
             let ae = treat_exp zeres env sstack e in
               CoTraceExp.of_constraint ae (e.src, get_src_stack e.src sstack (Some e))
           )
             (* impossible aprs le type-checking ! *)
        |  ICONST_n id
        |  RCONST_n id -> 
             raise (Internal_error ("Compile.treat_trace", "trace exp expected"))
               (* nouvelle macro *)
        |  LET_n (_, e) -> (
             (* Traitement des LET_n :
                Les [[let x = ...]], sans paramtres ( ne pas confondre avec les
                [[let x () = ...]]) sont expanss une seule fois,
                en un alias (ou une trace) unique, valable pour toutes
                les rfrences  [[x]].

                Le traitement n'est pas ffectu ici, mais  la premire
                utilisation. De cette manire, on ne fait pas de diffrence
                avec les [[let x = ...]] dfinis en dehors du noeud principal.

                En bref, sur un "let x" on n'a rien de spcial  faire !
             *)
             treat_trace zeres env sstack e
           )
             (* ident/call => dpend du type *)
        |       IDENT_n id -> (
                  treat_trace_ident_ref zeres env sstack id e
                )
        |  CALL_n (id, args) -> (
             treat_trace_call zeres env e sstack id args e
           )
             (* Nouveau support *)
        |  EXIST_n (idlst,ee) -> (
             (** Traitement des EXIST_n :
                 Pour chaque identificateur de [[idlst]], on cre
                 un target ident unique qu'on insre dans la table support,
                 puis on appele recursivement le traitement sur [[ee]]
             *)
             (* on value D'ABORD les vopt dans env
                (voir eval_init_in_var_decl)
             *)
             let newidlist = List.map (eval_init_in_var_decl zeres env sstack) idlst in
             let src_scope = CoIdent.get_src_stack e.src sstack (Some e) in 

             (* The exist-scope is created EMPTY ... *)
             let zescope = new_support_scope src_scope in
				(* ... then filled by added var one by one *)
             let f = function (i, t, ivopt, range_opt) -> (
               let _ = new_support_local zeres LocalOut (Some zescope) ivopt range_opt i t sstack
					in ()
             ) in
             List.iter f newidlist;
             (* on continu dans ce nouvel env ... *)
             let res = treat_trace zeres env sstack ee in
				   CoTraceExp.of_exist zescope.sco_escope res
           )
      |  ERUN_n (varlist, edef, e1) -> (
         (* edef doit etre un node call (pour l'instant !) *)
         match edef.it with
         | CALL_n (id, elst) -> (
            (* doit tre un node ... *)
            match (CheckEnv.nature_of_ident env id) with
            | Node_ident (Some ni, prof) -> (
					(* COMPILE le noued *)
					let recexpand = treat_node env ni zeres.gxlist in
					let instanceid = add_run_instance zeres id.it recexpand in

					(* EVALUE LES ARGUMENTS du run dans l'env courant *)
					let eval_arg e = (
						let ce = treat_exp zeres env sstack e in
						let _ = if (CoAlgExp.is_controlable ce) then (
                     raise ( Compile_error (e.src, "erun args sould not be controllable"))
						) in
						ce
					) in
      			let args = List.map eval_arg elst in

					(* EVALUE les eventuels init des resultat  dans l'env courant *)
					let outtypes = CkTypeEff.tuple_to_data_list (CheckEnv.get_exp_type env edef) in
             	(* on value D'ABORD les vopt dans env (voir eval_init_in_var_decl) *)
             	let newvarlist = List.map2 (eval_init_in_run_decl zeres env sstack) varlist outtypes in
             	let src_scope = CoIdent.get_src_stack e.src sstack (Some e) in 

					(* CREE un nouvel env *)
					(* The run-scope is created EMPTY ... *)
					let zescope = new_support_scope src_scope in
					(* ... then filled by added var one by one *)
					let f = function (i, t, ivopt, range_opt) -> (
						let _ = new_support_local zeres LocalIn (Some zescope) ivopt range_opt i t sstack
						in ()
					) in
					List.iter f newvarlist;

					(* on continu dans ce nouvel env ... *)
					let res = treat_trace zeres env sstack e1 in
					CoTraceExp.of_erun instanceid zescope.sco_escope args res 

            )
            (* ... ou une fonction externe 
            | External_func (lio, eio, prof) -> (
               let tel = rec_list_call elst in
               [ match_type_profile tel prof e.src ]
            )
				*)
            | _ -> (
               raise (Compile_error (e.src,
                  "identifier "^id.it^" cannot be used in run statement"))
            )
         )
         | _ -> raise (Compile_error
            (edef.src, "only node calls are supported in run statement"))
             (* traces *)
        )
      |  RUN_n (idlist, edef, e1opt) -> (
         (* edef doit etre un node call (pour l'instant !) *)
         match edef.it with
         | CALL_n (id, elst) -> (
            (* doit tre un node ... *)
            match (CheckEnv.nature_of_ident env id) with
            | Node_ident (Some ni, prof) -> (
					(* COMPILE le noued *)
					let recexpand = treat_node env ni zeres.gxlist in
					let instanceid = add_run_instance zeres id.it recexpand in

					(* EVALUE LES ARGUMENTS du run dans l'env courant *)
					let eval_arg e = (
						let ce = treat_exp zeres env sstack e in
						let _ = if (CoAlgExp.is_controlable ce) then (
                     raise ( Compile_error (e.src, "run args sould not be controllable"))
						) in
						ce
					) in
      			let args = List.map eval_arg elst in

					(* on se resert de ce qui a ete fait pour ERUN,
						pour chaque id de idlist
						- on verifie qu'il correspond a un CTRL dans l'env courant
						- on cree un scope ou l'ident devient un LocalIn
					*)
             	let src_scope = CoIdent.get_src_stack e.src sstack (Some e) in 
					let zescope = new_support_scope src_scope in
					let treat_res_id id = (
      				let info = CheckEnv.get_binding env id in
          			let id_decl = CkIdentInfo.def_ident info in
               	let res = match (src_decl_to_target_info zeres id_decl) with
               	| (TN_support, tgtid) -> (
							let curbinding = Util.hfind zeres.stab tgtid in
							let _ = if (not (CoAlgExp.is_controlable curbinding.si_ref_exp)) then (
                     	raise ( Compile_error (id.src, "run result must be controllable"))
							) in
							(* ICI : l'expression d'init est le pre de tgtid *)
							let init = alg_exp_of_support_pre_ref zeres tgtid in
							let tgtid' = new_support_local zeres LocalIn (Some zescope)
								(* curbinding.si_init curbinding.si_range id curbinding.si_type sstack; *)
								(Some init) None id curbinding.si_type sstack
							in
							let newbinding = Util.hfind zeres.stab tgtid' in
							CoAlgExp.of_eq curbinding.si_ref_exp newbinding.si_ref_exp
						)
						| _ -> assert false
						in
						res
					) in
					(* l'expression AND(xi = xi') est construite une fois pour toute *)
					let zeconstraint = CoAlgExp.of_big_and (List.map treat_res_id idlist) in
					
					let res = match e1opt with
					| Some e1 ->	
						(* on continu dans ce nouvel env ... *)
						treat_trace zeres env sstack e1
					| None ->
						(* ICI : pas efficace, faire un TE_run avec in optionnel *)
						CoTraceExp.of_loop (CoTraceExp.of_constraint (CoAlgExp.of_true) 
                                        (sstack))
					in
					CoTraceExp.of_run instanceid zeconstraint zescope.sco_escope args res  
                 (edef.src, get_src_stack e.src sstack (Some e))
            )
            (* ... ou une fonction externe 
            | External_func (lio, eio, prof) -> (
               let tel = rec_list_call elst in
               [ match_type_profile tel prof e.src ]
            )
				*)
            | _ -> (
               raise (Compile_error (e.src,
                  "identifier "^id.it^" cannot be used in run statement"))
            )
         )
         | _ -> raise (Compile_error
            (edef.src, "only node calls are supported in run statement"))
             (* traces *)
        )
        |  FBY_n (e1,e2) -> (
             let arg1 = treat_trace zeres env sstack e1 in
             let arg2 = treat_trace zeres env sstack e2 in
               CoTraceExp.of_fby arg1 arg2      
           )
             (* liste de priorit, rien de spcial ... *)
        |       PRIO_n elst -> (
                  let args = List.map (treat_trace zeres env sstack) elst in
                    CoTraceExp.of_prio args
                )       
                  (* idem pour parallle ... *)
        |       PARA_n elst -> (
                  let args = List.map (treat_trace zeres env sstack) elst in
                    CoTraceExp.of_para args
                )       
                  (* choix => il faut vrifier la controlabilit des poids *)
        |  CHOICE_n chargs -> (
             (* chargs : (val_exp * val_exp srcflaged option) list *)
             let doit (et, ewo) = (
               let tres = treat_trace zeres env sstack et in
               let wres = match ewo with
                   None -> None
                 |      Some ew -> (
                          let wval = treat_exp zeres env sstack ew.it in
                            if (CoAlgExp.is_controlable wval) then (
                              raise ( Compile_error (ew.src, "weights sould not be controllable"))
                            ) else Some wval
                        ) in
                 (tres, wres)
             ) in
               CoTraceExp.of_choice (List.map doit chargs)
           )
        |  LOOP_n (f,ee) -> (
             let arg = treat_trace zeres env sstack ee in
             match f with
             | Weak -> CoTraceExp.of_loop arg
             | Strong -> CoTraceExp.of_omega arg
           )
        |  LOOPI_n (emin,emax,ee) -> (
             let minres = treat_exp zeres env sstack emin in
             let maxres = treat_exp zeres env sstack emax in
             let eres = treat_trace zeres env sstack ee in
               if (CoAlgExp.is_controlable minres) then (
                 raise ( Compile_error (emin.src, "loop params sould not be controllable"))
               ) else if (CoAlgExp.is_controlable maxres) then (
                 raise ( Compile_error (emax.src, "loop params sould not be controllable"))
               ) else (
                 CoTraceExp.of_loopi minres maxres eres 
                   (ee.src, get_src_stack ee.src sstack (Some e))
               )
           )
        |  LOOPA_n (emoy,Some eect,ee) -> (
             let moyres = treat_exp zeres env sstack emoy in
             let ectres = treat_exp zeres env sstack eect in
             let eres = treat_trace zeres env sstack ee in
               if (CoAlgExp.is_controlable moyres) then (
                 raise ( Compile_error (emoy.src, "loop params sould not be controllable"))
               ) else if (CoAlgExp.is_controlable ectres) then (
                 raise ( Compile_error (eect.src, "loop params sould not be controllable"))
               ) else (
                 CoTraceExp.of_loopa  moyres (Some ectres) eres 
                   (ee.src, get_src_stack ee.src sstack (Some e))
               )
           )
        |  LOOPA_n (emoy,None,ee) -> (
             let moyres = treat_exp zeres env sstack emoy in
             let eres = treat_trace zeres env sstack ee in
               if (CoAlgExp.is_controlable moyres) then (
                 raise ( Compile_error (emoy.src, "loop params sould not be controllable"))
               ) else (
                 CoTraceExp.of_loopa  moyres None eres 
                   (ee.src, get_src_stack ee.src sstack (Some e))
               )
           )
        |  ASSERT_n (f,c,ee) -> (
             let cres = treat_exp zeres env sstack c in
             let eres = treat_trace zeres env sstack ee in
               match f with
					| Weak -> CoTraceExp.of_assert cres eres 
                   (ee.src, get_src_stack ee.src sstack (Some e))
					| Strong -> CoTraceExp.of_strong_assert cres eres 
                   (ee.src, get_src_stack ee.src sstack (Some e))
           )
        |       RAISE_n id -> (
                  (* le programme est correct, DONC tgtid existe !! *)
                  let tgtid = src_ref_to_target_id zeres env id in 
                    CoTraceExp.of_raise (CoIdent.to_string tgtid) 
                )
                  (* Nouvelles exceptions *)
        |  EXCEPT_n (exlst,ee) -> (
             (** Traitement des EXCEPTION_n :
                 Pour chaque identificateur de [[exlst]], on cre
                 un target ident unique qu'on insre dans la table
                 des exceptions, puis on appele recursivement
                 le traitement sur [[ee]]
             *)
             (* on cre les target ident associs *)
             let f = function ex -> (
               new_local_except zeres ex sstack
             ) in
               List.iter f exlst;
               (* on continu dans ce nouvel env ... *)
               treat_trace zeres env sstack ee
           )
             (* Dans la syntaxe intermdiaire, on n'a que 
                la construction catch
             *)
        |       TRY_n (e1, e2opt) -> (
                  let r1 = treat_trace zeres env sstack e1 in
                  let r2opt = match e2opt with
                      None -> None 
                    | Some e2 -> Some (treat_trace zeres env sstack e2)
                  in
                    CoTraceExp.of_try r1 r2opt
                )
        |       CATCH_n (id, e1, e2opt) -> (
                  (* le programme est correct, DONC tgtid existe !! *)
                  let tgtid = src_ref_to_target_id zeres env id in 
                  let tgtstr = (CoIdent.to_string tgtid) in
                  let r1 = treat_trace zeres env sstack e1 in
                  let r2opt = match e2opt with
                      None -> None 
                    | Some e2 -> Some (treat_trace zeres env sstack e2)
                  in
                    (* Pour simplifier : on INTERDIT
                       en interne le catch du "Deadlock" prdfini ->
                       on le remplace par la forme "try"
                    *)
                    if(tgtstr = LutPredef.kw_deadlock) then
                      CoTraceExp.of_try r1 r2opt
                    else
                      CoTraceExp.of_catch tgtstr r1 r2opt
                )
        |       TRAP_n (id, e1, e2opt) -> (
                  (* la seule difference avec le catch est la porte
                     de id : une fois cr le target unique, on peu
                     expanser en catch classique 
                  *)
                  (* on cre le target ident unique de id *)
                  new_local_except zeres id sstack;
                  (* le programme est correct, DONC tgtid existe !! *)
                  let tgtid = src_ref_to_target_id zeres env id in 
                  let tgtstr = (CoIdent.to_string tgtid) in
                  let r1 = treat_trace zeres env sstack e1 in
                  let r2opt = match e2opt with
                      None -> None 
                    | Some e2 -> Some (treat_trace zeres env sstack e2)
                  in
                    (* Pour simplifier : on INTERDIT
                       en interne le catch du "Deadlock" prdfini ->
                       on le remplace par la forme "try"
                    *)
                    if(tgtstr = LutPredef.kw_deadlock) then
                      CoTraceExp.of_try r1 r2opt
                    else
                      CoTraceExp.of_catch tgtstr r1 r2opt
                ) 

(*
   |  LOOPE_n (en,ee) -> (
        let nres = treat_exp zeres env sstack en in
        let eres = treat_trace zeres env sstack ee in
          if (CoAlgExp.is_controlable nres) then (
       raise (Compile_error (en.src, "loop params should not be controllable"))
          ) else (
       CoTraceExp.of_loope nres eres
          )
           )
*)
    )
(** 
2011 march
Separated from the exported "make" function in order to allow recursive calls.
The only global info from the package is the list of global except
N.B. => adding other global defs (e.g. data types) will require
to BETTER reorganized the infos !!
**)
and treat_node (env : CheckEnv.t) (ni : Syntaxe.node_info) (gxlist: Syntaxe.except_info list) = (
   (* create a fresh empty expanded program *)
   (* the main trace name is the name of the node *)
	let n = ni.ndi_ident.it in
   let mainid = (fixed_trace_id n) in
   let zeres = new_empty_t n mainid gxlist in

   (*********************************)
   (* EXCEPTIONS GLOBALES           *)
   (* => on garde leur nom tel quel *)
   (*********************************)
   let _ = List.iter (new_global_except zeres) gxlist in

   (*******************************)
   (* NOEUD PRINCIPAL             *)
   (*******************************)
   let nodelxm = ni.ndi_ident.src in
   let sstack = CoIdent.main_scope nodelxm in
   (*******************************)
   (* INPUTS ET OUTPUTS           *)
   (*******************************)
   (* pour les i/o, on a CoIdent.t = Syntaxe.ident *)
   (* value d'abord les init *)
   let inlist = List.map
           (eval_init_in_var_decl zeres env sstack)
           (ni.ndi_inputs)
   in
   let add_input = function (i, t, vopt, range_opt) -> (
           new_support_input zeres vopt i t sstack
   ) in
   List.iter add_input inlist;
   let outlist = List.map
           (eval_init_in_var_decl zeres env sstack)
           (ni.ndi_outputs)
   in
   let add_output = function (i, t, vopt, range_opt) -> (
           new_support_output zeres vopt range_opt i t sstack
   ) in
   List.iter add_output outlist;

   (***********************************)
   (* MAIN TRACE                      *)
   (***********************************)
   let maintrace = treat_trace 
           zeres
           env
           sstack
           ni.ndi_def 
   in
   (* on donne  la trace le nom du main node *)
   (new_trace zeres
           ni.ndi_ident
           mainid
           maintrace
           sstack);
   let pl = Hashtbl.fold
           ( fun id si acc -> match si.si_pre_ref_exp with
                           |       None -> acc
                           |       Some _ -> (id,si)::acc )
           zeres.stab []
   in
   (* Final result: need to reverse the lists ... *)
   zeres.prelist <- pl;
	zeres.alist <- List.rev zeres.alist;
	zeres.inlist <- List.rev zeres.inlist;
	zeres.outlist <- List.rev zeres.outlist;
	zeres.locoutlist <- List.rev zeres.locoutlist;
	zeres.locinlist <- List.rev zeres.locinlist;
	zeres
)


(**************************************************************
EXPANSION : procdure principale
---------------------------------------------------------------
Pour faire propre, il serait bon d'viter les variables globales !
Pour l'instant, on se contente  de les masquer ...
**************************************************************)

let node_name (x:t) = x.nme

(** accs aux rsultats *)

let support_pres (x:t) = x.prelist
let support_tab (x:t) = x.stab
let alias_tab (x:t) = x.atab
let alias_list (x:t) = x.alist
let input_list (x:t) = x.inlist
let output_list (x:t) = x.outlist
let local_in_list (x:t) = x.locinlist
let local_out_list (x:t) = x.locoutlist
let trace_tab (x:t) = x.ttab
let extern_tab (x:t) = x.etab
let main_trace (x:t) = x.mtrace

let ident_space (x:t) = x.idspace

let get_trace_info it id = (
	Util.hfind it.ttab id 
)

let make (env : CheckEnv.t) (p : Syntaxe.package) (n : string) = (
  let ni =
    try Syntaxe.pack_get_node p n
    with Not_found -> raise (
      Global_error (sprintf "node \"%s\" not found" n)
    )
  in
    treat_node env ni (Syntaxe.pack_except_list p)
)

