(*-----------------------------------------------------------------------
** Copyright (C) 2002 - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License
**-----------------------------------------------------------------------
**
** File: control.mli
** Author: jahier@imag.fr
*)

(** Defines a [state] ADT that is used for control purposes when
  executing [ima]. In particular, it is used to translate dynamically
  lengthed loops of Lutin. This ADT basically maps idents ([string]s)
  to [int]s.
*)

type state

val new_state : unit -> state
val print_state : state -> unit

val compose_state : state -> state -> state

val set : string -> int -> state -> state
(** [set id i st] sets [id] to [i] in [st]. *)

val set_between : string  -> int -> int -> int -> state -> state
(** [set id i min max st] sets [id] to [i] in [st], but allow values
  between [min] and [max] if no more transition can be made from the
  current state. *)

val draw_between : string -> int -> int -> state -> state
(** [loop_between id min max] sets [id] to a value uniformally drawn
  between [min] and [max] in [st]. *)

val draw_gauss : string -> float -> float -> state -> state
(** [loop_gauss id m std st] sets [id] to a vaue drawn according to a
  normal law of mean [m] and standard deviation [std] in [st]. *)

val dec : string -> state -> state
(** [dec id st] decrements [id] in [st]. *)

val return : string -> state -> int
(** [return id st] returns the binding of [id] in [st]. *)

(* val return_comp : string -> state -> int *)
(** [return id st] returns 1 if [id] is bound to 0 in [st], and
  returns 1 otherwise. *)


val is_recoverable : string -> state -> bool
(** [is_recoverable v st] iff [v] is recoverable, namely, if <<other
  values for the variable are possible according to the spec>>.

  For instance, if [v] has been drawn to 5 after a draw between 2 and
  9, it means that all values from 2 to 9 would have been legal
  too. Therefore, if:
  - a loop is garded by the value of v,
  - [v] is bound to 0 at the current step,
  - no satisfiable formula can be drawn in the end branch of the
    loop,

  then, since [v] could have been drawn to a bigger value (7, 8, or 9),
  it means that the body of the loop can be tried some more.

  Similarly, if:
  - [v] is not bound to 0 at the current step, but at least 2 steps
    have performed
  - the formula in the body of the loop is not satisfiable

  then it means that the end branch can be tried.

  In such cases, we say that variable [v] is recoverable.

  XXX Should variable resulting from a normal draw be always
  recoverable?  Mathematically, certainly, but in practice?
*)


type expr = state -> state
(** e.g., [loop_between foo 3 10] is a [expr].
  [(return "foo" (loop_between "foo" 3 10)] is a [post_expr].
*)

val compose_expr : expr -> expr -> expr

type test =
  | EqExpr of number * number
  | GtExpr of number * number
  | GeExpr of number * number
and number =
  | VarExpr of string
  | IntExpr of int


val ite : test -> expr -> expr -> state -> state
(**  [ite test e1 e2 st] evaluates [e1] if [test] is true, [e2] otherwise. *)

val state_size : state -> int
