(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License
**-----------------------------------------------------------------------
**
** File: command_line.ml
** Main author: jahier@imag.fr
*)



type optionsT = {
  mutable step_by_step : int option ;
  mutable display_local_var : bool ;
  mutable display_sim2chro : bool ;
(*   mutable cudd_heap_init : int ; *)
  mutable user_seed : int option ;
  mutable verb : int ;
  mutable show_step : bool ;
  mutable help : bool ;
  mutable output : string ;
  mutable draw_all_formula : bool;
  mutable draw_all_vertices : bool;
  mutable compute_volume : bool;
  mutable step_mode : Lucky.step_mode;
  mutable luciole_mode : bool;
  mutable oracle : bool ;
  mutable pp : string option ;
  mutable scade_gui : bool;
  mutable is_reactive : bool;
  mutable stdin : bool;
  mutable tmp_dir : string;
}

type cmd_line_optionT =
    Step | NoStep | Help
  | DisplayLocalVar | NoDisplayLocalVar
  | Sim2chro | NoSim2chro
  | AllVertices | AllFormula
(*   | CuddHeapInit  *)
  | Seed | Precision | NoOracle | Verbose | ShowStep | Output
  | ComputeVolume | StepInside | StepEdges | StepVertices | Stdin | PP | Reactive
  | ScadeGui | TmpDir

(* Names of the command line options to override the defaults. *)
let (string_to_option: (string * cmd_line_optionT) list) = [

  ("--help", Help);
  ("-h", Help);

  ("--verbose", Verbose);
  ("-v", Verbose);

  ("--no-oracle", NoOracle);

  ("--output", Output);
  ("-o", Output);

  ("--no-step", NoStep);
  ("--step", Step);
  ("-s", Step);

  ("--show-step", ShowStep);
  ("-ss", ShowStep);

  ("--with-seed", Seed);
  ("-seed", Seed);

  ("--precision", Precision);

  ("--call-sim2chro", Sim2chro);
  ("--do-not-call-sim2chro", NoSim2chro);
  ("-ns2c", NoSim2chro);

  ("--display-local-var", DisplayLocalVar);
  ("--do-not-display-local-var", NoDisplayLocalVar);
  ("-nlv", NoDisplayLocalVar);

  ("--pre-processor", PP);
  ("-pp", PP);

  ("--scade-gui", ScadeGui);
  
  ("--reactive", Reactive);

  ("--draw-all-vertices", AllVertices);
  ("--draw-all-formula", AllFormula);

  ("--fair", ComputeVolume);
  ("--compute-poly-volume", ComputeVolume);

  ("--tmp-dir", TmpDir);

  ("--step-inside", StepInside);
  ("--step-edges", StepEdges);
  ("--step-vertices", StepVertices);

  ("--stdin", Stdin);
  ("-stdin", Stdin);

(*   ("--init-cudd-heap", CuddHeapInit) *)
]

let (option_to_usage: cmd_line_optionT -> string) =
  fun opt ->
    match opt with
      | Step -> "Run lurette step by step.\n"
      | NoStep ->  "Do not run lurette step by step (Default).\n"
      | Output ->  "Output file name (the default file name is \"lurette.rif\").\n"
      | DisplayLocalVar -> "Display environment local variables in sim2chro (Default).\n"
      | NoDisplayLocalVar ->  "Do not display environment local variables in sim2chro.\n"
      | Sim2chro -> "Call sim2chro when lurette resumes (Default).\n"
      | NoSim2chro ->  "Do not call sim2chro when lurette resumes.\n"
(*       | CuddHeapInit -> "Set a magic number (related to the gc) that is used by mldd \n\t\t\tfor initializing Dd (Default is 20).\n" *)
      | Seed ->  "Set the value of the seed the random engine is initialized with (0 lets the system draw a seed).\n"
      | Precision ->  "Set the precision used for numerical values (number of digits).\n"
      | NoOracle -> "Do not run the oracle.\n "
      | Verbose -> "Set on the verbosity level (0, 1, 2).\n"
      | ShowStep -> "Set on a mode where the step number is printed on the standard output.\n"
      | Help -> "Print this message.\n"
      | AllFormula -> "Tries all formula.\n"
      | AllVertices -> "Tries all vertices.\n"
      | ComputeVolume -> (
	  "Compute the polyhedra volume before drawing: more fair, " ^
	  "but more expensive.")
      | StepInside -> "Step inside\n"
      | StepEdges -> "Step at edges\n"
      | StepVertices -> "Step at vertices\n"
      | PP -> "Pipe lucky file(s) through a preprocessor (e.g., cpp)"
      | TmpDir -> "Set the tmp dir"
      | ScadeGui -> "indicate that lurette is launched from the Scade GUI."
      | Reactive -> "Set on a reactive mode, in which, when no transition is possible, the previous values are returned.\n"
      | Stdin -> "Use apps that reads and writes RIF on stdin/stdout"

let (group_common_options: (string * cmd_line_optionT) list ->
       (string * cmd_line_optionT) list) =
  fun list ->
    List.fold_left
      (fun acc (str, opt) ->
         match acc with
             (str2, opt2)::tail ->
	       if
		 (opt = opt2)
	       then
		 (((str2  ^ "\n\t\t" ^ str), opt)::tail)
	       else
		 ((str, opt)::(str2, opt2)::tail)
	   | [] -> [(str, opt)]
      )
      []
      list
let usage_options =
  List.fold_left
    (fun acc (str, opt) ->
       acc ^ "\n\t\t" ^ str ^ "\n\t\t\t" ^ (option_to_usage opt))
    ""
    (List.rev (group_common_options string_to_option))

let usage =
  ("usage: ./lurette length p k1 k2 k3 [options]* (<file>.luc)+ \n" ^
   "   where: \n\t o `length' is the number of steps that will be run " ^

   "\n\t o `p' is the number of solutions that are drawn in each formula" ^
   "\n\t o `k1' is the number of solutions that are drawn inside" ^
   "\n\t o `k2' is the number of solutions that are drawn at edges" ^
   "\n\t o `k3' is the number of solutions that are drawn at vertices" ^


   "\n\t o `options' is a list of options. The available options are: " ^
   usage_options ^ "\n\n" ^

   "Example:    ./lurette 1000 3 4 env1.luc env2.luc env3.luc env4.luc\n\n ")


let cmd_line_string_to_int str errmsg =
  Util.my_int_of_string str
  (* Avoid crashing if int are too big *)
