(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License 
**-----------------------------------------------------------------------
**
** File: bddd.ml
** Main author: jahier@imag.fr
*)


(** In the following, we call a comb the bdd of a conjunction of
 litterals (var). They provide the ordering in which litterals
 appear in the bdds we manipulate.
*)

open Util
open Exp
open Value
open Constraint
open Store


type var_idx = int

(****************************************************************************)

open Sol_nb

(** snt Associates to a bdd the (absolute) number of solutions
  contained in its lhs (then) and rhs (else) branches. Note that
  adding those two numbers only give a number of solution that is
  relative to which bdd points to it. 
*)
type snt = (Bdd.t, sol_nb * sol_nb) Hashtbl.t

let snt_ref = ref (Hashtbl.create 1)
let snt_build = ref false

let (sol_number_snt : snt -> Bdd.t -> sol_nb * sol_nb) =
  fun snt bdd ->
    hfindl "bddd.sol_number_snt: " snt bdd 

let (sol_number : Bdd.t -> sol_nb * sol_nb) =
  (sol_number_snt !snt_ref) 
  

let (add_snt_entry : Bdd.t -> sol_nb * sol_nb -> unit) =
  fun bdd sol_nb -> 
      Hashtbl.replace !snt_ref bdd sol_nb

let (init_snt : unit -> unit) =
  fun _ -> 
    let bdd_true = Bdd.dtrue !Formula_to_bdd.bdd_manager in
    let bdd_false = Bdd.dfalse !Formula_to_bdd.bdd_manager in
      (Hashtbl.add !snt_ref (bdd_true) (one_sol, zero_sol));
      (Hashtbl.add !snt_ref (bdd_false) (zero_sol, one_sol))

let (clear_snt: unit -> unit) =
  fun _ -> 
    (* if (t mod 100) = 0      
       then  
       if Util.hashtbl_size snt > 1000  
       then  *)
    Hashtbl.clear !snt_ref;
    Polyhedron.clean_internal_tbl ();
    Formula_to_bdd.clear_all();
    snt_build := false;
    init_snt ()

let (sol_number_exists : Bdd.t -> snt -> bool) =
  fun bdd snt -> 
      Hashtbl.mem snt bdd


let rec (build_sol_nb_table_rec: Bdd.t -> Bdd.t -> snt -> sol_nb) =
  fun bdd comb snt -> 
    (** Returns the relative (to which bbd points to it) number of
        solutions of [bdd]. Also udpates the solution number table 
        for [bdd], and recursively for all its sub-bdds.

        [comb] is a positive cube that ougth to contain the indexes of
        boolean vars that are still to be generated, and the numerical
        indexes that appears in [bdd].  

        Note that this sol number make abstraction of the volume of
        polyhedron which dimension is greater than 2. It is very bad as
        far as the fairness of the draw is concerned, but really,
        computing it would be far too expensive for our purpose
        (real-time!). Cf fair_bddd.ml for a more fair version.
    *)
    let _ = assert (
      not (Bdd.is_cst bdd) && (Bdd.topvar comb) = (Bdd.topvar bdd) )
    in
    let sol_nb =
      try
	     (* either it has already been computed ... *)
	     let (nt, ne) = sol_number_snt snt bdd in
	       (add_sol_nb nt ne)

      with Not_found ->
	     (* ... or not. *)
	     let nt = compute_absolute_sol_nb (Bdd.dthen bdd) comb snt in
	     let ne = compute_absolute_sol_nb (Bdd.delse bdd) comb snt in
	       Hashtbl.replace snt bdd (nt, ne);
	       (add_sol_nb nt ne)
    in
	     sol_nb
and 
    (compute_absolute_sol_nb: Bdd.t  -> Bdd.t -> snt -> sol_nb) =
  fun sub_bdd comb snt -> 
    (*
      Returns the absolute number of solutions of [sub_bdd]  w.r.t. [comb], 
      where [comb] is the comb of the father of [sub_bdd].
      
      The [comb] is used to know which output Boolean variables are
      unconstraint along a path in the bdd. Indeed, the comb is made
      of all the Boolean output var indexes plus the num contraints
      indexes that appears in the bdd; hence, if the topvar of the
      bdd is different from the topvar of the comb, it means that
      the topvar of the comb is unsconstraint and we need to
      multiply the number of solution of the branch by 2.
    *)
    if 
      Bdd.is_false sub_bdd 
    then
      zero_sol
    else if 
      Bdd.is_true sub_bdd 
    then
      let sol_nb = 
	     if 
	       Bdd.is_true comb
	     then
	       one_sol
	     else
	       two_power_of (Bdd.supportsize (Bdd.dthen comb)) 
      in
	     sol_nb
    else 
      let topvar = Bdd.topvar sub_bdd in
      let rec
	       (count_missing_vars: Bdd.t -> var_idx -> int -> Bdd.t * int) =
	     fun comb var_idx cpt -> 
	       (* Returns [cpt] + the number of variables occurring in [comb]
	          before reaching [var_idx] ([var_idx] excluded). Also returns the comb
	          which topvar is [var_idx]. *)

	       let _ = assert (not (Bdd.is_cst comb)) in
	       let combvar = Bdd.topvar comb in
	         if var_idx = combvar
	         then (comb, cpt)
	         else count_missing_vars (Bdd.dthen comb) var_idx (cpt+1)
      in
      let (sub_comb, missing_vars_nb) = 
	     count_missing_vars (Bdd.dthen comb) topvar 0
      in
      let n0 = build_sol_nb_table_rec sub_bdd sub_comb snt in
      let factor = (two_power_of missing_vars_nb) in
      let res = mult_sol_nb n0 factor in
	     res



(** [build_sol_nb_table bdd comb] build an internal table that
  associates to each [bdd] its solution number. [comb] is a bdd made
  of the the conjunctions of variables of [bdd] union the Boolean
  variable to be generated (which migth not appear in [bdd]) ; it is 
  necessary because not all the variable to be generated appears
  in the [bdd].
*)	
	

let (build_sol_nb_table: var list -> Bdd.t -> Bdd.t -> unit) =
  fun _ bdd comb ->
    snt_build := true;
    if
      (sol_number_exists bdd !snt_ref)
    then
      ()
    else
      let rec skip_unconstraint_bool_var_at_top comb v =
	(* [build_sol_nb_table] supposes that the bdd and its comb have
	   the same top var.  *)
	     if Bdd.is_true comb then comb
	     else
	       let topvar = (Bdd.topvar comb) in
	       if v = topvar then comb
	       else skip_unconstraint_bool_var_at_top (Bdd.dthen comb) v
      in
	   if Bdd.is_true bdd then
	     let _ = build_sol_nb_table_rec bdd comb (!snt_ref) in
	     ()
	   else
	     let tvar = Bdd.topvar bdd in
	     let comb2 = skip_unconstraint_bool_var_at_top comb tvar in
	     let _ = build_sol_nb_table_rec bdd comb2 (!snt_ref) in
	     ()

(****************************************************************************)
(****************************************************************************)

(* 
   This table is used to associate an unique integer to a bdd, which
   is useful for debugging. That table is filled in Solver.print_bdd_with_dot,
   which use it. 
*)
let (bdd_to_int : (Bdd.t, int) Hashtbl.t)= Hashtbl.create 100 

(****************************************************************************)
(****************************************************************************)

let (toss_up_one_var: Var.name -> Var.subst) =
  fun var -> 
   (* *)
    let ran = Random.float 1. in
      if (ran < 0.5) then (var, B(true)) else (var, B(false))


let (toss_up_one_var_index: Var.env_in -> Var.env -> int -> string -> var_idx -> 
       Var.subst option) =
  fun input memory vl ctx_msg var_idx -> 
    (* if [var_idx] is a index that corresponds to a boolean variable,
       this fonction performs a toss and returns a substitution for
       the corresponding boolean variable. It returns [None]
       otherwise (ie if it is a num).

       Indeed, if it happens that a numerical constraint does not
       appear along a path, we simply ignore it and hence it will not
       be added to the store.
    *)
    let cstr = Formula_to_bdd.index_to_linear_constraint var_idx in
      match cstr with
          Bv(v) -> (
	    match (Var.default v) with
	      | Some (Formu f) ->
		  let bdd = Formula_to_bdd.f input memory ctx_msg vl f in
(*	    print_string ("\n give a default to " ^ (Var.name v) ^ "\n"); flush stdout; *)
		    if Bdd.is_false bdd then Some((Var.name v), B false)
		    else if Bdd.is_true bdd then Some((Var.name v), B true)
		    else
		      (
			print_string (
			  "*** Default values should not depend on " ^ 
			  "controllable variables, \nwhich is the case of " ^
			  (formula_to_string f) ^ ", the default value of " ^ 
			  (Var.name v) ^ "\n");
			exit 2
		      )
	      | Some (Numer _) -> assert false
	      | Some (Liste l) -> assert false
	      | None -> Some(toss_up_one_var (Var.name v))
	  )
	| _  -> None


let rec (draw_in_bdd_rec: Var.env_in -> Var.env -> int -> string -> Var.env * Store.t
	       -> Bdd.t -> Bdd.t -> Var.env * Store.t' * Store.p) = 
  fun input memory vl ctx_msg (sl, store) bdd comb ->
    (** Returns [sl] appended to a draw of all the boolean variables
        bigger than the topvar of [bdd] according to the ordering
        induced by the comb [comb]. Also returns the (non empty) store
        obtained by adding to [store] all the numeric constraints that
        were encountered during this draw.

        Raises the [No_numeric_solution] exception whenever no valid
        path in [bdd] leads to a satisfiable set of numeric
        constraints.  
    *)

    let snt = !snt_ref in
      if 
        Bdd.is_true bdd
      then
        (* Toss the remaining bool vars. *)
        let (store_int, store_poly) = 
	       Store.switch_to_polyhedron_representation vl store 
        in
        let vars_index = Bdd.list_of_support comb in
	       ( 
	         (* (List.append sl *)
	         (Value.OfIdent.add_list sl
	            (Util.list_map_option 
		            (toss_up_one_var_index input memory vl ctx_msg)
		            vars_index) 
	         ),
	         store_int,
	         store_poly
	       )
      else
        let _ = assert (not (Bdd.is_false bdd)) in
        let _ = assert (sol_number_exists bdd snt) in
        let bddvar = Bdd.topvar bdd in
        let cstr = (Formula_to_bdd.index_to_linear_constraint bddvar) in 
	       match cstr with
	         | Bv(var) ->
	             draw_in_bdd_rec_bool input memory vl ctx_msg var (sl, store) bdd comb snt
	         | EqZ(e) ->
	             draw_in_bdd_rec_eq input memory vl ctx_msg e (sl, store) bdd comb snt
	         | Ineq(ineq) ->
	             draw_in_bdd_rec_ineq input memory vl ctx_msg ineq (sl, store) bdd comb snt


and (draw_in_bdd_rec_bool: Var.env_in -> Var.env -> int -> string -> var -> 
      Var.env * Store.t -> Bdd.t -> Bdd.t -> snt -> 
	   Var.env * Store.t' * Store.p) = 
  fun input memory vl ctx_msg var (sl, store) bdd comb snt ->
    let bddvar = Bdd.topvar bdd in
    let topvar_comb  = Bdd.topvar comb in
      
      if
	     bddvar <> topvar_comb 
      then
	     (* that condition means that topvar_comb is an unconstraint
	        boolean var; hence we toss it up, or we compute its default
	        value (cf "~default" flag). *)
	     let new_sl = 
	       match toss_up_one_var_index input memory vl ctx_msg topvar_comb with
	           Some(s) -> (*s::sl*) Value.OfIdent.add sl s
	         | None -> sl
	     in
	       draw_in_bdd_rec input memory vl ctx_msg (new_sl, store) bdd (Bdd.dthen comb)
      else 
	     (* bddvar = combvar *) 
	     let (n, m) = sol_number_snt snt bdd in
	     let _ =
	       if ((eq_sol_nb n zero_sol) && (eq_sol_nb m zero_sol))
	       then (
	         if vl > 3 then (
	           print_string "BDD backtrack...\n";
	           flush stdout
	         );
	         raise No_numeric_solution 
	       )
	     in
	     let (
	       bdd1, bool1, sol_nb1,
	       bdd2, bool2, sol_nb2
	     ) =
	       let ran = Random.float 1. 
	       and sol_nb_ratio = float_of_sol_nb (div_sol_nb n  (add_sol_nb n m))
	       in
	         assert (not (Bdd.is_cst bdd));
	         if 
	           ran < sol_nb_ratio
		          (* 
		             Depending on the result of a toss (based on the number
		             of solution in each branch), we try the [then] or the
		             [else] branch first.  
		          *)
	         then
	           ((Bdd.dthen bdd), true, n,
	            (Bdd.delse bdd), false, m )
	         else 
	           ((Bdd.delse bdd), false, m,
	            (Bdd.dthen bdd), true, n )
	     in
	     let (sl1, sl2, new_comb) = (
	       (* (((Var.name var), B(bool1))::sl),  *)
	       (Value.OfIdent.add sl (Var.name var, B(bool1))), 
	       (* (((Var.name var), B(bool2))::sl), *)
	       (Value.OfIdent.add sl (Var.name var, B(bool2))), 
	       (if Bdd.is_true comb then comb else Bdd.dthen comb) 
	     )	    
	     in
	       (* 
	          A solution will be found in this branch iff there exists
	          at least one path in the bdd that leads to a satisfiable
	          set of numeric constraints. If it is not the case,
	          [res_opt] is bound to [None]. 
	       *)
	       try 
            if not (eq_sol_nb sol_nb1 zero_sol)
	         then (
              assert (not (Bdd.is_false bdd1)) ;
              draw_in_bdd_rec input memory vl ctx_msg (sl1, store) bdd1 new_comb)
            else raise No_numeric_solution
	       with
	           No_numeric_solution ->
		          if not (eq_sol_nb sol_nb2 zero_sol)
		          then (
                  assert (not (Bdd.is_false bdd1)) ;
                  draw_in_bdd_rec input memory vl ctx_msg (sl2, store) bdd2 new_comb)
		            (* 
		               The second branch is now tried because no path in
		               the first bdd leaded to a satisfiable set of
		               numeric constraints. 
		            *) 
		          else (	    
		            if vl > 3 then (
		              print_string "BDD backtrack...\n";
		              flush stdout
		            );
		            raise No_numeric_solution
		          )
		      | e ->
		          print_string ((Printexc.to_string e) ^ "\n");
		          flush stdout;
		          assert false
		            
and (draw_in_bdd_rec_ineq: Var.env_in -> Var.env -> int -> string -> Constraint.ineq -> 
      Var.env * Store.t -> Bdd.t -> 
      Bdd.t -> snt -> Var.env * Store.t' * Store.p) = 
  fun input memory vl ctx_msg cstr (sl, store) bdd comb snt ->
    (* 
       When we add to the store an inequality constraint GZ(ne) or
       GeqZ(ne) ([split_store]), 2 stores are returned. The first is a
       store where the inequality has been added; the second is a
       store where its negation has been added.
       
       Whether we choose the first one or the second depends on a toss
       made according the (boolean) solution number in both branches
       of the bdd. If no solution is found into that branch, the other
       branch with the other store is tried.
    *)
    let (n, m) = sol_number_snt snt bdd in
    let _ =
      if ((eq_sol_nb n zero_sol) && (eq_sol_nb m zero_sol))
      then (
	     if vl > 3 then (
	       print_string "BDD backtrack...\n";
	       flush stdout
	     );
	     raise No_numeric_solution 
      )
    in 
    let ran = Random.float 1. in
    let cstr_neg = Constraint.neg_ineq cstr in
    let (cstr1, bdd1, sol_nb1, cstr2, bdd2, sol_nb2) =
      if 
	     ran < (float_of_sol_nb (div_sol_nb n (add_sol_nb n m)))
      then
	     (cstr, (Bdd.dthen bdd), n, cstr_neg, (Bdd.delse bdd), m)
      else 
	     (cstr_neg, (Bdd.delse bdd), m, cstr, (Bdd.dthen bdd), n)
    in
      (*     let bddi = try hfindl "bddd: " bdd_to_int bdd with _ -> -1 *)
      (*     and bdd1i = try hfindl "bddd: " bdd_to_int bdd1 with _ -> -1 *)
      (*     and bdd2i = try hfindl "bddd: " bdd_to_int bdd2 with _ -> -1 in *)

    let store1 = add_constraint store (Ineq cstr1) in
      (* A solution will be found in this branch iff there exists
	      at least one path in the bdd that leads to a satisfiable
	      set of numeric constraints. If it is not the case,
	      [res_opt] is bound to [None]. *)
      
      try
	     if not (is_store_satisfiable vl store1) then 
	       raise No_numeric_solution;
	     draw_in_bdd_rec input memory vl ctx_msg (sl, store1) bdd1 comb
      with 
	       No_numeric_solution -> 
	         let store2 = add_constraint store (Ineq cstr2) in
	           (* 
		           The second branch is tried if no path in the first
		           bdd leaded to a satisfiable set of numeric
		           constraints.  
	           *)
	           if 
		          (not (eq_sol_nb sol_nb2 zero_sol)) 
		          && is_store_satisfiable vl store2
	           then
		          draw_in_bdd_rec input memory vl ctx_msg (sl, store2) bdd2 comb
	           else (
		          if vl > 3 then (
		            print_string "BDD backtrack...\n";
		            flush stdout
		          );
		          raise No_numeric_solution
	           )
	     | e ->
	         print_string ((Printexc.to_string e) ^ "\n");
	         flush stdout;
	         assert false

and (draw_in_bdd_rec_eq: Var.env_in -> Var.env -> int -> string -> 
      Ne.t -> Var.env * Store.t ->
      Bdd.t -> Bdd.t -> snt -> Var.env * Store.t' * Store.p) = 
  fun input memory vl ctx_msg ne (sl, store) bdd comb snt ->
    (*
      We consider 3 stores: 
      - store + [ne = 0]
      - store + [ne > 0]
      - store + [ne < 0]
      
      Whether we choose the first one or not depends on toss made
      according the (boolean) solution number in both branches of the
      bdd. If the else branch is choosen (if it is chosen in the
      first place, or if backtracking occurs because no solution is
      found in the then branch) whether we try the second or the
      third store first is (fairly) tossed up.
    *)
    let (n, m) = sol_number_snt snt bdd in
      (*     let bddi = try hfindl "bddd: " bdd_to_int bdd  with _ -> -1 in  *)
    let _ =
      if ((eq_sol_nb n zero_sol) && (eq_sol_nb m zero_sol))
      then (
	     if vl > 3 then (
	       print_string "BDD backtrack...\n";
	       flush stdout
	     );
	     raise No_numeric_solution 
      )
    in
    let ran0 = Random.float 1. 
    and ran  = Random.float 1. 
    and cstr = (EqZ ne)
    and not_cstr = (Ineq (GZ ne))
    and not_cstr2 = (Ineq (GZ (Ne.neg_nexpr ne)))
    in
      
    let sol_nb_ratio = float_of_sol_nb (div_sol_nb  n (add_sol_nb n m)) in
    let (
      cstr1, bdd1, sol_nb1, 
      cstr2, bdd2, sol_nb2,
      cstr3, bdd3, sol_nb3) =
      if 
	     ran0 < 0.5
	       (* 
	          When taking the negation of an equality, we can
	          either try > or <. Here, We toss which one we
	          will try first.  
	       *)
      then
	     if 
	       ran < sol_nb_ratio
	         (*
	           Depending on the result of a toss (based on the number
	           of solution in each branch), we try the [then] or the
	           [else] branch first.  
	         *)
	     then
	       (cstr,      (Bdd.dthen bdd), n,
	        not_cstr,  (Bdd.delse bdd), m,
	        not_cstr2, (Bdd.delse bdd), m)
	     else 
	       (not_cstr,  (Bdd.delse bdd), m,
	        cstr,      (Bdd.dthen bdd), n,
	        not_cstr2, (Bdd.delse bdd), m)
      else
	     if
	       ran < sol_nb_ratio
	         (* Ditto *)
	     then
	       (cstr,      (Bdd.dthen bdd), n,
	        not_cstr2, (Bdd.delse bdd), m,
	        not_cstr,  (Bdd.delse bdd), m)
	     else 
	       (not_cstr2, (Bdd.delse bdd), m,
	        cstr,      (Bdd.dthen bdd), n,
	        not_cstr,  (Bdd.delse bdd), m)
    in    
      (*     let bddi1 = try hfindl "bddd: " bdd_to_int bdd1 with _ -> -1 in  *)
      (*     let bddi2 = try hfindl "bddd: " bdd_to_int bdd2 with _ -> -1  in  *)
      (*     let bddi3 = try hfindl "bddd: " bdd_to_int bdd3 with _ -> -1  in  *)
    let store1 = add_constraint store cstr1 in 
      (* 
	      A solution will be found in this branch iff there exists
	      at least one path in the bdd that leads to a satisfiable
	      set of numeric constraints. 
      *)
      try
	     (
	       if 
	         (not (is_store_satisfiable vl store1)) || (sol_nb1 = zero_sol)
	       then 
	         raise No_numeric_solution
	     );
	     draw_in_bdd_rec input memory vl ctx_msg (sl, store1) bdd1 comb
      with 
	       No_numeric_solution -> 
	         (* 
	            The second branch is tried if no path in the first bdd
	            leaded to a satisfiable set of numeric constraints. 
	         *)
	         let store2 = add_constraint store cstr2 in
	           try
		          if 
		            (not (is_store_satisfiable vl store2)) || (sol_nb2 = zero_sol) 
		          then 
		            raise No_numeric_solution;
		          draw_in_bdd_rec input memory vl ctx_msg (sl, store2) bdd2 comb
	           with 
		            No_numeric_solution -> 
		              let store3 = add_constraint store cstr3 in
		                if 
			               (not (is_store_satisfiable vl store3)) || (sol_nb3 = zero_sol) 
		                then (
			               if vl > 3 then (
			                 print_string "BDD backtrack...\n";
			                 flush stdout
			               );
			               raise No_numeric_solution
		                );
		                draw_in_bdd_rec input memory vl ctx_msg (sl, store3) bdd3 comb
		          | e -> 
		              print_string ((Printexc.to_string e) ^ "\n");
		              flush stdout;
		              assert false


(* exported *)
let rec (draw_in_bdd: 
           Var.env_in -> Var.env -> int -> string -> var list -> Bdd.t ->
          Bdd.t -> Var.env * Store.t' * Store.p) = 
  fun input memory vl ctx_msg num_vars_to_gen bdd comb ->
    build_sol_nb_table num_vars_to_gen bdd comb;
    draw_in_bdd_rec input memory vl ctx_msg (Value.OfIdent.empty, Store.create num_vars_to_gen) bdd comb



(****************************************************************************)

(*
type sol_nb = float
let add_sol_nb =  (+.)   
let mult_sol_nb n m = n *. m   
let zero_sol = 0.   
let one_sol = 1.   
let eq_sol_nb = (=)   

let two_power_of n = 
  let res =  2. ** (float_of_int n) in
  let _ = assert (res <> infinity) in
    res

let float_of_sol_nb sol = sol   
let string_of_sol_nb = string_of_float   
*)

