(** 

ON THE FLY generation of automata

For full automata gen. see AutoGen

*)

open LutErrors;;
open Printf;;
open CoIdent ;;
open CoAlgExp ;;
open CoTraceExp ;;
open LutPredef ;;
open Expand ;;


let dbg = Verbose.get_flag "AutoExplore"

(** N.B. On utilise des AlgExp.t de type "CkTypeEff.weight" pour
reprsenter les poids dans les transitions *)

(* tree-like transitions:
- leaves:
	"Vanish" for normal termination
	"Raise exception" for abnormal termination
	"Goto" for an actual transition
- n-ary node with weights
*)

(** WEIGHTS *)
type weightexp = 
|  W_huge 
|  W_exp of CoAlgExp.t

let opt_weight_of_opt_algexp = function
|  None -> None
|  Some e -> Some (
   if (e = CoAlgExp.of_huge_weight) then W_huge
   else W_exp e
)

let huge_weight = W_huge

(** EXCEPTIONS *)
type raise_desc = string

(** GOTO *)
type stable_state = CoTraceExp.t

(* type cond = CoAlgExp.t list *)
type cond = Guard.t

type goto = cond * stable_state

(** TRANSITION TREE *)
type ttree =
   Vanish
|  Raise of raise_desc
|  Goto of goto
|  Split of (ttree * weightexp option * cond) list

let ttree_stats = function
| None -> "empty ttree"
| Some t -> (
	let rec f t = (
		match t with
		| Split l -> (
			let g (nn0, nf0) (t,_,_) = (
				let (nn, nf) = f t in
				(nn0+nn,nf0+nf)
			) in
			List.fold_left g (1,0) l
		)
		| _ -> (0, 1)
	) in
	let (nn, nf) = f t in
	Printf.sprintf "nodes:%d, leaves:%d" nn nf
) 


(* DATA CONTEXT managment (for simulation) *)
type config = { data: Guard.store option; control: string }

let make_config dataopt ctrl =
	let st = match dataopt with
		|	None -> None
		|	Some (i,p) -> Some { Guard.curs = i; Guard.pres = p} 
	in { data = st; control = ctrl }

(** STATE MANAGMENT *)
type state_info =
   SS_stable of CoTraceExp.t
|  SS_transient
|  SS_final of string

(* POINT TO POINT TRANSITION:
   intermediate representation
   for full automaton generation *)
type trans = {
   src: string;
   wgt: weightexp option; 
   form: cond;
   dest: string; 
}

(* THE MAIN TYPE
	- (control) states are CoTraceExp.t
	- (control) states are hashed, and labelled by a unique string
	- a config is a variable config + a (control) state
*)
type t = {
   source_code : Expand.t;
   mutable nb_stables : int;
   mutable nb_transients : int;
   mutable init_control : string;
   mutable final_control : string;
   states : (string, state_info) Hashtbl.t ;
   mutable transitions : trans list;

   (* Gestion des puits *)
   mutable nb_sinks : int;
   _state2trace : (string, CoTraceExp.t) Hashtbl.t ;
   _trace2state : (CoTraceExp.t, string) Hashtbl.t;
   _config2ttree : (config, ttree) Hashtbl.t;

   (* liste des control inexplors *)
   mutable todo : string list;

   (* mode global/dynamique *)   
}

(* Misc infos *)
let source (it:t) = it.source_code
let states (it:t) = it.states
let init_control (it:t) = it.init_control
let transitions (it:t) = it.transitions

(* ADD/REPLACE pre values in data strore *)
let add_pres (unalias:Guard.unalias) (data:Guard.store option) (pctx:CoTraceExp.escope) = (
	match pctx with
	| [] -> data
	|	_ -> (
		let ctx = match data with
		|	Some d -> d
		|	None -> Guard.empty_store
		in
		let addp accin (id, eo) = (
			match eo with
			| None -> accin
			| Some e -> (
				let v = try (
					Guard.value_of_algexp unalias ctx e 
				) with Guard.Not_constant e -> raise (Internal_error ("AutoExplore.add_pres",
            	("initial value of \""^id^"\" must be a constant expression")))
				in
				Value.OfIdent.add accin (id, v)
			)
		) in
		let new_pres = List.fold_left addp ctx.Guard.pres pctx in
		Some {Guard.curs=ctx.Guard.curs; Guard.pres=new_pres}
	)
) 

let get_cpt i = CoIdent.of_cpt i

let dynamic_weight_exp (f : string) (c : int) (args : CoAlgExp.t list) = (
   let pcpt = CoAlgExp.of_pre (get_cpt c) CkTypeEff.integer in
   CoAlgExp.of_call (CoIdent.from_string f) CkTypeEff.integer (pcpt::args) 
)

let incr_loop_cpt (c : int) = (
   let nme = get_cpt c in
   let precpt = CoAlgExp.of_pre nme CkTypeEff.integer in
   let cpt = CoAlgExp.of_support nme CkTypeEff.integer true in
   CoAlgExp.of_call (CoIdent.from_string LutPredef.kw_eq) CkTypeEff.boolean [
      cpt ;
      CoAlgExp.of_call (CoIdent.from_string LutPredef.kw_plus) CkTypeEff.integer [
         precpt;
         CoAlgExp.of_iconst "1"
      ]
   ]
)
let reset_loop_cpt (c : int) = (
   let nme = get_cpt c in
   let cpt = CoAlgExp.of_support nme CkTypeEff.integer true in
   CoAlgExp.of_call (CoIdent.from_string LutPredef.kw_eq) CkTypeEff.boolean [
      cpt ;
      CoAlgExp.of_iconst "0"
   ]
)

(** dump des transition branchues pour debug *)
let dump_weight = ( function
|  Some W_huge    -> printf "huge"
|  Some (W_exp w) -> CoAlgExp.dump w
|  None           -> printf "1"
)

(* obsolete
let (algexp_of_weight: weightexp -> CoAlgExp.t) = (
   function w -> w
)
*)

let dump_ttree topt = (
   let rec rdump t pfx = (
      match t with
      Vanish -> (
         printf "%sVANISH\n" pfx;
      ) |
      Raise e -> (
         printf "%sRAISE %s\n" pfx e;
      ) |
      Goto (c, n) -> (
         printf "%sGOTO:[\n" pfx;
         printf "%s   cond:(" pfx;
         (* Utils.iter_sep (CoAlgExp.dump) (function _ -> printf " & ") c ; *)
			Guard.dumpf stdout c;
         printf " )\n%s   next: <" pfx;
         CoTraceExp.dump n;
         printf ">\n%s]\n" pfx
      ) |
      Split twl -> (
         printf "%sSPLIT:\n" pfx;
         let recpfx = sprintf "%s   " pfx in
         let dchoice (t,wo,cl) = (
            printf "%s[\n" pfx;
            (match wo with
               Some w -> (
                  printf "%sWEIGHT: " recpfx;
                  dump_weight wo ; 
                  printf "\n"
               ) | None -> ()
            );
				printf "%sCOND: (" recpfx;
				Guard.dumpf stdout cl;
				printf ")\n";
            rdump t recpfx ;
            printf "%s]\n" pfx
         )  in
         List.iter dchoice twl
      )
   ) in
   (
   match topt with
      None -> printf "%s" LutPredef.kw_deadlock
   |  Some t -> rdump t ""
   );
   printf "\n" 
)



(****************************************************************************
TRACE -> TRANSITIONS BRANCHUES
-----------------------------------------------------------------------------
C'est essentiellement une fonction :
   CkTypeExp.t -> ttree
Le traitement des "feuilles" est ralis par des fonctions
passes en paramtre (callback) :
- terminaisons normales (vanish)
- terminaisons anormales (raise)
- une pour traiter les continuations (goto)

N.B. ces deux fonctions peuvent ne pas rendre de rsultat en cas
de ``DeadLock'' statique.
****************************************************************************)

(** utilitaire : mise en squence tenant compte du TE_eps *)
let put_in_seq te1 te2 = (
   if(te1 = TE_eps) then te2
   else if(te2 = TE_eps) then te1
   else (TE_fby (te1,te2))
)
(** utilitaire : mise en parallle *)
let put_in_para te1 te2 = (
   match (te1,te2) with
   | (TE_eps, x) -> x
   | (x, TE_eps) -> x
   | (x, TE_para yl) -> TE_para (x::yl)
   | (x,y) -> TE_para [x; y]
)


(* For old algo with several callbacks *)
(** type de la fonction de traitement des vanish *)
type goto_callback = goto -> ttree option

(** type de la fonction de traitement des raise *)
type raise_callback = raise_desc -> ttree option

(** type de la fonction de traitement des goto *)
type vanish_callback = unit -> ttree option

(* For NEW algo with a UNIQUE callback *)
type callback = ttree option -> ttree option 

(** CoTraceExp.t -> ttree *)

(* MAIN (RECURSIVE) TRAVERSAL OF A LUTIN STATE
	New algo -> 2009 semantics, with:
	- inherited acc-umulator for constraints
	- all the rest (mainly control) performed by cont-inuation

	N.B. cases that impact acc-umulator are:
		- TE_constraint
		- TE_para and also TE_assert (special case)

   with partial evaluation of contraints
*)

let gentrans 
    (xenv : Expand.t)
	 (data : Guard.store option)  (* data env = inputs + pres *)
    (x : CoTraceExp.t)     (* control = lutin trace *)
    = (
      (*-------------------------------------------*)
      (* Correspondance id de trace -> trace exp
         N.B. on traque les rcursions ?  *)
      (*-------------------------------------------*)
      let id2trace s = (
        (Util.hfind (Expand.trace_tab xenv) s).ti_def_exp
      ) in
	   let unalias s = (
        (Util.hfind (Expand.alias_tab xenv) s).ai_def_exp
	   ) in
        
      (*-------------------------------------------*)
      (* LA FONCTION RCURSIVE *)
      (*-------------------------------------------*)
      let rec rec_gentrans
		    (data : Guard.store option)  (* data env = inputs + pres *)
          (x : CoTraceExp.t)
          (acc : cond)
          (cont : callback)
          = (
            Verbose.exe ~level:13 
              (fun () -> 
                 Printf.printf
                   "rec_gentrans \"%s\"\n" (CoTraceExp.dumps x));
            match x with
                (***** EPSILON => vanish ... *****)
              | TE_eps -> (
                  cont (Some Vanish)
                )
                  (***** TRACE NOMME => ON CHERCHE SA DEF ******) 
              | TE_ref s -> (
                  rec_gentrans data (id2trace s) acc cont
                )
                  (***** CONTRAINTE => on appelle le callback *)
              | TE_constraint (ae,si) -> (
			         (* HERE DO EVAL ! *)
			         try (
				        let new_acc = Guard.add ~unalias:unalias ~context:data ae acc si in

				          (* ICI *)
				          (* cont (Some (Goto (Guard.of_exp ~unalias:unalias ~context:data ae, TE_eps))) *)

				          cont (Some (Goto (new_acc, TE_eps)))
			         ) with Guard.Unsat -> (
      		        Verbose.put ~flag:dbg " !!!!! CUT A BRANCH\n";
				        (* HERE: cont or not cont ???? *)
				        cont None
			         )
                )
                  (***** SEQUENCE *****)
                  (* on gnre la trace de te1 avec :
                     - goto(cl,n) => goto(cl,n fby te2)
                     - vanish => te2
                  *)
              | TE_fby (te1, te2) -> (
			         let rec_cont = 
                    function
				          |	None -> None
				          |	Some c -> (
					           match c with
					             | Split _ -> assert false
					             | Goto (cl,n) -> cont (Some (Goto (cl, put_in_seq n te2)))
					             | Vanish -> rec_gentrans data te2 acc cont
					             | _ -> cont (Some c)
         	            )
			         in
                    rec_gentrans data te1 acc rec_cont
                )
                  (**** CHOIX *****)
                  (* chaque choix est trait dans
                     l'environnement global,
                     on dcore la branche avec le poids
                     correspondant et une action vide *)

              (* ICI *)

              | TE_choice wtel -> (
                  let res = ref [] in
                  let tc (te, we) =
                    match (rec_gentrans data te acc cont) with
                      | None -> ()
                      | Some tt -> res := (tt, opt_weight_of_opt_algexp we, Guard.empty)::!res
                  in
                    List.iter tc wtel;
                    match !res with
                      | [] -> None
                      | ttlist -> Some (Split ttlist)
                )
                  (*** PRIORITE ****)
              | TE_prio tel -> (
                  let rec doit = ( 
                    function
                        [] -> assert false
                      |  [te] -> (
                           rec_gentrans data te acc cont
                         )
                      |  te::tail -> (
                           let first = rec_gentrans data te acc cont in
                           let others = doit tail in 
                             match (first, others) with
                                 (None,None) -> None
                               |  (Some f, None) -> Some f
                               |  (None, Some o) -> Some o
                               |  (Some f, Some o) -> Some
                                    (Split [(f, Some huge_weight, Guard.empty) ; 
                                            (o, None, Guard.empty)])
                         )
                  ) in doit tel
                )
                  (*** BOUCLE "INFINIE" ***)
              | TE_loop te -> (
		            let rec_cont = ( function
				                         |	None -> None
				                         |	Some c -> (
					                          match c with
					                            | Split _ -> assert false
					                            | Goto (cl,n) -> cont (Some (Goto (cl, put_in_seq n x)))
					                            | Vanish -> None
					                            | _ -> cont (Some c)
				                           )
                                 ) in
                    (* on gnre en priorit les cas ``boucle'' *)
                  let goon = rec_gentrans data te acc rec_cont in
                    (* on considre aussi le cas vanish *)
                  let stop = cont (Some Vanish) in
                    match (goon,stop) with
                        (None,None) -> None
                      |  (Some g, None) -> Some g
                      |  (None, Some s) -> Some s
                      |  (Some g, Some s) -> Some
                           (Split [(g, Some huge_weight, Guard.empty) ; (s, None, Guard.empty)])
                )
                  (*** BOUCLE intervale ****)
              | TE_loopi (cpt,min,max,te,si) -> (
                  (* similaire  la boucle sauf pour les poids et les effets
                     de bord :
                     goon -> interval_goon(pre cpt, min, max)
                     + loop_incr_exp cpt
                     stop -> interal_stop(pre cpt, min, max)
                     + loop_reset_exp cpt
                  *)
		            let rec_cont = 
                    function
				          |	None -> None
				          |	Some c -> (
					           match c with
					             | Split _ -> assert false
					             | Goto (cl,n) -> cont (Some (Goto (cl, put_in_seq n x)))
					             | Vanish -> None
					             | _ -> cont (Some c)
				            )
                  in
                  let goon = rec_gentrans data te acc rec_cont in
                  let stop = cont (Some Vanish) in
                    match (goon,stop) with
                        (None,None) -> None
                      |  (Some g, None) -> Some g
                      |  (None, Some s) -> Some s
                      |  (Some g, Some s) -> (
                           let gw = dynamic_weight_exp LutPredef.kw_interval_goon cpt [min; max]  in
                           let ga = incr_loop_cpt cpt in
                           let sw = dynamic_weight_exp LutPredef.kw_interval_stop cpt [min; max]  in
                           let sa = reset_loop_cpt cpt in
                             Some
                               (Split [(g, Some (W_exp gw), Guard.of_exp ga si) ; 
                                       (s, Some (W_exp sw), Guard.of_exp sa si)])
                         )
                )
                  (*** BOUCLE moyenne ****)
              | TE_loopa (cpt,av, ecopt,te,si) -> (
                  (* similaire  la boucle sauf pour les poids :
                     goon -> loopa_goon(pre cpt, min, max)
                     stop -> loopa_stop(pre cpt, min, max)
                  *)
		            let rec_cont = 
                    function
				          |	None -> None
				          |	Some c -> (
					           match c with
					             | Split _ -> assert false
					             | Goto (cl,n) -> cont (Some (Goto (cl, put_in_seq n x)))
					             | Vanish -> None
					             | _ -> cont (Some c)
				            )
                  in
                  let goon = rec_gentrans data te acc rec_cont in
                  let stop = cont (Some Vanish) in
                    match (goon,stop) with
                        (None,None) -> None
                      |  (Some g, None) -> Some g
                      |  (None, Some s) -> Some s
                      |  (Some g, Some s) -> (
                           let ec = match ecopt with
                               Some x -> x
                             | None -> CoAlgExp.of_iconst "0" 
                           in 
                           let gw = dynamic_weight_exp LutPredef.kw_average_goon cpt [av; ec]  in
                           let ga = incr_loop_cpt cpt in
                           let sw = dynamic_weight_exp LutPredef.kw_average_stop cpt [av; ec]  in
                           let sa = reset_loop_cpt cpt in
                             Some
                               (Split [(g, Some (W_exp gw), Guard.of_exp ga si) ; 
                                       (s, Some (W_exp sw), Guard.of_exp sa si)])
                         )
                )
                  (** Assert 
                  *)
              | TE_assert (a, te, si) -> (
		            let rec_cont = 
                    function
				          |  None -> None
				          |  Some c -> (
					            match c with
					              | Split _ -> assert false
					              | Goto (cl,n) -> cont (Some (Goto (cl, TE_assert (a,n,si))))
					              | _ -> cont (Some c)
				             )
			         in
			           (* HERE: CONTINUE EVEN IF IT IS UNSAT *)
			           try (
				          let rec_acc = Guard.add ~unalias:unalias ~context:data a acc si in
         	            rec_gentrans data te rec_acc rec_cont
			           ) with Guard.Unsat -> (
				          (* HERE: cont or not cont ???? *)
      		          Verbose.put ~flag:dbg " !!!!! CUT A BRANCH\n";
				          cont None
			           )
                )
                  (** exist: matters is some vars are re-initialised
			             recursive call with a new store
                  *)
              | TE_exist (ectx, te) -> (
		            let new_data = add_pres unalias data ectx in 
                    rec_gentrans new_data te acc cont  
                )
                  (** RAISE 
                      on apple simplement le callback avec le nom de l'exception *)
              | TE_raise s -> (
		            cont (Some (Raise s))
                )
                  (** on apple recursivement le traitement de e avec un nouveau raise callback :
                      - si le callback est lev avec i, apple recursivement :
                      * si eco = Some ec : ec avec les callbacks de l'appel
                      * sinon le vanish de l'appel
                      - sinon apple le callback raise de l'appel
                      ET avec un nouveau goto callback :
                      * goto(cl,n) => goto(cl, catch(i,n,eco))
                  *)
              | TE_catch (i,e,eco) -> (
		            let rec_cont = ( function
				                         |	None -> None
				                         |	Some c -> (
					                          match c with
					                            | Split _ -> assert false
					                            | Goto (cl,n) -> cont (Some (Goto (cl, TE_catch(i, n, eco))))
					                            | Raise x -> (
            		                             if ( x == i) then (
               		                            match eco with
               		                                Some ec -> (
                  		                               rec_gentrans data ec acc cont
               		                                ) |
               		                                    None -> cont (Some Vanish)
            		                             ) else (
               		                            cont (Some (Raise x))
            		                             )  
					                              )
					                            | _ -> cont (Some c)
				                           )
                                 ) in
                    rec_gentrans data e acc rec_cont
                )
                  (** TRY *)  
                  (* optimisation : try(eps,_) = eps *)
              | TE_try (TE_eps,_) -> cont (Some Vanish)
              | TE_try (e,eco) -> (
                  (** on cr un choix binaire entre :
                      - priorit (poids infini) e avec un nouveau goto callback :
                      * goto(cl,n) -> goto(cl, try(n,eco))
                      - sinon :
                      * si "eco = Some ec" alors appel rec. sur ec
                      * si "eco = None" alors appel du vanish 
                  *)
                  (* la branche prio *)
		            let rec_cont = ( function
				                         |	None -> None
				                         |	Some c -> (
					                          match c with
					                            | Split _ -> assert false
					                            | Goto (cl,n) -> cont (Some (Goto (cl, TE_try(n,eco))))
					                            | _ -> cont (Some c)
				                           )
                                 ) in
                  let prio = rec_gentrans data e acc rec_cont in

                  (* la branche par defaut *) 
                  let defaut = ( match eco with
                                   | Some ec -> (
                                       rec_gentrans data ec acc cont
                                     )
			                          | None -> cont (Some Vanish)
                               ) in
                    match (prio,defaut) with
                        (None,None) -> None
                      |  (Some p, None) -> Some p
                      |  (None, Some d) -> Some d
                      |  (Some p, Some d) -> (
                           Some 
                             (Split [(p, Some huge_weight, Guard.empty) ; (d, None, Guard.empty)])
                         )
                )
                  (** Parallle : il y en a toujours au - un !  *)
              | TE_para ([]) -> assert false
              | TE_para ([e]) -> (
                  (* le dernier continue son chemin *)
                  rec_gentrans data e acc cont
                )
              | TE_para (e::el) -> (
		            (* continuation for the head statement *)
		            let head_cont =  
                    function 
				          |	None -> None
				          |	Some c -> (
					           match c with
					             | Split _ -> assert false
					             | Vanish -> (
         			              (* 1st vanishes: others continue *)
						              rec_gentrans data (TE_para(el)) acc cont
					               )
					             | Raise s -> (
         			              (* 1st raises s: whole raises s *)
						              cont (Some (Raise s))
					               )
					             | Goto (cl,n) -> (
						              (* 1st do a trans ... *)
						              (* HERE: DO EVAL! *) 
						              let tail_cont = 
                                  function
							               | None -> None
							               | Some c -> (
								                match c with
								                  | Split _ -> assert false
								                  | Vanish -> (
									                   (* others vanish, 1st continue *)
									                   cont (Some (Goto (cl,n)))
								                    )
								                  | Raise s -> (
									                   (* others raise, whole raises *)
									                   cont (Some (Raise s))
								                    )
								                  | Goto (tcl, tn) -> (
									                   (* HERE -> something to do ? *)
									                   cont (Some (Goto (tcl, put_in_para n tn)))
								                    )
							                 )
                                in try (
							               (* let tail_acc = Guard.merge ~unalias:unalias ~context:data cl acc in *)
            			               rec_gentrans data (TE_para(el)) cl tail_cont
						                ) with Guard.Unsat -> (
							               (* HERE: cont or not cont ???? *)
      					               Verbose.put ~flag:dbg " !!!!! CUT A BRANCH\n";
							               cont None
						                )
					               )
					                 (* | _ -> cont (Some c) *)
				            )
		            in
                    rec_gentrans data e acc head_cont
                )
	           | TE_strong_assert (_, _, _)
	           | TE_dyn_loop (_, _, _)
	           | TE_omega _
	           | TE_noeps _
	           | TE_dyn_choice (_, _)
		          -> assert false
          ) in
        (* top-level cont *)
      let top_cont = 
        function
		    | None -> None
		    | Some c -> (
			     match c with
			       | Split _ -> assert false
			       | _ -> Some c
		      )
	   in
      let res = rec_gentrans data x Guard.empty top_cont in
        (*dump_ttree res;*)
        Verbose.exe ~flag:dbg 
          (fun () -> 
             Printf.printf
               "gentrans stats: %s\n" (ttree_stats res));
        res
    )

(****************************************************************************
GENERATION DE L'AUTOMATE COMPLET
-----------------------------------------------------------------------------
Nommage des tats :
- les stables sont "state<index de stable>
- les transients sont "<father>_<sous-index>

- N.B. il existe toujours un tat final potentiel
  qu'on nomme "vanish"

****************************************************************************)

(** dump des transitions *)
let dump_trans tr = (
printf "TRANSITION [\n";
printf "   SRC: %s\n" tr.src;
printf "   WEIGHT: ";
   dump_weight tr.wgt ;
printf "\n";
printf "   COND: " ;
   Guard.dumpf stdout tr.form ;
printf "\n";
printf "   DEST: %s\n" tr.dest;
printf "]\n"
)

let new_stable_state (it: t) (e : CoTraceExp.t) = (
   let ssi = it.nb_stables in
   it.nb_stables <- it.nb_stables + 1;
   let res = sprintf "state%d" ssi in
   Hashtbl.add it.states res (SS_stable e);
   res
)

let new_transient_state (it: t) (father: string) (index: int) = (
   it.nb_transients <- it.nb_transients + 1;
   let res = sprintf "%s_%d" father index in
   Hashtbl.add it.states res SS_transient;
   res
)

(** recherche/cre une association trace/state *)
let get_stable (it:t) e = (
   try (
      Util.hfind it._trace2state e
   ) with Not_found -> (
      let res = new_stable_state it e in
Verbose.exe ~level:3
  (fun () -> Printf.printf  "##new state=\"%s\" exp=%s\n" res (CoTraceExp.dumps e));
      Hashtbl.add it._trace2state e res;
      Hashtbl.add it._state2trace res e ;
      it.todo <- res :: it.todo;
      res
   )
)


(** recherche/cre un tat puits
   N.B, on garde tel que l'ident qui est suppose tre unique !
*)
let get_sink (it:t) x = (
   try (
      let _ = Util.hfind it.states x in x
   ) with Not_found -> (
Verbose.put ~level:3 "##new sink=\"%s\"\n" x ;
      Hashtbl.add it.states x (SS_final x);
      it.nb_sinks <- it.nb_sinks + 1 ;
      x
   )
)

let init (xenv : Expand.t) = (
   let res = {
      source_code = xenv;
      nb_stables = 0;
      nb_transients = 0;
      init_control = "";
      (** L'tat final est un puit *)
      final_control = "";
      states = Hashtbl.create 100;
      transitions = [];

      nb_sinks = 0;
      _state2trace = Hashtbl.create 50; 
      _trace2state = Hashtbl.create 50;
      _config2ttree = Hashtbl.create 50;

      (* liste des inexplors *)
      todo = [];
   } in
   let is = Expand.main_trace xenv in
   let ie = (Util.hfind (Expand.trace_tab xenv) is).ti_def_exp in
   res.init_control <- get_stable res ie;
   res.final_control <- get_sink res "vanish";
   res
)


let rec ttree2trans (it:t) (src: string) (tt : ttree) = (
   match tt with
   | Vanish -> 
      [ { src = src; wgt = None ; form = Guard.empty; dest = it.final_control } ]
   | Raise x ->
      [ { src = src; wgt = None ; form = Guard.empty; dest = get_sink it x; } ]
   | Goto (cl, n) -> 
      [ { src = src; wgt = None ; form = cl; dest = get_stable it n; } ]
   | Split twl -> (
      let child_cpt = ref 0 in
      let treat_choice trs (t, wo, a) = (
         let dest = new_transient_state it src !child_cpt in
         incr child_cpt;
         let t0 = { src = src; wgt = wo ; form = a ; dest = dest; } in
         (ttree2trans it dest t) @ (t0 :: trs)
      ) in
      ( List.fold_left treat_choice [] twl)
   )
)


let get_state_def (it:t) (ix: string) = (
   Util.hfind it._state2trace ix
)

let get_state_info (it:t) (ix: string) = (
   Util.hfind it.states ix
)

(*
*)
let config2ttree (it:t) (cfg: config) = (
	let ix = cfg.control in
   let e = Util.hfind it._state2trace ix in
   let data = cfg.data in
	(* use cash *)
	let res = try (
		let tt = Util.hfind it._config2ttree cfg in
		Verbose.put ~level:2 "##config2ttree: \"%s\" cached\n" ix ;
		if (Utils.paranoid ()) then (
			let tt' = gentrans it.source_code data e in
			if(tt' <> (Some tt)) then assert false
		);		
		tt
	) with Not_found -> (
		Verbose.exe ~level:2 
                (fun ()  -> Printf.printf "##config2ttree: \"%s\" = %s\n" ix (CoTraceExp.dumps e));
     	match ( gentrans it.source_code data e) with
     	(* match ( gentrans_old it.source_code e) with *)
     	Some tt -> (
        	Hashtbl.add it._config2ttree cfg tt;
        	(* ttree2trans it ix tt *)
        	tt
     	) |
     	None -> raise (Failure "unexpected toplevel Deadlock")
   ) in
	res
)

type gtree = string * gtree_node 
and gtree_node =
|  GT_leaf of (cond * string)
|  GT_choice of (weightexp option * gtree) list
|  GT_stop of string

(* debug : size ? *)
let rec gtree_size (_,gt) = (
	match gt with
	|	GT_leaf _ -> 1
	|	GT_stop _ -> 1
	|	GT_choice ll -> (
		List.fold_left (fun a (_, g) -> a + (gtree_size g)) 1 ll
	)
)

let rec ttree2gtree (it:t) (acc: cond) (src: string) (tt : ttree) = (
   match tt with
   | Vanish ->  (src, GT_stop it.final_control)
   | Raise x -> (src, GT_stop (get_sink it x))
   | Goto (cl, n) -> (src, GT_leaf (Guard.merge acc cl, get_stable it n))
   | Split twl -> (
(* |  Split of (ttree * weightexp option * CoAlgExp.t list) list *)
      let child_cpt = ref 0 in
      let treat_choice :
         (ttree * weightexp option * cond) -> (weightexp option * gtree) =
      fun (t, wo, a) -> (

         let dest = new_transient_state it src !child_cpt in
         incr child_cpt;
			let new_acc = if (a = Guard.empty) then acc else Guard.merge acc a in
         let cht = ttree2gtree it new_acc  dest t in
         (wo, cht)
      ) in
      (src, GT_choice (List.map treat_choice twl))
   )
)

let rec config2gtree (it:t) (cfg: config) = (
	let ix = cfg.control in
   let tt = config2ttree it cfg in
   ttree2gtree it Guard.empty ix tt
)

let config2trans (it:t) (cfg: config) = (
	let ix = cfg.control in
   let tt = config2ttree it cfg in
   ttree2trans it ix tt
)

let dump (auto : t) = (
   printf "AUTOMATON (init: %s stables: %d transients: %d)\n"
      auto.init_control auto.nb_stables auto.nb_transients
   ;
   List.iter dump_trans auto.transitions
)
