(* Time-stamp: <modified the 24/07/2018 (at 16:03) by Erwan Jahier> *)
(* Mimick the behavior of 'rdbg -lurette', but without the dependency
   on ocaml *)
open Event

open RdbgWrapperArgs

let url_doc="http://www-verimag.imag.fr/DIST-TOOLS/SYNCHRONE/lustre-v6/"

let _ =
  lurette := true;
  usage := (Sys.argv.(0) ^
               " is equivalent to 'rdbg -lurette', except that the lustre-v6
and the lutin plugins are statically linked to it, to avoid the need of an 
ocaml compiler. One can not use other language plugins as 'rdbg -lurette'
would allow.
cf also "^url_doc^"

usage: "^ Sys.argv.(0) ^" [<options>] 
     where <options> are:")

             
let get_extension f =
  try
    let bf = Filename.chop_extension f in
    let lf,lbf = String.length f, String.length bf in
    String.sub f (lbf+1) (lf - lbf - 1)
  with _ -> ""

let args= ref []
let ml_files = ref []

let add_args arg = 
  if get_extension arg = "ml" 
  then 
    ml_files := arg::!ml_files 
  else
    args := (arg::!args)

let others = ref []
let (add_other : string -> unit) =
  fun s -> 
    others := s:: !others

let other_args:string =
  mkoptab ();
  Arg.parse_argv Sys.argv !arg_options add_args !usage;
  String.concat " " (List.rev !others)

(**********************************************************************************)
let (gen_reactive_program : string -> RdbgArg.reactive_program) =
  fun str -> 
  let args = Str.split (Str.regexp "[ \t]+") str in
  let tool = Filename.basename (List.hd args) in
  let plugin =
    match tool with
    | "lutin"   -> LutinRun.make (Array.of_list args)
    | "lv6" | "lus2lic" -> Lv6Run.make  (Array.of_list (args@["--expand-io-type"]))
    | "ocaml" -> OcamlRun.make (List.nth args 1)
    | _ -> StdioRun.make str
  in
  Ocaml(plugin)

open RdbgArg    
let _ =        
  let sut_plugins = List.map gen_reactive_program (!suts @ !suts_nd) in
  let env_plugins = List.map gen_reactive_program (!envs @ !envs_nd) in
  let oracle_plugins = List.map gen_reactive_program (!oracles @ !oracles_nd) in

  args.suts <- sut_plugins;
  args.envs <- env_plugins;
  args.oracles <- oracle_plugins;

  args.step_nb <- !test_length;
  args.display_sim2chro <- !display_sim2chro ;
  args.display_gnuplot <- !display_gnuplot ;
  args.verbose <- if !verbose then 1 else 0 ;
  args.output <- !output_file ;
  
  args.debug_rdbg <- !drdbg;
  args.rdbg <- false;

  args.cov_file <- "lurette.cov";

  ()

(**********************************************************************************)

   

let _ =
  if args.suts = [] then (
    Printf.printf "\027[35mW: Please set at least one sut \027[00m\n\n" ;
    flush stdout;
    help()
  )
  else
  try RdbgRun.lurette_start()
  with
  | RdbgRun.OracleError str ->
     Printf.printf "\027[35m %s \027[00m\n"  str;
     flush stdout
                     
  | Dynlink.Error msg -> (* deadcode *)
     Printf.eprintf "\n*** error in lurette (Dynlink.loadfile %s).\n*** %s.\n" 
       (List.fold_left (fun acc x -> acc^" "^x) "" args._others)
       (Dynlink.error_message msg);
     RdbgRun.clean_terminate(); 
     flush stderr;
     exit 2
  | Event.End(i) ->
     RdbgRun.clean_terminate();
     exit i
  | pb ->
     RdbgRun.clean_terminate();
     output_string args.ocr (Printexc.to_string pb);
     Printf.printf "bye\n"; flush stdout; exit 2
;;

let _ =
  Printf.printf "lurette: bye\n"; flush stdout; exit 0

