(** Miscellaneous general purposes functions that depends on env var.  *)

(************************************************************************)


(** [gv ps_file] calls the post-script visualizer [gv] on [ps_file]. *)
let (gv: string -> unit) = Util.gv

let mygetenv x = 
  let x = 
    match Sys.os_type with
      | "Win32" -> (x^"_DOS")
      | _ ->  x
  in
    try Unix.getenv x 
    with Not_found -> x^" env var not defined"

let mygetenv_def def x = 
  let x = 
    match Sys.os_type with
      | "Win32" -> (x^"_DOS")
      | _ ->  x
  in
    try Unix.getenv x 
    with Not_found -> def

(** [ps2pdf ps_file] calls the post-script visualizer [ps2pdf] on [ps_file]. *)
let (ps2pdf: string -> unit) =
  fun pdf_file ->
    let pdf_viewer =
      try Unix.getenv "PS2PDF"
      with _ ->
	     print_string "*** Can not find PDF_VIEWER env variable.\n";
	     flush stdout;
	     ""
    in
      ignore (Util.my_create_process pdf_viewer [pdf_file])


	  

(* useful to use scade compiler under cygwin *)
let (cygpath : string -> string) =
  fun file ->
    if
      Unix.getenv "HOST_TYPE" = "cygwin"
    then
(*       ("`cygpath -w " ^ file ^ "` ") *)
      file
    else
      file


let lurette_path =
  try mygetenv "LURETTE_PATH" with _ -> ""


(* XXX *)
(*     print_string "Environment var LURETTE_PATH is unset.\n"; *)
(*     exit 2 *)

let lurette_doc = Filename.concat lurette_path "doc"
let lurette_lib = Filename.concat lurette_path "lib"
let lurette_man() = Util.pdf (Filename.concat lurette_doc "lurette-man.pdf")
let lutin_man()   = Util.pdf (Filename.concat lurette_doc "lutin-man.pdf")
let lutin_tuto()  = Util.pdf (Filename.concat lurette_doc "lutin-tuto-pdf.pdf")
	
(** [lus2ec prog node] calls the lus2ec compiler *)
let (lus2ec: string -> string -> string -> bool) =
  fun lustre_prog lustre_node dir ->
    try
      let lus2ec = 
        try mygetenv "LUS2EC" with _ -> "lus2ec" in
	     ignore (Util.my_create_process ~std_out:Unix.stderr ~wait:true lus2ec
	               [
	                 ("" ^ lustre_prog ^ "");  lustre_node;
	                 "-o"; ("" ^ (Filename.concat dir (lustre_node ^ ".ec")) ^ "")
	               ]);
        true
    with e -> 
      print_string (Printexc.to_string e);
      flush stdout;
      false
let lv42ec = lus2ec

(** [c2dro cfile]  *)
let (c2dro: string -> bool) =
  fun cfile ->
    let base = Filename.chop_extension cfile in
    let dro =  base ^ ".dro" in
    let include_dir = Filename.concat lurette_path "include" in
      try
        let cc = mygetenv_def "gcc" "GCC" in
        let cc::opt = Str.split (Str.regexp " ") cc in
	       ignore (Util.my_create_process ~std_out:Unix.stderr ~wait:true cc
	                 (opt@[cfile;  "-fPIC";  "-shared"; "-o"; dro; "-I"; include_dir ]));
          true
      with e -> 
        print_string (Printexc.to_string e);
        flush stdout;
        false


let (string_to_string_list : string -> string list) = Util.string_to_string_list

(* Should be an env var ? *)
let scade_option () =
  let str =
    try Unix.getenv "SCADE_COMPIL_OPTION"
    with _ ->
(*       print_string "*** Warning: can not find SCADE_COMPIL_OPTION env variable.\n"; *)
(*       flush stdout; *)
      ""
  in
    string_to_string_list str
	


let not_installed prog =
  output_string stdout (
    "\n*** " ^ prog ^ " does not seem to be installed on this machine.\n" ^
    "*** Once you have installed it, you need to rerun the INSTALL script in" ^
    "the lurette install dir. \n"
  );
  flush stdout


let (del : string -> out_channel -> out_channel -> unit) =
  fun file oc ec -> 
    let host = Sys.os_type in
    let cmd = 
      if host = "Win32" then
	     "del /Q " ^ file
      else
	     "rm -f " ^ file
    in
      output_string ec (cmd ^ "\n");
      flush ec;
      if Sys.command cmd  <> 0 then
	     (
	       output_string ec ("Warning: " ^cmd ^ " command failed.\n");
	       flush ec
	     )

open Util
let (scade2lustre: string -> bool) =
  fun prog ->
    let _ = assert (Filename.check_suffix prog ".saofdm") in
      try
	 let compiler = Unix.getenv "SCADE2LUSTRE" in
	   if compiler = "no" then (not_installed "scade2lustre"; false) else
	     (
	       del ((Filename.chop_extension prog) ^ ".lus") stdout stderr;
	       match Util.my_create_process ~wait:true compiler
		 [
		   "-model";
		   prog
		 ] with
               | KO -> false
               | PID _ | OK -> (
		 let f = (Filename.chop_extension prog) ^ ".lus" in 
		   if Sys.file_exists f then
		     (
		       output_string stderr (f ^ " has been created.\n"); flush stderr;
		       true
		     )
		   else
		     (
		       output_string stderr ("*** Error: "^ f ^ " has NOT been created.\n"); flush stderr;
		       false
		     )
	       )
	     )
       with e ->
	 print_string (Printexc.to_string e);
	 flush stdout;
	 false



let (etp_to_saofdm: string -> string -> bool) =
  fun project node ->
    let _ = assert (
      Filename.check_suffix project ".etp" ||
      Filename.check_suffix project ".vsp" 
    ) 
    in
      try
	let compiler = Unix.getenv "SCADE" in
	  if compiler = "no" then (not_installed "scade";false) else
	    match Util.my_create_process ~wait:true compiler
	      [
		"-convert";  
		"-tosaofdm";  
		project;
		((Filename.chop_extension project) ^ ".saofdm")
	      ]
	    with
                KO -> false
              | PID _ | OK -> 
	          Sys.file_exists ((Filename.chop_extension project) ^ ".saofdm")
      with e -> 
	print_string (Printexc.to_string e);
	flush stdout;
	false

let (scade_cg: string -> string -> string -> bool) =
  fun prog node dir ->
    let _ = assert (Filename.check_suffix prog ".saofdm") in
      try 
	     let compiler = Unix.getenv "SCADE_CG" in
	       if compiler = "no" then 
	         (
	           not_installed "scade_cg"; 
	           false
	         ) 
	       else
	         match Util.my_create_process ~wait:true compiler
	           ([
		           "-model";  
		           (cygpath prog);
		           "-node"; 
		           node;
		           "-target_dir"; 
		           (cygpath dir)
	            ] @ scade_option())
            with
              | KO  -> false
              | PID _ | OK -> true
                  (* 	    && *)
                  (* 	    Sys.file_exists ((Filename.chop_extension prog) ^ ".lus") *)
      with e -> 
	     print_string (Printexc.to_string e);
	     flush stdout;
	     false


let (lustre2C: string -> string -> string -> bool) =
  fun prog node dir ->
    let _ = assert (Filename.check_suffix prog ".lus") in
      try
	let flag = try [Unix.getenv "C_GENERATOR_FLAG"] with _ -> [] in
	let compiler = Unix.getenv "LUSTRE2C" in
	  if not (Sys.file_exists node) then Unix.mkdir node 700;
	  if compiler = "no" then (not_installed "lustre2C"; false) else
	    match Util.my_create_process ~wait:true compiler
	      ([
		 prog;
		 "-node";
		 node] @ 
		 scade_option() @ [
		 "-target_dir";
		 dir
	       ] 
	       @ flag
	      )
            with
              | KO  -> false
              | PID _ | OK -> true
(* 	    &&  *)
(* 	    Sys.file_exists ((Filename.chop_extension prog) ^ ".c")  *)

      with e -> 
	print_string (Printexc.to_string e);
	flush stdout;
	false
		
(** [ec2c prog] calls the ec2c compiler *)
let (ec2c: string -> string -> bool) =
  fun lustre_prog dir ->
    try
      let ec2c = mygetenv "EC2C" in
      let flag = try [Unix.getenv "C_GENERATOR_FLAG"] with _ -> [] in
	     match Util.my_create_process ~wait:true ec2c 
	       ([
	          "-o";
	          ("" ^ (Filename.concat dir lustre_prog) ^ "");
	          (lustre_prog ^ ".ec")
	        ]
	        @ flag
	       )
        with
          | KO  -> false
          | PID _ | OK -> true
    with e -> 
      print_string (Printexc.to_string e);
      flush stdout;
      false
	
(** [lv62ec prog] generates ec .code from lustre V6 programs *)
let (lv62ec: string -> string -> string -> bool) =
  fun lustre_prog lustre_node dir ->
    try
      let lv6 = mygetenv "LUS2LIC" in
	     match Util.my_create_process ~std_out:Unix.stderr ~wait:true lv6
	       [
	         lustre_prog; "--node" ; lustre_node; "-ec";
	         "-o"; (Filename.concat dir (lustre_node ^ ".ec"))
	       ]
        with
          | KO  -> false
          | PID _ | OK -> true
    with e -> 
      print_string ("LUS2LIC:" ^ (Printexc.to_string e));
      flush stdout;
      false


let (gnuplot: bool -> string -> unit) =
  fun verb riffile ->
    let gnuplotrif = mygetenv_def "gnuplot-rif" "GNUPLOTRIF" in
    let cmd =  (gnuplotrif ^ " " ^ riffile) in
    let cmd = if verb then cmd ^ " -verbose" else cmd in
    let res = Sys.command cmd in
      if res > 0 then (
        output_string stderr ("'"^ cmd ^ "' has been launched but failed.\n"); 
        flush stderr
      )

(* Returns the function to kill the gnuplot process *)
let (gnuplot_dyn : bool -> string -> (unit -> unit)) =
    fun verb file ->
      let dummy = fun () -> () in
      try
        let gnuplotrif = mygetenv_def "gnuplot-rif" "GNUPLOTRIF" in
        let killme =
	       match Util.my_create_process ~wait:false gnuplotrif ["-dyn"; file ]
          with
            | KO  -> dummy
            | PID pid  -> (fun () -> Unix.kill pid Sys.sigkill)
            | OK -> dummy
        in
        killme
      with e -> 
        print_string (Printexc.to_string e);
        flush stdout;
        dummy

let (gnuplot_ps: string -> unit) =
  fun file ->
    let gnuplotrif = mygetenv_def "gnuplot-rif" "GNUPLOTRIF" in
    let cmd =  (gnuplotrif ^ " -cps " ^ file) in
    let res = Sys.command cmd in
    if res > 0 then (
      output_string stderr ("'"^ cmd ^ "' has been launched but failed.\n"); 
      flush stderr
    )


let (sim2chro_dyn : unit -> out_channel) =
  fun () ->
    let sim2chro = mygetenv "SIM2CHRO" in
    let oc = Unix.open_process_out (sim2chro ^ " -ecran /dev/null") in
      oc          

let (sim2chro: string -> bool) =
  fun file ->
    try
      let sim2chro = mygetenv "SIM2CHRO" in
	     match Util.my_create_process ~wait:false sim2chro
	       [
	         "-ecran"; 
            "-in";
            file;
	         "/dev/null"
	       ]
        with
          | KO  -> false
          | PID _ | OK -> true
    with e -> 
      print_string (Printexc.to_string e);
      flush stdout;
      false

let (slash_path :string -> string) =
  fun path -> 
    Str.global_replace (Str.regexp_string "/") "\\\\" path


let (cp : string -> string -> out_channel -> out_channel -> bool) =
  fun file dir oc ec -> 
    let host = Sys.os_type in
    let cmd = 
      if host = "win32" then
	     "copy " ^ (slash_path file) ^ " " ^ (slash_path dir)
      else
	     "cp " ^ file ^ " " ^ dir
    in
      (*       output_string ec (cmd ^ "\n"); *)
      (*       flush ec; *)
      if Sys.command cmd  <> 0 then
	     (
	       output_string oc (cmd ^ " command failed.\n");
	       flush oc;
	       false
	     )
      else
	     true

(****************************************************************************)
(** Call cpp on file and returns the file name containing the result *)
(* let (cpp : string -> string) = *)
(*   fun file -> *)
(*     let _ = assert (Filename.check_suffix file ".h") in *)
(*     let dir = Filename.dirname file in *)
(*     let base = Filename.basename file in *)
(*     let new_file = Filename.concat dir ((Filename.chop_extension base) ^ "_cpp.h") in *)
(*       try *)
(* 	let cpp = try Unix.getenv "CPP" with _ -> "cpp" in *)
(* 	  ignore (Util.my_create_process cpp [file; "-C"; "-o"; new_file]); *)
(* 	  new_file *)
(*       with e ->  *)
(* 	print_string (Printexc.to_string e); *)
(* 	flush stdout; *)
(* 	file *)
(*  *)


