

open LtopArg

let blank_star = (Str.regexp "[ \t]+") 

let (f : unit -> int) =
  fun () ->
    try
      let seed_str =
	     match args.seed with
	         None -> []
	       | Some i -> "-seed"::[(string_of_int i)]
      and precision_str = "-precision"::[string_of_int args.precision]
      and compute_volume_str =
	     if (args.compute_volume) then ["--compute-poly-volume"] else []
      and verb_str = ["-v"; (string_of_int (args.verbose))]
      and step_mode_str = [step_mode_to_string args.step_mode]
      and orac_str =
	     match args.oracle_cmd with
	         "" -> []
	       | str -> 
		        "--oracle"::[(Str.global_replace blank_star "+" str)]

      and step_nb_str = "-l"::[(string_of_int args.step_nb)]
      and dlvr_str =
	     if (args.display_local_var) then ["-locals"] else []
      and env_str =
	     let lut_list = Util2.string_to_string_list args.env in
	       lut_list
      in
      let lutin = "lutin" ^ ExtTools.dot_exe in
      let prefix = Util2.string_to_string_list args.prefix in
	     
      let arg_list0 =
	     List.flatten
	       [
	         prefix;
	         [lutin];
	         ["-boot"];
	         step_nb_str ;
	         orac_str;
   	      step_mode_str;
	         seed_str;
	         precision_str;
	         compute_volume_str;
	         verb_str ;
   	      dlvr_str ;
	         env_str
	       ]
      in
      let prog = if args.prefix = "" then lutin else List.hd prefix in
      let arg_list = Util.rm "" arg_list0 in
      let arg_array = Array.of_list arg_list in
      let args_str = List.fold_left (fun x acc -> (x^" " ^acc)) "\n" arg_list in

      let args_str_sut = args.sut_cmd in
      let arg_list_sut0 = Util2.string_to_string_list args_str_sut in
      let arg_list_sut = 
	     match arg_list_sut0 with
	         "ecexe"::tail -> 
	           if 
		          (* force the use of -r with ecexe *)
		          List.mem "-r" tail 
	           then 
		          arg_list_sut0  
	           else
		          "ecexe"::"-r"::tail
	       | _ -> arg_list_sut0
      in
      let prog_sut = List.hd arg_list_sut in
      let args_sut = Array.of_list arg_list_sut in

      let (lutin_stdin_in,  lutin_stdin_out) = Unix.pipe () in
      let (lutin_stdout_in, lutin_stdout_out) = Unix.pipe () in

      let (sut_stdin_in,  sut_stdin_out) = Unix.pipe () in
      let (sut_stdout_in, sut_stdout_out) = Unix.pipe () in

      let luc_ic = Unix.in_channel_of_descr lutin_stdout_in in
      let luc_oc = Unix.out_channel_of_descr lutin_stdin_out in

      let sut_ic = Unix.in_channel_of_descr sut_stdout_in in
      let sut_oc = Unix.out_channel_of_descr sut_stdin_out in

      let _ = 
	     set_binary_mode_in  luc_ic false;
	     set_binary_mode_in  sut_ic false;
	     set_binary_mode_out luc_oc false;
	     set_binary_mode_out sut_oc false;
	     
	     (* XXX won't work under windows with mingw and MVC++ !! *)
	     Unix.set_nonblock lutin_stdout_in;
	     Unix.set_nonblock sut_stdout_in
      in
      let rif = 
	     if Filename.is_relative args.output then
	       open_out (Filename.concat args.sut_dir args.output)
	     else
	       open_out args.output
      in
      let rec (read_ic : in_channel -> string) =
	     fun ic ->
	       let rec read_loop acc = 
	         try 
	           let line = (input_line ic) in
	           let lgt = String.length line in
	           let line' = 
		          if
		            lgt < 5 || String.sub line 0 5 <> "#outs"
		          then
		            line
		          else
		            (* do not send #outs to the sut, because it does not 
		               understand it *)
		            String.sub line 5 (lgt-5)
	           in
	           let new_acc = acc ^ line' ^ "\n" in
		          if 
		            (args.show_step) && Util.is_substring "# step" line
		          then
		            (* For the progress bar *)
		            (
		              output_string args.ocr 
		                ("--- " ^ 
		                   (String.sub line 2 ((String.length line) - 2)) ^
		                   ":\n");
		              flush args.ocr
		            );
		          
		          if Util.is_substring "The oracle Pid is" line then
		            (
		              output_string args.ocr (line ^ "\n");
		              flush args.ocr
		            );

		          output_string rif (line ^ "\n");
		          read_loop new_acc
	         with Sys_blocked_io -> acc
	       in
	         read_loop ""
      in
      let pid_lutin = 
	     Unix.create_process prog arg_array 
	       lutin_stdin_in lutin_stdout_out (Unix.descr_of_out_channel args.ecr)
      in
      let pid_sut =
	     Unix.create_process prog_sut args_sut
	       sut_stdin_in sut_stdout_out (Unix.descr_of_out_channel args.ecr)
      in

      let _ =
	     output_string args.ecr (args_str ^ "\n");
	     List.iter (fun x -> output_string args.ecr (x ^ " ")) arg_list_sut;
        output_string args.ecr "\n";

	     flush args.ecr;
	     output_string args.ocr (
	       "\nThe Pid of lutin is   " ^ (string_of_int pid_lutin) ^
	         "\nThe Pid of the sut is " ^ (string_of_int pid_sut) ^ "\n");
	     flush args.ocr
      in
      let times0 = Unix.times () in

      let rec lurette_loop sut_out =
	     
	     let luc_out = 
	       output_string luc_oc sut_out; 
	       flush luc_oc;
	       read_ic luc_ic 
	     in
	     let new_sut_out =
	       output_string sut_oc luc_out;
	       flush sut_oc;
	       read_ic sut_ic
	     in
	       if 
	         Util.is_substring "#end" luc_out 
	       then
	         (
	           (* 
		           The lutin process is dead ; kill the sut and 
		           waits for the termination of both
	           *)
	           (try Unix.kill (pid_sut) Sys.sigkill with _ -> ()); 
	           let _ = (Unix.waitpid [Unix.WUNTRACED] pid_sut) in
		          if args.oracle_cmd <> "" then
		            (
		              output_string args.ocr
		                "\n ==> The test completed; no property has been violated.\n\n";
		              flush args.ocr;
		            );
		          snd (Unix.waitpid [Unix.WUNTRACED] pid_lutin)
	         )
	       else if 
	         Util.is_substring "#oracle_returned_false" luc_out 
	       then
	         (
	           (* Ditto + printf *)
	           output_string args.ocr "\n*** The oracle returned false\n";
	           flush args.ocr;
	           (try Unix.kill (pid_sut) Sys.sigkill with _ -> ()) ;
	           let _ = (Unix.waitpid [Unix.WUNTRACED] pid_sut) in
		          snd (Unix.waitpid [Unix.WUNTRACED] pid_lutin)
	         )
	       else
	         (* If one of the 2 processes has been killed before the end 
	            of a normal execution, e.g., if one has cliked on the 
	            xlurette stop button.
	         *) 
	         let (p1, s1) = Unix.waitpid [Unix.WNOHANG] pid_sut in
	         let (p2, s2) = Unix.waitpid [Unix.WNOHANG] pid_lutin in
	           if 
		          p1 <> 0 
	           then
		          (
		            (try Unix.kill (pid_lutin) Sys.sigkill with _ -> ());
		            snd (Unix.waitpid [Unix.WUNTRACED] pid_lutin)
		          )
	           else if
		          p2 <> 0
	           then
		          (
		            (try Unix.kill (pid_sut) Sys.sigkill with _ -> ());
		            let _ = (Unix.waitpid [Unix.WUNTRACED] pid_sut) in
		              s2
		          )
	           else
		          lurette_loop new_sut_out
      in 
	     
      let status_lutin = lurette_loop "" in
	     
	     flush rif;
	     close_out rif;
	     Unix.close lutin_stdin_in;
	     Unix.close lutin_stdout_out;
	     Unix.close lutin_stdin_out;
	     Unix.close lutin_stdout_in;
	     
	     Unix.close sut_stdin_in;
	     Unix.close sut_stdout_out;
	     Unix.close sut_stdin_out;
	     Unix.close sut_stdout_in;
	     
	     close_in luc_ic;
	     close_in sut_ic;
	     close_out luc_oc;
	     close_out sut_oc;
	     
	     ( match status_lutin with
	           
	         | Unix.WEXITED i ->
		          if i = 127 then 1 (* ??? *)
		          else
		            (
		              (* Unix.times is not implemented for the Win32 port of ocaml *)
		              if Sys.os_type <> "Win32" then ( 
		                let times1 = Unix.times () in 
		                let times = 
			               times1.Unix.tms_cutime +.times1.Unix.tms_cstime 
			               -.times0.Unix.tms_cutime -.times0.Unix.tms_cstime 
		                in 
		                let times_str = string_of_float times in 
			               output_string args.ocr 
			                 ("\n      The execution lasted " ^ times_str ^ 
                             " second" ^ 
			                    (if times >= 2. then "s" else "") ^ ".\n" ^ 
                             "********************************************************************\n"); 
			               flush args.ocr); 
		              0
		            )
	         | Unix.WSIGNALED i ->
		          output_string args.ecr(
		            "lutin was killed by signal nb " ^ (string_of_int i) ^ ".\n");
		          flush args.ecr;
		          0
	         | Unix.WSTOPPED i ->
		          output_string args.ecr (
		            "lutin was stopped by signal nb " ^ (string_of_int i) ^ ".\n");
		          flush args.ecr;
		          0
	     )	
    with
	     e -> 
	       output_string args.ocr (Printexc.to_string e);
	       flush args.ocr;
	       1

