(* Time-stamp: <modified the 05/07/2018 (at 14:30) by Erwan Jahier> *)

open RdbgPlugin
type vars = (string * Data.t) list

let debug = ref false
let debug_msg msg = if !debug then (output_string stderr ("*** dbg: "^msg) ; flush stderr)

let output_msg2 msg = output_string stdout msg; flush stdout

(* Which one should I use??? *)
let my_string_of_float = string_of_float
let my_string_of_float = Util.my_string_of_float

let subst_to_string (n,v) = n ^ "=" ^ (Data.val_to_string my_string_of_float v)

let (step_channel : in_channel -> out_channel -> vars -> vars ->
     Data.subst list -> Data.subst list) =
  fun ic oc in_vars out_vars sl ->
      let in_vals_str =
        List.fold_left
          (fun acc (name, _) ->
             let value =
               try List.assoc name sl
               with Not_found ->
                 Printf.fprintf stdout "*** Don't find %s among: %s\n"
                   name (String.concat ", " (List.map fst sl));
                 flush stdout;
                 assert false
             in
               acc ^ " "^ (Data.val_to_string my_string_of_float value)
          )
          ""
          in_vars
      in
      let res =
        debug_msg  ("Writing '" ^ in_vals_str ^"' to channel\n");
        output_string oc (in_vals_str ^"\n");
        flush oc;
        RifIO.read ic None out_vars
      in
        res

(* wrap it with type transformation  *)
let get_io_from_lustre a b =
  let il, ol = Util.get_io_from_lustre a b in
  let il = List.map (fun (id,t) -> id, Data.type_of_string t) il in
  let ol = List.map (fun (id,t) -> id, Data.type_of_string t) ol in
  il, ol


 (* XXX Doable with DynLink? Or via Ezdl? *)

let (make_ec : string -> RdbgPlugin.t) =
  fun ec_file -> 
  
  let ec_in, ec_out = get_io_from_lustre ec_file None in
  let (ec_stdin_in,  ec_stdin_out) = Unix.pipe () in
  let (ec_stdout_in, ec_stdout_out) = Unix.pipe () in
  
  let ec_ic = Unix.in_channel_of_descr ec_stdout_in in
  let ec_oc = Unix.out_channel_of_descr ec_stdin_out in

  let _ = 
    set_binary_mode_in  ec_ic false;
    set_binary_mode_out ec_oc false
  in
  let pid_lustre = 
    let arg_list = ["ecexe"^(Util.exe ()); "-r"; "-rif"; ec_file] in
    let arg_array = Array.of_list arg_list in
    let prog = List.hd arg_list in
    try 
      if !debug then ( 
	     List.iter (fun x -> output_string stderr (x ^ " ")) arg_list;
	     output_string stderr "\n"; 
        flush stderr
      );
      Unix.create_process prog arg_array 
                          ec_stdin_in ec_stdout_out ec_stdout_out
    with Unix.Unix_error(e,_, prog) -> 
      let msg = Unix.error_message e in
      Printf.eprintf "*** Error when creating process with %s: %s\n" prog msg;
      exit 2
  in
  let _ = Printf.eprintf "Process %d (ecexe) created\n" pid_lustre; flush stderr in
  let kill msg =
    (* Printf.print "EOF" *)
    Unix.close ec_stdin_in;
    Unix.close ec_stdin_out;
    Unix.close ec_stdout_in;
    Unix.close ec_stdout_out;
    (try 
        Printf.eprintf "%s\nKilling process %d\n" msg pid_lustre;
        flush stderr;
        Unix.kill pid_lustre Sys.sigterm 
      with e -> (Printf.printf "Killing of ecexe process failed: %s\n" (Printexc.to_string e) ))
  in
  let step = step_channel ec_ic ec_oc ec_in ec_out in
  let step_dbg sl ctx cont =
    let enb = ctx.Event.nb in
    let ctx = Event.incr_event_nb ctx in
    {
      Event.nb = enb;
      Event.step = ctx.Event.step;
      Event.depth = ctx.Event.depth;
      Event.kind = Event.Exit;
      Event.lang = "lustre";
      Event.name=ec_file;
      Event.inputs = [] ;
      Event.outputs = [];
      Event.locals = []; 
      Event.sinfo = None;
      Event.data = ctx.Event.data;
      Event.next = (fun () -> cont (step sl) ctx);
      Event.terminate = ctx.Event.terminate;
    } 
  in 
  {
    id = "";
    inputs = ec_in;
    outputs= ec_out;
    reset= (fun () -> RifIO.write ec_oc "#reset\n";  flush ec_oc);
    kill= kill;
    init_inputs= [];
    init_outputs= [];
    step = step;
    step_dbg = step_dbg
  }

(* Via une edition de liens dynamique *)
let (make_ec_dynlink: string -> string -> string -> RdbgPlugin.t) =
  fun node ec_file dl_file -> 
    let ec_in, ec_out = get_io_from_lustre ec_file None in
    let dl = Ezdl.dlopen dl_file in
    let new_ctx_cfunc =  Ezdl.dlsym dl (node^ "_new_ctx") in
    let step_cfunc = Ezdl.dlsym dl (node^ "_step") in

    let null_ptr = Ezdl.Ptr_carg (Ezdl.cptr_of ()) in
(*     let ctx = Ezdl.cargs2cptr new_ctx_cfunc null_ptr in *)
(*     let step = Ezdl.cargs2void step_cfunc (Ezdl.Ptr_carg ctx) in *)

      assert false

(**********************************************************************************)
let (make_v6 : string -> string -> RdbgPlugin.t) =
  fun lus_file node -> 
    let dir = Filename.dirname lus_file in
      if Util2.lv62ec lus_file node dir then 
        make_ec (node ^ ".ec")
      else
        failwith ("Error when compiling " ^ lus_file ^ " with node " ^ node ^"\n")

(**********************************************************************************)
let (make_v4 : string -> string -> RdbgPlugin.t) =
  fun lus_file node -> 
    let dir = Filename.dirname lus_file in
      if Util2.lv42ec lus_file node dir then 
        make_ec (node ^ ".ec")
      else
        failwith ("Error when compiling " ^ lus_file ^ " with node " ^ node ^"\n")

(**********************************************************************************)
let rec connect_loop sock addr k =
  try Unix.connect sock addr
  with _ -> 
    if k > 0 
    then (
      if !debug then (
        let ni = Unix.getnameinfo addr [] in
          Printf.fprintf stderr "connect %s:%s failed;  try again in a second.\n" 
            ni.Unix.ni_hostname ni.Unix.ni_service; 
          flush stderr
      );
      Unix.sleep 1; 
      connect_loop sock addr (k-1) 
    )
    else failwith "lustreRun: cannot connect to the socket"


let (make_socket_do : string -> int -> in_channel * RdbgPlugin.t) =
  fun sock_adr port -> 
    let _ =
      if !debug then (
        Printf.fprintf stderr "Start a connection on %s:%d\n" sock_adr port; 
        flush stderr)
    in
    let inet_addr = Unix.inet_addr_of_string sock_adr in
    let sock = Unix.socket Unix.PF_INET Unix.SOCK_STREAM  0 in
    let (sock_in, sock_out) =
      try
        connect_loop sock (Unix.ADDR_INET(inet_addr, port)) 100 ;
        if !debug then (
          Printf.fprintf stderr "Socket %s:%d connected \n" sock_adr port; 
          flush stderr);
        (Unix.in_channel_of_descr sock, Unix.out_channel_of_descr sock)
      with 
          Unix.Unix_error(errcode, funcstr, paramstr) ->
            failwith ("LustreRun connect failure: " ^ (Unix.error_message errcode) ^
                         "(" ^ funcstr ^ " " ^ paramstr ^")\n")
    in
    let kill msg =
      Printf.printf "Killing the socket process (%s:%i)\n" sock_adr port;
      print_string ("'"^msg^"'");
      flush stdout;
      output_string sock_out msg;
      flush sock_out;
      let str = input_line sock_in in
      (* make sure that the sut process has read the quit before closing socks *)
      print_string (str ^"\n");
      flush stdout;
      Unix.shutdown sock Unix.SHUTDOWN_ALL
    in
    let label = Printf.sprintf "[%s:%i] " sock_adr port in
    let vars_in, vars_out = 
      if !debug then (
        Printf.fprintf stderr "\nWait for interface declarations on %s:%i.\n" sock_adr port; 
        flush stderr);
      RifIO.read_interface ~label:label sock_in (if !debug then Some stderr else None)
    in
    let step = step_channel sock_in sock_out vars_in vars_out in
    let step_dbg sl ctx cont =
      let enb = ctx.Event.nb in
      let ctx = Event.incr_event_nb ctx in
      {
        Event.step = ctx.Event.step;
        Event.data = ctx.Event.data;
        Event.depth = ctx.Event.depth;
        Event.nb = enb;
        Event.kind = Event.Exit;
        Event.lang = "socket";
        Event.name=sock_adr ^ ":" ^ (string_of_int port);
        Event.inputs = [] ;
        Event.outputs = [];
        Event.locals = [];
        Event.sinfo = None;
        Event.next = (fun () -> cont (step sl) ctx);
        Event.terminate = ctx.Event.terminate;
      } 
    in
    let plugin = {
      id = "";
      inputs = vars_in;
      outputs= vars_out;
      reset= (fun () -> RifIO.write sock_out "#reset\n";  flush sock_out);
      kill= kill;
      init_inputs= [];
      init_outputs= [];
      step = step;
      step_dbg = step_dbg
    }
    in
    if !debug then (
      Printf.fprintf stderr "\nInterface declarations on %s:%i, ok.\n" sock_adr port; 
      flush stderr
    );
    sock_in, plugin 

(* exported *)
let (make_socket : string -> int -> RdbgPlugin.t) =
  fun sock_adr port -> 
    let _, p = make_socket_do sock_adr port in
    p
(* exported *)
let (make_socket_init : string -> int -> RdbgPlugin.t) =
  fun sock_adr port -> 
    let sock_in, p = make_socket_do sock_adr port in
    let out_init = RifIO.read sock_in None p.outputs in
    let in_init = RifIO.read sock_in None p.inputs in
    { p with 
      init_inputs= in_init;
      init_outputs= out_init;
    }

(**********************************************************************************)
let (make_ec_exe : string -> RdbgPlugin.t) =
  fun ec_file -> 
    let exe = (Filename.chop_extension ec_file) ^ (Util.exe()) in
    let _ = if not (Sys.file_exists exe) then (
      Printf.printf "*** Error: Can not find the executable %s\n" exe;
      flush stdout;
      exit 2
    ) else (
      Printf.printf "The executable %s exist\n" exe;
      flush stdout
    )
    in
    let ec_in, ec_out = get_io_from_lustre ec_file None in
    let (ec_stdin_in,  ec_stdin_out) = Unix.pipe () in
    let (ec_stdout_in, ec_stdout_out) = Unix.pipe () in
      
    let ec_ic = Unix.in_channel_of_descr ec_stdout_in in
    let ec_oc = Unix.out_channel_of_descr ec_stdin_out in

    let _ = 
      set_binary_mode_in  ec_ic false;
      set_binary_mode_out ec_oc false
    in
    let pid_lustre = 
      let arg_list = [exe] in
      let arg_array = Array.of_list arg_list in
      let prog = List.hd arg_list in
        try 
          if !debug then ( 
	         List.iter (fun x -> output_string stderr (x ^ " ")) arg_list;
	         output_string stderr "\n"; 
            flush stderr
          );
          Unix.create_process prog arg_array 
            ec_stdin_in ec_stdout_out ec_stdout_out
        with Unix.Unix_error(e,_, prog) -> 
          let msg = Unix.error_message e in
            Printf.eprintf "*** Error when creating process with %s: %s\n" prog msg;
            exit 2
    in
    let _ = Printf.eprintf "Process %d (%s) created\n" pid_lustre exe; flush stderr in
    let kill msg =
      Unix.close ec_stdin_in;
      Unix.close ec_stdin_out;
      Unix.close ec_stdout_in;
      Unix.close ec_stdout_out;
      (try 
         Printf.eprintf "%s\nKilling process %d\n" msg pid_lustre;
         flush stderr;
         Unix.kill pid_lustre Sys.sigterm 
       with e -> (Printf.printf "Killing of %s process failed: %s\n" exe (Printexc.to_string e) ))
    in
    let step = step_channel ec_ic ec_oc ec_in ec_out in
    let step_dbg sl ctx cont =
      let enb = ctx.Event.nb in
      let ctx = Event.incr_event_nb ctx in
      {
        Event.step = ctx.Event.step;
        Event.data = ctx.Event.data;
        Event.nb = enb;
        Event.depth = ctx.Event.depth;
        Event.kind = Event.Exit;
        Event.lang = "ec";
        Event.name = ec_file;
        Event.inputs = [] ;
        Event.outputs = [];
        Event.locals = [];
        Event.sinfo = None;
        Event.next = (fun () -> cont (step sl) ctx);
        Event.terminate = ctx.Event.terminate;
      } 
    in
     {
      id = "";
      inputs = ec_in;
      outputs= ec_out;
      reset= (fun () -> RifIO.write ec_oc "#reset\n";  flush ec_oc);
      kill= kill;
      init_inputs= [];
      init_outputs= [];
      step = step;
      step_dbg = step_dbg
    }

(**********************************************************************************)

let (make_luciole : string -> vars -> vars -> 
      (string -> unit) * (Data.subst list -> Data.subst list)) =
  fun dro_file luciole_inputs luciole_outputs -> 
    if luciole_outputs <> ["Step",Data.Bool] || luciole_outputs <> [] then (
      Printf.eprintf "Inputs are missing. Try to generate them with luciole\n"; 
      Printf.eprintf "Luciole: generate lurette_luciole.c\n"
    ); 
    let luciole_outputs = List.map (fun (id,t) -> id, Data.type_to_string t) luciole_outputs in
    let luciole_inputs = List.map (fun (id,t) -> id, Data.type_to_string t) luciole_inputs in
    Luciole.gen_stubs "lurette" luciole_outputs luciole_inputs;
    Printf.eprintf "Luciole: generate lurette.dro from lurette_luciole.c\n"; 
    flush stderr;
    if Util2.c2dro "lurette_luciole.c" then () else 
      ( 
        Printf.eprintf "*** Lurette: Fail to generate lurette.dro for luciole! bye...\n"; 
        flush stderr;
        exit 2
      );    

    Printf.eprintf "\nluciole: launch simec_trap on lurette.dro\n"; 
    let (luciole_stdin_in,  luciole_stdin_out ) = Unix.pipe () in
    let (luciole_stdout_in, luciole_stdout_out) = Unix.pipe () in

    let luciole_ic = Unix.in_channel_of_descr  luciole_stdout_in in
    let luciole_oc = Unix.out_channel_of_descr luciole_stdin_out in
    let _ = 
      if Sys.os_type <> "Win32" then Unix.set_nonblock luciole_stdin_out;
      if Sys.os_type <> "Win32" then Unix.set_nonblock luciole_stdout_out;
      set_binary_mode_in  luciole_ic false;
      set_binary_mode_out luciole_oc false;
    in
    let prog = "simec_trap" ^ (if Sys.os_type="Win32" then ".bat" else "") in
    let args = [dro_file; string_of_int (Unix.getpid())] in
    let pid = 
      match Util.my_create_process 
        ~std_in:luciole_stdin_in ~std_out:luciole_stdout_out
        ~wait:false
        prog
        args
      with
        | Util.KO -> failwith ("error when calling simec_trap" ^ dro_file);
        | Util.OK -> assert false
        | Util.PID pid -> 
            debug_msg (prog ^ " " ^ dro_file ^ ": ok\n");
            pid
    in
    let kill msg = 
      close_out luciole_oc;
      close_in luciole_ic;
      (try 
         Printf.eprintf "%s\nKilling process %d\n" msg pid;
         flush stderr;
         Unix.kill pid Sys.sigterm 
       with e -> (Printf.printf "Killing of luciole process failed: %s\n" (Printexc.to_string e) ))
    in
    let (step : Data.subst list -> Data.subst list) = 
      fun sl -> 
        (* Sends values to luciole *)
        List.iter
          (fun (n,t) -> 
             let value = try List.assoc n sl with Not_found -> 
               let l = String.concat "," (List.map fst sl) in
                 if !debug then 
                   Printf.fprintf stdout "Reading luciole inputs:  %s not found in: %s ; " n l;
                 match t with
                     (* use fake value as luciole input are only displayed ; 
                        hence its not worth exiting when inputs are missing (at first step)
                     *)
                     "bool" -> Data.B(true)
                   | "int"  -> Data.I(42)
                   | "real" -> Data.F(42.0)
                   | _ -> 
                     Printf.fprintf stdout "*** cannot handle %s type as input of Luciole\n" t;
                     assert false
             in
             let val_str = (Data.val_to_string my_string_of_float value) ^"\n" in
               if !debug then
                 Printf.fprintf stdout "write_luciole_inputs: %s = %s\n" n val_str;
               output_string luciole_oc val_str)
          luciole_inputs;
        flush luciole_oc;

        debug_msg "Lurette: Start reading Luciole outputs...\n";
        (* Reads values from luciole *)
        let sl_out =
          List.map 
            (fun (name, vtype) -> 
               let str = 
                 debug_msg ("read_luciole_outputs: reading " ^name ^"\n");
                 let rstr = ref (input_line luciole_ic) in
                   debug_msg ("XXX: '" ^ !rstr ^ "'\n");
                   if (String.length !rstr >1 && String.sub !rstr 0 2 = "#q") then (
                     debug_msg ("luciole process has terminated \n");
                     failwith "luciole process has terminated"
                   );
                   while String.length !rstr = 0 || String.sub !rstr 0 1 = "#" do
                     debug_msg ("Skipping " ^ !rstr ^ "...\n");
                     rstr :=  input_line luciole_ic
                   done;
                   !rstr
               in
                 debug_msg ("read_luciole_outputs:"^ str^"\n");
                 let value = 
                   match vtype with
                     | "bool" -> 
                         if str = "t" then Data.B(true) else if str = "f" then Data.B(false) else (
                           output_msg2 ("read_luciole_outputs:Can not convert the value of "
                                        ^name^" into a bool:'"^str^"'\n");
                           exit 2
                         )
                     | "int" -> (
                         try Data.I(Util.my_int_of_string str)
                         with e ->  
                           output_msg2 ("read_luciole_outputs:Can not convert the value of "^
                                          name^" into an int:'"^str^"'\n"^
                                          (Printexc.to_string e));
                           exit 2
                       )
                     | "real" -> (
                         try Data.F(float_of_string str)
                         with e ->  
                           output_msg2 ("read_luciole_outputs:Can not convert  the value of "
                                        ^name^" into a float:'"^str^"'\n"^
                                          (Printexc.to_string e));
                           exit 2)
                     |  _ -> assert false
                 in
                   (name, value)
            )
            luciole_outputs
        in
          debug_msg "Lurette: read_luciole_outputs: done.\n";
          sl_out
    in
      kill, step

(**********************************************************************************)
let (make_dro : string -> vars * vars * 
      (string -> unit) * (Data.subst list -> Data.subst list)) =
  fun dro -> 
    assert false 
      (* finish me *)

(**********************************************************************************)
