(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
**
** File: lurettetop.ml
** Main author: jahier@imag.fr
*)

(** lurette toplevel loop. *)


open LtopArg

(***********************************************************************)
(* Parsing .lurette_rc and command line options *)


let main_read_arg () =
  let sut_dir = (Unix.getcwd ())  in
  let _ = args.sut_dir <- sut_dir in
  let lurette_rc = (Filename.concat args.sut_dir ".lurette_rc") in
  let _ =
    (* Read command in the .lurette_rc file *)
    (if Sys.file_exists lurette_rc then
       let ic = (open_in lurette_rc) in
	      try
	        while true do
             let str = input_line ic in
	          ignore (Cmd.read str)
	        done
	      with End_of_file ->
	        close_in ic
     else
       ()
    );
    let (explicit_the_luc_files : string -> unit) =
      fun s -> 
        if Filename.is_implicit s
        then args.env <- Filename.concat args.sut_dir s
        else args.env <- s;
        if not (Sys.file_exists args.env) then (
          Printf.printf "*** File %s does not exist!\n" s;
          flush stdout
        )
    in
      (* read the lurettetop command line options (that will override the
         .lurette_rc ones) *)
    let env_saved = args.env in
      args.env <- "";      
      ( try Arg.parse LtopArg.speclist explicit_the_luc_files usage
	     with
	         Failure(e) ->
	           output_string args.ocr e;
	           flush args.ocr ;
	           flush args.ecr ;
	           exit 2
	       | e ->
	           output_string args.ocr (Printexc.to_string e);
	           flush args.ocr;
	           exit 2
      );
      if args.sut_node = "" then 
	     args.sut_node <- (Util.chop_ext_no_excp (Filename.basename args.sut));
      if args.root_node = "" then args.root_node <- args.sut_node;
      if (args.env = "") then args.env <- env_saved;
  in  
  let lurette_tmp_dir = 
    match args.tmp_dir_provided with 
	     None -> 
          if args.direct_mode then "." else  Util.get_fresh_dir Sys.os_type 
      | Some file -> file  
  in 
  let _ = 
    args.tmp_dir <- lurette_tmp_dir; 
    Unix.putenv "TMPDIR" (String.escaped lurette_tmp_dir) ; 
  in 
  let source_dir = (Filename.concat (ExtTools.lurette_path()) "source") in     
    match args.sut_compiler with
      | Scade  -> assert false     
      | VerimagV4
      | VerimagV6
      | ScadeGUI
      | Sildex
      | Stdin
      | Ocamlopt -> ()
	      
let _ = main_read_arg ()

(***********************************************************************)
(* Socket administration.

Indeed,  xlurette calls lurettetop (as a client) via sockets
*)

let _ = match args.socket_port, args.socket_inet_addr with
  | None, None -> ()
  | None, _ -> failwith "*** --socket-port expected.\n"
  | _, None -> failwith "*** --socket-inet-addr expected.\n"
  | Some port, Some sock_inet_addr_str ->       
      let port_err = 
	     match args.socket_err_port with
	         None -> if not args.log then 
	           failwith "--log or --socket-err-port excpected.\n" else 0
	       | Some port_err -> port_err
      in
      let sock_addr = Unix.inet_addr_of_string sock_inet_addr_str in
      let sock_io =  Unix.socket Unix.PF_INET Unix.SOCK_STREAM  0 in
      let sock_err = Unix.socket Unix.PF_INET Unix.SOCK_STREAM  0 in

      (* loop to avoid the race between connect and accept *)
      let rec connect_loop s saddr p cpt =
	     try
	       Unix.connect s (Unix.ADDR_INET(saddr, p))
	     with Unix.Unix_error(errcode, funcstr, paramstr) ->
	       Unix.sleep 1; (* so that xlurette have the time to connect... Beurk! *)
	       if cpt = 0 then 
	         (
	           output_string args.ocr "ltop connect failure: ";
	           output_string args.ocr (Unix.error_message errcode);
	           flush args.ocr;
	           exit 2
	         )
	       else 
	         (
	           output_string args.ocr "ltop: retry connecting ...\n";
	           flush args.ocr;
	           connect_loop s saddr p (cpt-1)
	         )
      in
	     connect_loop sock_io sock_addr port 10;
        let ic = Unix.in_channel_of_descr sock_io in 
	     let oc =
          if args.log 
	       then 
	         open_out 
		        (Filename.concat args.tmp_dir 
		           (Filename.concat ".."
		              (Filename.concat ".." "lurette_stdout.log")))
	       else
	         Unix.out_channel_of_descr sock_io
	     in
	       if (args.verbose > 0) then (
	         output_string args.ocr ("connect stdout on port " ^ 
				                          (string_of_int port) ^ "\n");
	         flush args.ocr
	       );
	       args.icr <- ic;
	       args.ocr <- oc;
	       
	       flush oc;
	       let x = input_line args.icr in (* that one is blocking (hopefully) *)
	         if x = "hello." then ( ) else
	           (
		          output_string args.ocr (
		            "ltop: socket connection error.\n" ^ x ^ "<>hello.\n");
		          flush args.ocr;
		          exit 2
	           );	   	    
	         let ec = 
	           if args.log 
	           then 
		          open_out
		            (Filename.concat args.tmp_dir 
		               (Filename.concat ".."
		                  (Filename.concat ".." "lurette_stderr.log")))
	           else 
		          (
		            connect_loop sock_err sock_addr port_err 10; 
		            Unix.out_channel_of_descr sock_err 
		          )
	         in
	           if (args.verbose > 0) then 
		          (
		            output_string args.ecr (
		              "connect stderr on port " ^ (string_of_int port_err) ^ "\n"); 
		            flush args.ecr
		          );
	           args.ecr <- ec

(**************************************************************************)

let rec (main_loop : int -> unit) =
  fun cpt ->
    if args.socket_port = None then
      (output_string args.ocr
	      (match args.prompt with
	           None -> "<lurette " ^ (string_of_int cpt) ^ "> "
	         | Some prompt -> prompt
	      );
       flush args.ocr
      );
    let str = input_line args.icr in
    let continue = Cmd.read str in
      if continue then main_loop (cpt+1)

let lurettetop_quit msg () =
  let tmp_file = (Filename.temp_file "lurette" "") in
    if Filename.dirname tmp_file = Filename.dirname args.tmp_dir 
      && not (args.direct_mode)
    then
	   (* do not clean if the tmp dir is a user dir (--tmp-dir option) *) 
	   Util.rm_dir stdout args.tmp_dir;

    Sys.remove tmp_file ;    
    Unix.chdir args.sut_dir;
    flush args.ecr;
    output_string args.ecr (msg^"\nlurettetop: bye!\n");
    flush args.ocr
      

let main_loop_start () =
  output_string args.ocr ("This is Lurette Version "^(Version.str)^" (\""^Version.sha^"\") \n");
  flush args.ocr;
  if not (args.go) then (main_loop 1; Unix.chdir args.sut_dir) else
    (if args.direct_mode || Build.f args then (
	    Unix.chdir args.sut_dir;
       let res = Run.f () in
	      if (res) <> 0 then (
	        Printf.fprintf args.ocr "\nLurette launched a process that failed (exit %d).\n \n" res;
           flush args.ocr; 
           Sys.catch_break false;
	        exit res
	      );
     )
     else
	    (
	      output_string args.ocr "Can not build lurette, sorry\n \n \n";
	      flush args.ocr; 
	      exit 2
	    )
    )


let _ =  
  Sys.catch_break true;
  at_exit (lurettetop_quit "break signal catched\n"); 
  if args.verbose > 0 then output_string args.ocr "lurettetop: starting...\n";
  flush args.ocr;
  try  main_loop_start ()
  with e -> 
    print_string ("*** lurettetop: " ^ (Printexc.to_string e) ^ "\n");
    flush stdout;
    exit 2
;;

