(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License
**-----------------------------------------------------------------------
**
** File: lurette.ml
** Main author: jahier@imag.fr
*)

(*------------------------------------------------------------------------*)

open Util    
open Exp
open Var
open List
open Command_line
open Value
open Prog

(*------------------------------------------------------------------------*)


let (options:Command_line.optionsT) = {
  step_by_step = None ;
  display_local_var = false ;
  display_sim2chro = true ;
  user_seed = None ;
  verb = 0 ;
  show_step = false ;
  help = false ;
  output = "lurette.rif" ;
  draw_all_formula = false;
  draw_all_vertices = false;
  compute_volume = false;
  step_mode = Lucky.StepInside;
  luciole_mode = false;
  oracle = true ;
  pp = None;
  scade_gui = false;
  is_reactive = false;
  stdin = false;
  tmp_dir = "";
}

let itime = ref 0


(*------------------------------------------------------------------------*)

let output_msg msg =
  output_string stdout msg;
  flush stdout

let output_msg2 rif msg =
  let rif_msg = "#" ^ (Str.global_replace (Str.regexp "\n") "\n#") msg in
  output_string stdout msg;
  output_string rif (rif_msg);
  flush stdout
  


let debug_msg msg = if options.verb > 1 then output_msg msg else ()

let (print_vn_str : (string * string) list -> unit) =
  fun vl ->
    List.iter
      (fun (v, t) ->
	 output_string stdout ("\n\t\"" ^ v ^ "\"\t of type " ^ t ^ ",")
      )
      vl;
    output_string stdout "\n";
    flush stdout


    	
(*------------------------------------------------------------------------*)
	

(* Get the lists of var names and types. *)
let sut_i_vntl = Sut.get_input_var_name_and_type ()
let sut_o_vntl = Sut.get_output_var_name_and_type ()


let arg_nb = (Array.length Sys.argv)

(* Get the list of input and output var *)
let input_list_ref = fst (split sut_i_vntl)
let output_list_ref = fst (split sut_o_vntl)



(* I defined mine because i need to know the seed that has been drawn by self_init. *)
let random_seed () =
  let () = Random.self_init () in
    Random.int 10000000
      (* I cannot put max_int as glade (or gtk ?) seems to bug for too
	 big integers *)



(********************************************************************************)
let luciole_pid = ref 0

let lurette_exit i =
  if !luciole_pid <> 0 then (Unix.kill !luciole_pid Sys.sigkill);
  output_msg (Printf.sprintf "Try killing process  %d (simec).\n" !luciole_pid);
  if i<>0 then output_msg (Printf.sprintf "Lurette exit with code %d.\n" i);
  exit i

let print_failure i o oo l t locals rif =
  if (l <> Value.OfIdent.empty) then (
     output_msg "\n* environment local variables:\n" ;
     Value.OfIdent.print l stdout
  );

  output_msg "\n* sut input variables:\n" ;
  Value.OfIdent.print i stdout;
(*   print_subst_list i stderr; *)
  output_msg "\n* sut output variables:\n\t" ;
  Value.OfIdent.print  o stdout;
  output_msg "\n* oracle output variables:\n" ;
  Value.OfIdent.print oo stdout;
(*   print_subst_list l stderr; *)
  

  output_string rif "\n#oracle_failure at\n";
  Sim2chro.put_current_step_values
    rif
    t
    i
    o
    l
    options.display_local_var
    sut_o_vntl
    sut_i_vntl 
    locals;
  output_string rif  "\n# oracle output variables:\n" ;
  output_string rif (Value.OfIdent.to_string "#" oo)
  

let soi = string_of_int


(**************************************************************************)

let (type_of_string : string -> Type.t) =
  fun vt -> 
    match vt with
      | "float" -> Type.FloatT
      | "int" -> Type.IntT
      | "bool" -> Type.BoolT
      | _ -> failwith ("*** Bad type: '" ^ vt ^ "'. Only 'int', 'bool', and 'float' are allowed.\n")


let list_minus a b =  List.filter (fun v -> not (List.mem v b)) a

let (set_luciole_mode_if_necessary : Prog.state -> (Var.name * string) list ->
      (Var.name * string) list -> (Var.name * string) list ->
      (Var.name * string) list -> unit) =
  (* Use Luciole whenever
     - at least one Env input in missing (as sut outputs), or
     - at least one SUT input in missing (as env outputs),
  *)
  fun state in_sut out_sut in_oracle out_oracle ->

    let f v = Var.name v, Var.typ v in
    let out_env = List.map f state.s.out_vars 
    and in_env  = List.map f state.s.in_vars in
    let f (n,tstr) = n, type_of_string tstr in
    let out_sut = List.map f out_sut in
    let in_sut = List.map f in_sut in
    let out_luciole = (list_minus in_env out_sut)@(list_minus in_sut out_env) in
    
    let in_luciole = list_minus (out_env@out_sut) out_luciole in
    let _ =
      let missing = list_minus in_env out_sut in
      let missing2 = list_minus in_sut out_env in
      if missing <> [] then (
        let str = String.concat "," (List.map fst missing) in
        debug_msg ("Env inputs are missing ("^str^"): Luciole will generate them.\n"); 
        options.luciole_mode <- true);
      if missing2 <> [] then (
        let str = String.concat "," (List.map fst missing2) in
        debug_msg ("Sut inputs are missing ("^str^"): Luciole will generate them.\n"); 
        options.luciole_mode <- true);

      (* Some (useless) paranoiac checks *)
      if options.luciole_mode && (out_luciole@in_luciole) = [] then assert false;
      if not (list_is_included  in_luciole (out_env@out_sut)) then assert false;
                                                                   
(*       List.iter (fun (v,t)->  *)
(*                       if not(List.mem (v,t)  (out_luciole@in_luciole)) then *)
(*                         print_string (" Pb with " ^ v ^ "\n"); *)
(*                       ) in_sut; *)
(*       flush stdout;  *)
 
      if not (list_is_included  in_sut (out_env@out_luciole)) then (
        assert false
      );
      if options.verb > 1 then (
        let v_tostr (v,t) = v in
        let vl_tostr vl =  String.concat "," (List.map v_tostr vl) in
        print_string ("sut inputs : " ^ (vl_tostr in_sut) ^ "\n" );
        print_string ("sut outputs : " ^ (vl_tostr out_sut) ^ "\n" );
        print_string ("env inputs : " ^ (vl_tostr in_env) ^ "\n" );
        print_string ("env outputs : " ^ (vl_tostr out_env) ^ "\n" );
        print_string ("luciole inputs : " ^ (vl_tostr in_luciole) ^ "\n" );
        print_string ("luciole outputs : " ^ (vl_tostr out_luciole) ^ "\n" );
        flush stdout
      );

      (*  
          nb: I need to have "out_env U out_luciole  >  in_sut"
          where out_luciole = in_env \ out_sut

          nb : A > B <=> B \ A = {}

          hence it is equivalent to check that 
          in_sut \ (out_env U luciole_out) = {}
      *)
    in
    let missing = list_minus in_sut (out_env@out_luciole) in
      if missing <> []  then
        let  missing_str = String.concat ", " (List.map fst missing) in
          if List.length missing = 1 then 
            output_msg ("A SUT input is missing: "^ missing_str ^ "\n")
          else
            output_msg ("Some SUT inputs are missing: "^ missing_str ^ "\n");
          lurette_exit 1


let string_of_node n = n

let l_average = ref 0.0
let step_cpt = ref 0


let rec (test_manager : unit -> unit) =
  fun _ ->
    Array.iter (fun x -> output_string stderr (x ^ " ")) Sys.argv;
    output_string stderr "\n";
    flush stderr;
    try
      if
	     (arg_nb < 7)
      then
	     if
	       arg_nb >= 2 & ((    (Sys.argv.(1) = "--help")
			                     || (Sys.argv.(1) = "-help")
			                     || (Sys.argv.(1) = "-h")   ))
	     then
	       (
	         output_msg usage ;
	         lurette_exit 0
	       )
	     else
	       (
	         output_string stderr usage ;
	         lurette_exit 2
	       )
      else
	     let s = (cmd_line_string_to_int Sys.argv.(1)
		             ("*** int expected as first argument. " ^
		                Sys.argv.(1) ^ " is not an int.") )

	     and p = (cmd_line_string_to_int Sys.argv.(2)
		             ("*** int expected as third argument. " ^
		                Sys.argv.(2) ^ " is not an int.") )
	     and k1 = (cmd_line_string_to_int Sys.argv.(3)
		              ("*** int expected as third argument. " ^
		                 Sys.argv.(3) ^ " is not an int.") )
	     and k2 = (cmd_line_string_to_int Sys.argv.(4)
		              ("*** int expected as third argument. " ^
		                 Sys.argv.(4) ^ " is not an int.") )
	     and k3 = (cmd_line_string_to_int Sys.argv.(5)
		              ("*** int expected as third argument. " ^
		                 Sys.argv.(5) ^ " is not an int.") )
	     in
	       if 
	         options.stdin 
	       then
	         (
	           output_msg "\nNot yet implemented, sorry\n";
	           lurette_exit 0
	         )
	       else
	         match main2 s p k1 k2 k3 with
		          Some state -> 
		            if (options.step_by_step = None && options.display_sim2chro)
		            then Sim2chro.call_sim2chro state options.output;
	           | None -> lurette_exit 29
	               
    with
	     Failure(errmsg) ->
          (* 	  Env_state.dump_env_state_stat stdout; *)
	       output_msg errmsg;
	       lurette_exit 1
      | e ->
          (* 	  Env_state.dump_env_state_stat stdout; *)
	       output_msg (Printexc.to_string e);
	       lurette_exit 1

and
    (get_lurette_options: int -> int) =
  fun n ->
    try
      begin
	     let opt = List.assoc Sys.argv.(n) Command_line.string_to_option in
	       match opt with
	           Step ->
		          let str = (Sys.argv.(n+1)) in
		            options.step_by_step <- Some (cmd_line_string_to_int str
		                                            ("*** Error when calling lurette: an " ^
		                                               "integer is expected after the " ^
		                                               "option -step\n")) ;
		            n+2 
	         | NoStep -> options.step_by_step <- None ; (n+1)
		          
	         | DisplayLocalVar   -> options.display_local_var <- true ; (n+1)
	         | NoDisplayLocalVar -> options.display_local_var <- false ; (n+1)
		          
	         | Sim2chro  -> options.display_sim2chro <- true ; (n+1)
	         | NoSim2chro -> options.display_sim2chro <- false ; (n+1)

	         | StepInside -> options.step_mode <- Lucky.StepInside ; (n+1)
	         | StepEdges -> options.step_mode <- Lucky.StepEdges ; (n+1)
	         | StepVertices -> options.step_mode <- Lucky.StepVertices ; (n+1)

	         | Stdin -> options.stdin <- true ; (n+1)

	         | ScadeGui -> options.scade_gui <- true; (n+1)
	         | NoOracle -> options.oracle <- false ; (n+1)
	         | ShowStep  -> options.show_step <- true ; (n+1)
	         | AllFormula -> options.draw_all_formula <- true ; (n+1)
	         | AllVertices -> options.draw_all_vertices <- true ; (n+1)
	         | Reactive  -> options.is_reactive <- true ; (n+1)
	         | Help   -> options.help <- true ; (n+1)
	         | Output ->
		          let str = (Sys.argv.(n+1)) in
		            options.output <- str ;
		            n+2
	         | Verbose  -> 
		          let str = (Sys.argv.(n+1)) in
		            options.verb <- (cmd_line_string_to_int str
		                               ("*** Error when calling lurette: an " ^
		                                  "integer is expected after the " ^
		                                  "option --verbose\n")) ;
		            n+2
	         | Seed ->
		          let str = (Sys.argv.(n+1)) in
		            options.user_seed <- Some (cmd_line_string_to_int str
		                                         ("*** Error when calling lurette: an " ^
		                                            "integer is expected after the " ^
		                                            "option --with-seed\n")) ;
		            n+2
	         | Precision ->
		          let str = (Sys.argv.(n+1)) in
		            Util.precision := (cmd_line_string_to_int str
		                                 ("*** Error when calling lurette: an " ^
		                                    "integer is expected after the " ^
		                                    "option --precision\n")) ;
		            Util.update_eps ();
		            n+2

	         | PP  ->
		          let pp = (Sys.argv.(n+1)) in
		            if pp <> "" then options.pp <- Some pp ; 
		            (n+2)

	         | ComputeVolume ->
		          options.compute_volume <- true ; (n+1)

            | TmpDir -> 
                let tmp_dir = (Sys.argv.(n+1)) in
                  options.tmp_dir <- tmp_dir;
                  if not (Sys.file_exists tmp_dir) then (
	                 output_msg (tmp_dir ^ " does not exist\n");
	                 lurette_exit 2
	               );
                  (n+2)

      end
    with Not_found -> n
and
    (** Returns the environment file names given at top-level into a
        list of list.

        Also set the lurette command line options if any.
    *)
    (get_env_from_args: int -> string list -> string list) =
  fun n file_l ->
    if (n = arg_nb) then file_l
    else
      let m = get_lurette_options n in
	     (* m > n iff Sys.argv.(n) is an option *)
	     if
	       (m > n)
	     then
	       get_env_from_args m file_l
	     else
	       let arg = Sys.argv.(m) in
	         if
	           (arg = "x")
	         then
	           (* ignore x for backward compatibility *)
	           let env = Sys.argv.(n+1) in
                get_env_from_args (n+2) (env::file_l)
	         else
	           get_env_from_args (n+1) (arg::file_l)

and
    (main2 : int -> int -> int -> int -> int -> Prog.state option) =
  fun s p k1 k2 k3 -> 
    (* Clean up tables as non-reg assert stuff migth have filled them *)
    (
      let _ =
        Formula_to_bdd.clear_all ()
      in

      let env_list = (get_env_from_args 6 []) in

      (* XXX LUTIN *)

      let state0 = LucProg.make_state options.pp env_list None in
      let init_state_dyn = { state0.d with verbose = options.verb } in

      let init_state = {
        d = init_state_dyn ;
        s = state0.s
      }
      in

      let rif = open_out options.output in

      let local_var_name_and_type_list_unsorted0 =
        (* remove aliases *)
        fst (List.partition
	            (fun v -> (Var.alias v) = None) init_state.s.loc_vars)
      in
      let local_var_name_and_type_list_unsorted =
        List.map (fun v -> ((Var.name v), (Type.to_string (Var.typ v))))
	       local_var_name_and_type_list_unsorted0
      in
      let local_var_name_and_type_list =
        Util.sort_list_string_pair local_var_name_and_type_list_unsorted
      in
        set_luciole_mode_if_necessary init_state sut_i_vntl sut_o_vntl [] []; 
        Random.init seed ;
        output_msg ("\nThe random engine was initialized with the seed "^(soi seed)^"\n");

        (* Initialisation of the sut and the oracle *)
        Sut.init ();

        (* Sim2chro *)
        output_string rif ("# seed = " ^ (soi seed) ^ "\n");
        (match options.step_by_step with
	        | Some i -> 
	            step_cpt := i; (* so that it stops at the first step *)
	            if 
	              (init_state.s.gen_dot  init_state.d.ctrl_state []
		              ("environment" ^ (soi (Hashtbl.hash Sys.argv)))) = 0
	            then
	              Util.pdf ("environment" ^ (soi (Hashtbl.hash Sys.argv)) ^ ".pdf");
	            
	            (Sim2chro.put_var_decl
		            ("lurette chronogram -- " ^
		               (fold_left
		                  (fun acc str ->
			                  (acc ^ " " ^ str)) (hd env_list) (tl env_list))
		            )
		            sut_i_vntl
		            sut_o_vntl
		            local_var_name_and_type_list
		            stderr options.display_local_var);
	            flush stderr
	              
	        | None -> 
	            (
	              Sim2chro.put_var_decl
		             ("lurette chronogram (" ^
		                (fold_left (fun acc str -> (acc ^ " " ^ str)) "" env_list) ^ ")")
		             sut_i_vntl
		             sut_o_vntl
		             local_var_name_and_type_list
		             rif
		             options.display_local_var
	            )
        );
        (* Initializing Dd's libs. *)
        (*       Manager.disable_autodyn (Env_state.bdd_manager ()); *)
        
        (* selecting the draw mode *)
        if
	       options.compute_volume
        then
	       Solver.set_fair_mode ()
        else
	       Solver.set_efficient_mode ();
        
        (* Initializing the solution number table *)
        !Solver.init_snt ();
        itime := 0;

        (* Luciole communication channels *)
        let (luciole_ic,  luciole_oc) = 
          if options.luciole_mode then (
            let (luciole_stdin_in,  luciole_stdin_out ) = Unix.pipe () in
            let (luciole_stdout_in, luciole_stdout_out) = Unix.pipe () in

            let luciole_ic = Unix.in_channel_of_descr  luciole_stdout_in in
            let luciole_oc = Unix.out_channel_of_descr luciole_stdin_out in
            let _ = 
              Unix.set_nonblock luciole_stdin_out;
              Unix.set_nonblock luciole_stdout_out;
	           set_binary_mode_in  luciole_ic false;
	           set_binary_mode_out luciole_oc false;
            in
            let prog = "simec_trap" in
            let args = [Filename.concat options.tmp_dir "lurette.dro"; soi (Unix.getpid())] in
            let pid = 
	           match Util.my_create_process 
                ~std_in:luciole_stdin_in ~std_out:luciole_stdout_out
                ~wait:false
                prog
                args
              with
                | KO -> failwith "error when calling simec lurette.dro";
                | OK -> assert false
                | PID pid -> 
                    debug_msg ("simec ./lurette.dro: ok\n");
                    pid
            in
              luciole_pid := pid;
              luciole_ic, luciole_oc
          )
          else
            (* stub to avoid an option type *)
            (Unix.in_channel_of_descr Unix.stdin, Unix.out_channel_of_descr Unix.stdout)
        in
          (* 
             luciole I/O 
             luciole outs = env in \ sut outs U sut_in \ env_out 
             luciole in   = (env outs U sut outs) \ luciole outs     
          *)

          
        let sut_out = List.map 
          (fun (vn,t) -> 
             try List.find (fun var -> Var.name var = vn) init_state.s.in_vars
             with Not_found  -> 
               Var.make "" vn (type_of_string t) Var.Output) 
          sut_o_vntl 
        in
        let sut_in = List.map 
          (fun (vn,t) -> 
             try List.find (fun var -> Var.name var = vn) init_state.s.out_vars
             with Not_found  -> 
               Var.make "" vn (type_of_string t) Var.Input) 
          sut_i_vntl 
        in
        let luciole_inputs: Exp.var list  = init_state.s.out_vars @ sut_out in
        let luciole_outputs1: Exp.var list = list_minus init_state.s.in_vars sut_out in
        let luciole_outputs2: Exp.var list = list_minus sut_in init_state.s.out_vars in
        let luciole_outputs = luciole_outputs1@luciole_outputs2 in

        let compare_var_by_name v1 v2 = compare (Var.name v1) (Var.name v2) in 
        let luciole_inputs = list_minus luciole_inputs luciole_outputs in
        let luciole_outputs = List.sort compare_var_by_name luciole_outputs 
        and luciole_inputs  = List.sort compare_var_by_name luciole_inputs 
        in
        let _ = if options.luciole_mode then (
          debug_msg ("luciole_inputs:  "^ (String. concat "," (List.map Var.name luciole_inputs ))^"\n");
          debug_msg ("luciole_outputs: "^ (String. concat "," (List.map Var.name luciole_outputs))^"\n"))
        in
        let read_luciole_outputs tbl =
          debug_msg "Lurette: Start reading Luciole outputs...\n";
          let tbl = (
            List.fold_left 
              (fun acc var -> 
                 let str = 
                   debug_msg ("read_luciole_outputs: reading " ^(Var.name var) ^"\n");
                   let rstr = ref (input_line luciole_ic) in
                     while String.sub !rstr 0 1 = "#" do
                       debug_msg ("Skipping " ^ !rstr ^ "...\n");
                       rstr :=  input_line luciole_ic
                     done;
                     !rstr
                 in
                   debug_msg ("read_luciole_outputs:"^ str^"\n");
                   let value = 
                     match Var.typ var with
                       | Type.BoolT -> 
                           if str = "t" then Value.B(true) else if str = "f" then Value.B(false) else (
                             output_msg2 rif ("read_luciole_outputs:Can not convert the value of "
                                              ^(Var.name var)^" into a bool:'"^str^"'\n");
                             lurette_exit 2
                           )
                       | Type.IntT -> (
                           try Value.N(Value.I(Num.num_of_string str)) 
                           with _ ->  
                             output_msg2 rif ("read_luciole_outputs:Can not convert the value of "^
                                                (Var.name var)^" into an int:'"^str^"'\n");
                             lurette_exit 2
                         )
                       | Type.FloatT -> (
                           try Value.N(Value.F(float_of_string str))
                           with _ ->  
                             output_msg2 rif ("read_luciole_outputs:Can not convert  the value of "
                                              ^(Var.name var)^"into a float:'"^str^"'\n");
                             lurette_exit 2)
                       | Type.UT _ -> assert false
                   in
                     Value.OfIdent.add acc (Var.name var, value)
              )
              tbl
              luciole_outputs
	       ) in
            debug_msg "Lurette: read_luciole_outputs: done.\n";
            let outvals = List.fold_left 
  	           (fun acc var ->
	              let value = try Value.OfIdent.get tbl (Var.name var) with Not_found ->
	                output_msg2 rif ("Reading luciole outputs: the value of " ^ (Var.name var) ^ " is unknown.\n");
	                lurette_exit 2
	              in
	                Value.OfIdent.add acc (Var.name var, value))
	           Value.OfIdent.empty luciole_outputs
	         in
              (tbl, outvals)
        in
        let write_luciole_inputs sl =
          if options.luciole_mode then (
            List.iter
              (fun var -> 
                 (* let value = try List.assoc (Var.name var) sl with Not_found ->  *)
                 let value = try Value.OfIdent.get sl (Var.name var) with Not_found -> 
                   output_msg2 rif ("Reading luciole inputs: the value of " ^ (Var.name var) ^ " is unknown.\n");
                   lurette_exit 2
                 in
                 let val_str = (Value.to_string value) ^"\n" in
                   debug_msg ("write_luciole_inputs: "^ (Var.name var) ^ "= "^ val_str^"\n");
                   output_string luciole_oc val_str) 
              luciole_inputs;
            flush luciole_oc
          )
        in
        let input_init, luciole_outputs = 
          if options.luciole_mode then read_luciole_outputs Value.OfIdent.empty
	       else (Value.OfIdent.empty, Value.OfIdent.empty)
        in
        let close_me () = (
	       flush stdout;
	       flush rif;
          close_out luciole_oc;
          close_in luciole_ic;
	       close_out rif;
          debug_msg "Killing luciole process...\n";
          if options.luciole_mode then (Unix.kill !luciole_pid Sys.sigkill)
        ) in
	       try (
            let final_state =
              if not (options.help)
              then (
                try (
	               main_loop 1 s p k1 k2 k3 rif input_init local_var_name_and_type_list 
                    init_state write_luciole_inputs 
                    read_luciole_outputs luciole_outputs
                ) with Not_found -> assert false 
              ) else
                init_state
            in 
              close_me ();
 	           Some final_state
          ) with
	         | Not_found ->
                assert false
            | e -> 
	             (
		            close_me ();
                  Printf.fprintf stderr "EXCEP %s\n"  (Printexc.to_string e);
		            assert false
	             )
    )
and (main_loop :
       int -> int -> int -> int -> int -> int -> out_channel 
      -> Value.OfIdent.t
      -> (string * string) list
      -> Prog.state 
      -> (Value.OfIdent.t -> unit)
      -> (Value.OfIdent.t -> Value.OfIdent.t * Value.OfIdent.t)
      -> Value.OfIdent.t
      -> Prog.state 
    ) = fun
  t s p k1 k2 k3 rif input local_var_name_and_type_list state write_luciole_inputs 
  read_luciole_outputs luciole_outputs ->

    let ral = LucFGen.get input state in 
    let _ =
      if (k1 = 0 && k2 = 0 && k3 = 0 && not(options.draw_all_vertices)) then 
        () 
      else
        let num_thickness =
	       (k1, k2, 
	        if options.draw_all_vertices then Thickness.All else Thickness.AtMost k3)
        in
        let bool_thickness = options.draw_all_formula, p in
        let ((ral', outputs_loc): FGen.t list * (env_out * env_loc) list)
	         = 
	       try
	         Lucky.env_try (bool_thickness, num_thickness) input state ral
	       with 
            | Not_found -> assert false
            | FGen.NoMoreFormula -> 
                output_string stdout ("# " ^ (Prog.ctrl_state_to_string_long state.d.ctrl_state));
	             flush stdout;
                if state.s.is_final state.d.ctrl_state then 
	               lurette_exit  0
                else
	               lurette_exit 21
        in
	       
        (* Extracts the outputs and locals from the couple *)
        let (outputs, locals) = List.split outputs_loc in
          (* let outputs = List.map (fun o -> List.rev_append luciole_outputs o) outputs in  *)
        let outputs = List.map (Value.OfIdent.union luciole_outputs) outputs in 
          
        (* Tries the sut `n*p'^nth times *)
        let (inputs: env_in list) = List.map Sut.trie outputs in	
        let l = (List.length inputs) in
	       l_average := !l_average +. (float_of_int l)
	         
    in
      (* Performs the steps *)

    let (next_state, (output, loc)) = 
      try
        Lucky.env_step (options.step_mode) input state ral
      with FGen.NoMoreFormula -> ( 
        output_string stdout ("# : " ^ (Prog.ctrl_state_to_string_long state.d.ctrl_state));
        flush stdout;
        if state.s.is_final state.d.ctrl_state then 
	       lurette_exit  0
        else
          lurette_exit 23
      )
    in
    let output = Value.OfIdent.union luciole_outputs output in
    let sut_output = Sut.step output in

    let new_input, new_luciole_outputs = 
      if options.luciole_mode then (
        write_luciole_inputs (Value.OfIdent.union sut_output output);
        read_luciole_outputs sut_output
      ) else ( sut_output, Value.OfIdent.empty )
    in

    let _ =
      if options.show_step then
        (
	       let citime = int_of_float (Unix.time ()) in
	         if
	           (not options.scade_gui || citime <> !itime) 
	         then
	           (
	             itime := citime;
	             output_string stdout ("\n--- step " ^ (soi t) ^ ":\n");
	             flush stdout 
	           )
        );
      if state.d.verbose > 1 then (
        output_string stdout ("# : " ^ (Prog.ctrl_state_to_string_long state.d.ctrl_state));
        flush stdout
      )
    in
    let str =
      match (options.step_by_step) with
	       Some i -> 
	         let skip = (i - !step_cpt > 0) in
	           if (not skip 
		              (* 		&& state.d.ctrl_state <> next_state.d.ctrl_state *)
	              ) then
	             (
		            let _err_code = (state.s.gen_dot
				                         next_state.d.ctrl_state
		                               state.d.ctrl_state
				                         ("environment" ^ (soi (Hashtbl.hash Sys.argv)))
			                         )
		            in
		              ()
	             );
              (*HERE*) assert (new_input <> Value.OfIdent.empty);
	    Sim2chro.put_current_step_values
	      stdout
	      t
	      output
	      new_input
	      loc
	      options.display_local_var
	      sut_o_vntl
	      sut_i_vntl 
         local_var_name_and_type_list;
	    if 
	      skip 
	    then
	      (
		     incr(step_cpt); 
		     " "
	      )
	    else
	      (
		     step_cpt := 1;
		     output_string stdout
		       (*  ZZZ this string is matched in xlurette *)
		       "\nOne more loop ? [type 's' to stop, `CR' to continue, or an integer to change the number of steps to skip.]\n";
		     let str = read_line () in
		       try 
		         let i = int_of_string str in
		           options.step_by_step <- Some i;
		           str
		       with _ -> 
		         str
	      )
	     | None ->
	         (
              (*HERE*) assert (new_input <> Value.OfIdent.empty);
	           Sim2chro.put_current_step_values
	             rif t output new_input loc
	             options.display_local_var
	             sut_o_vntl sut_i_vntl  local_var_name_and_type_list;
	           ""
	         )
    in
      flush rif;
      (* Decides whether to loop once more *)
      if
        ((str <> "s") && (s > t))
          (*       ((str = "" || str = " ") && (s > t))   *)
      then
        main_loop (t+1) s p k1 k2 k3 rif new_input local_var_name_and_type_list next_state write_luciole_inputs
          read_luciole_outputs new_luciole_outputs
      else
        (
	       let exec_times = 
	         let ptime = Unix.times () in
	           ptime.Unix.tms_utime +. ptime.Unix.tms_stime
	       in
	       let time_msg = 
	         "      The execution lasted " ^ (my_string_of_float_precision exec_times 2) ^ 
	           " second"^ (if exec_times < 2.0 then ".\n" else "s.\n")
	       in			 
	         output_msg (
	           "      The Test Thickness average was " ^
	             (my_string_of_float_precision (1.0 +. !l_average /. (float_of_int t)) 1) ^ "\n");
	         output_msg ("The generated data can be found in the file " ^ 
		                    options.output ^ "\n");
	         output_msg2 rif time_msg;
            lurette_exit 0
        )



let lurette_main _ = 
    test_manager ()

(* So that the main can be called from C.

nb : i want the main to be in C in order to avoid to require an ocaml compiler...  *)
let _ =
  Callback.register "lurette_main" lurette_main
;;



(* To to able to use ocamldebug *)
(* let _ = lurette_main (); print_string "** WARNING : Debug mode\n";;   *)
