
let usage = "
usage: lurettetop [<options>]

lurettetop is a top level loop that let one use lurette.
Type help and/or man at the prompt for more info.

launch 'lurettetop --help' to see the available options.
"

let rp_help =" 
  To specify a reactive program to be used in the test session, one should
  use a string with the following format: \"machine_kind:language:file:node\" 
      where:
     - machine_kind can be : 'sut', 'oracle', or 'env'
     - language can be : 
        + 'v4'     to use the Lustre V4 programs
        + 'v6'     to use the Lustre V6 programs
        + 'lutin'  to use the Lutin programs
        + 'ocaml'  to use the Ocaml programs [ocaml]
        + 'ec'     to use the ec programs
        + 'ec_exe' to use a standalone executable obtained from an .ec file [ex_exe] 
     - file should be an existing file (compatible with the ''compiler'' field)
     - node should be a node of file (if meaningful) or empty

  [ocaml] In the 'ocaml' mode, the file can be an f.ml file, or a f.cmxs file.
  If an ml file is provided, lurettetop try to generate a cmxs file from it.
  If your caml program depends on library, or on other files, please generate
  the f.cmxs file by yourself (cf the ocaml documentation).

  [ec_exe] In the 'ec_exe' mode, lurette suppose that 'file.ec' has been compiled
  into an executable that is named 'file' (for instance, via ec2c -loop). 
  That executable must read/write its I/O using the RIF convention.
  The rationale for that mode is to be able to deal with Lustre programs that
  use C code. The 'file.ec' is just used to retrieve the I/O var names 
  and types actually.

  An alternative format is the following: \"machine_kind:socket:sock_addr:port\" where
     - machine_kind is as above
     - sock_addr is a valid internet adress
     - port is a free port on sock_addr

  The lurettetop process play the role of the client ; exchanges on the socket
  should respect the RIF (Reactive Input Format).

  Hence, to sum-up, we currently support:

  \"<sut|env|oracle>:lutin:prog:node\"        For lutin programs
  \"<sut|env|oracle>:v6:prog:node\"           For lustre V6 programs
  \"<sut|env|oracle>:v4:prog:node\"           For lustre V4 programs
  \"<sut|env|oracle>:ec:prog:\"               For lustre expanded code programs
  \"<sut|env|oracle>:ec_exe:prog:\"           For lustre expanded code programs that have been compiled
  \"<sut|env|oracle>:socket:addr:port\"       For reactive programs that read/write on a socket
  \"<sut|env|oracle>:socket_init:addr:port\"  Ditto + read I/O init values before the first step

    Examples: 
     \"sut:v6:controler.lus:main\"
     \"env:lutin:train.lut:tgv\" 
     \"oracle:socket:127.0.0.0:2042\" 
 
   If one needs to pass other options, one just need to add it 
   at the end of the rp, separating options by ':'. 
   
   For instance, if the train.lut requires an extern dynamic library
   libm.so, one would need to pass the option \"-L libm.so\" to the 
   Lutin interpreter. In order to do the same from lurettetop, 
   one would write:
    
      \"env:lutin:train:tgv:-L:libm.so\" 
"

(* compiler used to compiler sut and oracles *)
(* XXX obselete soon! *)
type compiler = VerimagV4 | VerimagV6 | Scade | ScadeGUI | Sildex | Stdin | Ocamlopt

type step_mode = | Inside | Edges | Vertices
let step_mode_to_string = function
  | Inside -> "--step-inside"
  | Edges -> "--step-edges"
  | Vertices -> "--step-vertices"

type verbose_level = int

type program_kind = SUT | Env | Oracle | PK_error of string


type reactive_program =
  | LustreV4 of string * string 
  | LustreV6 of string array
  | LustreEc of string
  | LustreEcExe of string
  | Lutin of string array
  | Socket of string * int
  | SocketInit of string * int
  | Ocaml of string 

let program_kind_of_string = function
  | "sut" -> SUT
  | "oracle" -> Oracle
  | "env" -> Env
  | s -> PK_error("*** Error: Unsupported kind of reactive program: \""^s ^"\"")

let program_kind_to_string = function
  | SUT -> "sut"
  | Oracle ->  "oracle"
  | Env -> "env"
  | PK_error msg -> "Error:" ^ msg

let reactive_program_to_string = function
  | LustreV4(f,node) -> "v4:"^f^":"^node
  | LustreV6(args) -> "v6:"^(String.concat ":" (Array.to_list args))
  | LustreEc(f) -> "ec:"^f^":"
  | LustreEcExe(f) -> "ec_exe:"^f^":"
  | Lutin(args)  -> "lutin:"^(String.concat ":" (Array.to_list args))
  | Socket(addr,port) -> "socket:"^addr^":"^(string_of_int port)
  | SocketInit(addr,port) -> "socket_init:"^addr^":"^(string_of_int port)
  | Ocaml(str) -> "ocaml:" ^ str
type t = {

  mutable suts : reactive_program list ;
  mutable envs : reactive_program list ;
  mutable oracles : reactive_program list ;


(* Obselete !!!! *)
  mutable sut_cmd : string ;
  mutable oracle_cmd : string ;
  mutable sut : string ;
  mutable sut_node : string ;
  mutable oracle : string option ;
  mutable oracle_node : string ;
  mutable env : string;
  mutable env_node : string ;
  mutable sut_compiler : compiler ;
  mutable oracle_compiler : compiler ;

  mutable step_nb : int;
  mutable draw_nb : int ;

  mutable draw_inside : int ;
  mutable draw_edges : int ;
  mutable draw_vertices : int ;
  mutable all_formula : bool ;
  mutable all_vertices : bool ;

  mutable step_mode : step_mode;
  mutable luciole_mode : bool;
  mutable delay_env_outputs : bool;

  mutable step_by_step : int option ; 
  mutable display_local_var : bool ;
  mutable display_sim2chro : bool;
  mutable display_gnuplot : bool;
  mutable seed : int option ;
  mutable precision : int ;
  mutable verbose : verbose_level ;
  mutable reactive : bool ;
  mutable show_step : bool ;
  mutable output : string ;
  mutable overwrite_output : bool;

  mutable make_opt : string ;
  mutable prompt : string option ;
  mutable extra_cfiles : string option ;
  mutable extra_libs : string option ;
  mutable extra_libdirs : string option ;
  mutable extra_includedirs : string option ;
  mutable go : bool ;
  mutable restore : string option;
(* this is a flag to know whether lurette needs to be (re-)build *)
  mutable prefix : string ;
  mutable sut_dir : string ;
  mutable compute_volume : bool;
  mutable pp : string option;
  mutable tmp_dir : string;
  mutable tmp_dir_provided : string option;
  mutable c_generator : string;

  mutable direct_mode : bool;
  mutable root_node : string; (* different from the sut_node sensible only for the scade gui *)
  mutable log : bool;
  mutable scade_gui : bool;
  mutable socket_inet_addr : string option; (* if None, we use stdin/stdout *)
  mutable socket_port : int option;
  mutable socket_err_port : int option;
  mutable debug_ltop : bool;
  mutable ldbg : bool;
  
  mutable cov_file : string;
  mutable reset_cov_file : bool;
  mutable stop_on_oracle_error : bool;

(* 
   I am using references for that in order to be able to replace them
   by sockets if necessary (i.e., once the sockets are connected) *)
  mutable icr : Pervasives.in_channel;
  mutable ocr : Pervasives.out_channel;
  mutable ecr : Pervasives.out_channel;
}


let (args : t) = {
  suts = [];
  oracles= [];
  envs = [];

  oracle_cmd = "" ;
  sut_cmd = "" ;
  sut = "" ;  
  sut_node = "" ;
  oracle = None ;
  oracle_node = "" ;
  env = "";
  env_node = "";
  sut_compiler = VerimagV6;
  oracle_compiler = VerimagV6;
  make_opt = "nc" ;
  step_nb = 10;
  draw_nb  = 1 ;
  draw_inside = 0 ;
  draw_edges = 0 ;
  draw_vertices = 0 ;
  all_formula = false ;
  all_vertices = false ;
  step_mode = Inside ;
  luciole_mode = false;
  delay_env_outputs = false;
  step_by_step = None ;
  show_step = false ;
  display_local_var = false ;
  display_sim2chro = true ;
  display_gnuplot = true ;
  seed = None ;
  precision = 2;
  verbose = 0 ;
  reactive = false ;
  output = "lurette.rif" ;
  overwrite_output = false;
  prompt = None ;
  extra_cfiles = None ;
  extra_libs = None ;
  extra_libdirs = None ;
  extra_includedirs = None ;
  go = false ;
  restore = None ;
  prefix = "";
  sut_dir = ".";
  compute_volume = false;
  pp = None;
  tmp_dir = ".";
  tmp_dir_provided = None;
  c_generator = "";

  direct_mode = true;

  root_node = "";
  log = false;
  scade_gui = false;
  socket_inet_addr = None;
  socket_port = None;
  socket_err_port = None;
  debug_ltop = false;
  ldbg = false;

  cov_file = "lurette.cov";
  reset_cov_file = false;
  stop_on_oracle_error = false;
 
  ocr =  stdout;
  icr =  stdin;
  ecr =  stderr;
}

(* exported 
   usefull to generete batch file and .lurette_rc files
*)
let (to_string : t -> (string * string * string) list) = 
  fun args -> 
    let rp_to_string rp kind = (" \"" ^kind^":" ^ (reactive_program_to_string rp) ^ "\"") in
    let soi = string_of_int in
    let bool_opt b x1 x2 = if b then (x1,x2,"") else "","","" in
    let quote str = "\"" ^ str ^"\"" in
      (List.map (fun rp -> "add_rp", "-rp", rp_to_string rp "sut") args.suts) @
      (List.map (fun rp -> "add_rp", "-rp", rp_to_string rp "env") args.envs) @
      (List.map (fun rp -> "add_rp", "-rp", rp_to_string rp "oracle") args.oracles) @
      [
        "stl ", "--test-length", soi args.step_nb;
        
        "set_draw_nb",           "--thick-draw", soi args.draw_nb;
        "set_draw_inside",       "--draw-inside", soi args.draw_inside;
        "set_draw_edges",        "--draw-edges", soi args.draw_edges;
        "set_draw_vertices",     "--draw-vertices", soi args.draw_vertices;

        bool_opt args.direct_mode  "set_direct_mode"  "--direct";
        bool_opt (not args.direct_mode)  "set_old_mode"  "--old-mode";
        bool_opt args.all_formula  "set_draw_all_formula"  "--draw-all-formula";
        bool_opt args.all_vertices "set_draw_all_vertices" "--draw-all-vertices";

        bool_opt args.display_local_var "set_display_local_var true" "--local-var";
        bool_opt (not args.display_sim2chro) "set_display_sim2chro" "--no-sim2chro";
        bool_opt (not args.display_gnuplot) "set_display_gnuplot" "--no-gnuplot";

        bool_opt args.log "log" "--log";
        bool_opt args.compute_volume "set_fair_mode" "--compute-poly-volume";

        bool_opt args.stop_on_oracle_error "stop_on_oracle_error true" "--stop-on-oracle-error";        
        bool_opt args.debug_ltop  "set_dbg_on"  "--dbg";
        bool_opt args.ldbg  "set_ldbg_on"  "--ldbg";


        "set_verbose", "--verbose", soi args.verbose ;
        "set_precision", "--precision" , soi args.precision;
        "set_rif", "--output", args.output;
        bool_opt args.overwrite_output "set_overwrite_output" "--overwrite-output" ;
        
        (match args.step_by_step with
           | None -> "set_step_by_step_off", "", ""
           | Some i -> "set_step_by_step", "--step", soi i);

        (match args.seed with
           | None -> "set_seed_randomly", "", ""
           | Some i -> "set_seed", "-seed", soi i);
        (match args.extra_cfiles with
           | None -> "","",""
           | Some str -> "set_extra_cfiles", "--extra-source-files", quote str) ;
        (match args.extra_libs with
           | None -> "","",""
           | Some str -> "set_extra_libs", "--extra-libs", quote str) ;
        (match args.extra_libdirs with
           | None -> "","",""
           | Some str -> "set_extra_libdirs", "--extra-libdirs", quote str) ;

        (match args.extra_includedirs with
           | None -> "","",""
           | Some str -> "set_extra_includedirs", "--extra-includirs", quote str) ;


        "set_cov_file", "--cov-file ", quote args.cov_file;

        (match args.step_mode with
             Inside  -> "set_step_mode", "--step-mode", "inside"
           | Edges  -> "set_step_mode", "--step-mode", "edges"
           | Vertices -> "set_step_mode", "--step-mode", "vertices"
        )

      ]

let gen_lurette_rc () =
  let oc = open_out ".lurette_rc" in
  let l = List.filter (fun (cmd,opt,arg) -> cmd<>"" || arg<>"" || opt <> "") (to_string args) in
  let l = List.map (fun (cmd, _, arg) -> cmd ^ " " ^ arg ^ "\n") l in
    List.iter (output_string oc) l;
    flush oc;
    output_string args.ecr ("'.lurette_rc' has been updated.\n");
    flush args.ecr;
    close_out oc

let gen_lurettetop_call args  =
  let l = List.filter (fun (cmd,opt,arg) -> cmd<>"" || arg<>"" || opt <> "") (to_string args) in
  let l = List.map (fun (_, opt, arg) -> opt ^ " " ^ arg ^ " ") l in
    String.concat "" l

let gen_batch file =
  let file = if file = "" then "lurette.batch" else file in
  let oc = open_out (Filename.concat args.sut_dir file) in
  let str = gen_lurettetop_call args in
    output_string oc "lurettetop -go ";
    output_string oc str;
    flush oc;
    output_string args.ecr ("The batch file " ^ file ^ " has been created.\n");
    flush args.ecr;
    close_out oc

let string_to_step_mode = function
  | "inside" -> Inside
  | "edges" -> Edges
  | "vertices" -> Vertices
  | _ ->
      output_string args.ocr "\n Warning: bad step mode. ";
      flush args.ocr;
      Inside


let (parse_rp_string : string -> unit) =
  fun str -> 
    (*     try *)
    let l = (Str.split (Str.regexp ":") str) in
    let rp_args = List.tl l in
    let rp = 
      match rp_args with
        | ["lutin";prog; node] -> 
            (* for backward compatibility, i add the 'main' in necessary... *)
            let rp_args = ("lutin"::prog::"-main"::node::[]) in
            let rp_args = 
              (match args.seed with
                   None ->  rp_args
                 | Some i -> rp_args@["-seed";string_of_int i])
            in
                Lutin(Array.of_list rp_args )
        | "lutin"::_ ->
            let rp_args = 
              (match args.seed with
                   None ->  rp_args
                 | Some i -> rp_args@["-seed";string_of_int i])
            in
                Lutin(Array.of_list rp_args )
                
        (*
          for lutin programs we accept:
          "lutin:toto.luc::"
          "lutin:toto.luc:toto:"
          "lutin:toto.luc:-main toto:"
        *)

        | "v6"::prog::node::opts -> 
            let args = ("lv6"::prog::"-node"::node::"--expand-io-type"::opts) in
            LustreV6(Array.of_list args)
        | ["ec_exe"; prog] -> LustreEcExe(prog)
        | ["ec"; prog] -> LustreEc(prog)
        | ["ec"; prog; _] -> LustreEc(prog)
        | ["v4"; prog; node] -> LustreV4(prog, node)
        | ["ocaml"; cmxs] -> Ocaml(cmxs)
        | ["socket"; addr; port] -> Socket(addr, int_of_string port)
        | ["socket_init"; addr; port] -> SocketInit(addr, int_of_string port)
        | _ -> failwith ("*** Error: Unsupported kind of reactive program: \""
                         ^ str ^ "\"\n" ^ rp_help)
    in
      match program_kind_of_string (List.hd l) with
        | SUT -> if not (List.mem rp args.suts) then args.suts <- rp::args.suts
        | Env -> if not (List.mem rp args.envs) then args.envs <- rp::args.envs
        | Oracle -> if not (List.mem rp args.oracles) then args.oracles <- rp::args.oracles
        | PK_error msg -> failwith msg
(*     with *)
(*         e  -> failwith ("error in --reactive-program: " ^ Printexc.to_string e *)
(*                        ) *)



let (string_to_compiler:string -> compiler option) =
  fun s ->
    match s with
      | "verimag" -> Some VerimagV4
      | "Verimag" -> Some VerimagV4
      | "lv4" -> Some VerimagV4
      | "v4" -> Some VerimagV4
      | "lv6" -> Some VerimagV6
      | "v6" -> Some VerimagV6
      | "scade-gui" -> Some ScadeGUI
      | "scade_gui" -> Some ScadeGUI
      | "scade" ->   Some Scade
      | "Scade" ->   Some Scade
      | "sildex" ->   Some Sildex
      | "Sildex" ->   Some Sildex
      | "stdin/stdout" -> Some Stdin
      | "stdin" -> Some Stdin
      | "Stdin" -> Some Stdin
      | "ocaml" -> Some Ocamlopt
      | _ -> None

let (compiler_to_string : compiler -> string) =
  fun c ->
    match c with
      | VerimagV4 -> "lv4"
      | VerimagV6 -> "lv6"
      | Scade -> "scade"
      | ScadeGUI -> "scade-gui"
      | Sildex -> "sildex"
      | Stdin -> "stdin"
      | Ocamlopt -> "ocaml"


(************************************************************************)

let (get_full_path: string -> string -> string) =
  fun dir str ->
    (* str is supposed to be a blank separated list of files. If those files
       have no path, explicitely append it the one of the user directory. *)
    let l0 = Util2.string_to_string_list str in
    let l1 =
      List.map
	(fun s -> if  (Filename.is_relative s) then (Filename.concat dir s) else s)
	l0
    in
      List.fold_left (fun acc s -> acc ^ s ^ " ") "" l1

let usage_out = Util.usage_out

(***********************************************************************)

let old_speclist = [
   "--sut", Arg.String
      (fun file -> args.sut <- file),
    "<string>\tFile name of the system under test [works with --old-mode only!].";
    "-sut", Arg.String
      (fun file -> args.sut <-  file),
    " <string>\n" ;

    "--sut-cmd", Arg.String
      (fun cmd -> args.sut_cmd <- cmd),
    "<string>\tCommand that launches the system under test [works with --old-mode only!].";
    "-sut-cmd", Arg.String
      (fun cmd -> args.sut_cmd <- cmd),
    " <string>\n" ;

    "--oracle-cmd", Arg.String
      (fun cmd -> args.oracle_cmd <- cmd),
    "<string>\tCommand that launches the oracle [works with --old-mode only!].";
    "-oracle-cmd", Arg.String
      (fun cmd -> args.oracle_cmd <- cmd),
    " <string>\n" ;

    "--main-sut-node", Arg.String
      (fun s -> args.sut_node <- s),
    "<string>\tMain node name of the system under test [works with --old-mode only!].";
    "-msn", Arg.String
      (fun s -> args.sut_node <- s),
    " <string>\n" ;

    "--main-env-node", Arg.String
      (fun s -> args.env_node <- s),
    "<string>\tMain node name of the environment (meaningful for lutin only) [works with --old-mode only!].";
    "-men", Arg.String
      (fun s -> args.env_node <- s),
    " <string>\n" ;

    "--oracle", Arg.String (fun file -> args.oracle <- 
			                     Some (file)),
    "<string>\tFile name of the oracle [works with --old-mode only!].";
    "-oracle", Arg.String (fun file -> args.oracle <- 
			                    Some (file)),
    " <string>\n";

    "--main-oracle-node", Arg.String
      (fun s -> args.oracle_node <- s),
    "<string>\tMain node name of the oracle [works with --old-mode only!].";
    "-man", Arg.String
      (fun s -> args.oracle_node <- s),
    " <string>\n" ;

    "--sut-compiler", Arg.String
      (fun s ->
	      match (string_to_compiler s) with
	          Some ScadeGUI -> 
	            args.sut_compiler <- ScadeGUI;
	            args.scade_gui <- true
	        | Some comp -> args.sut_compiler <- comp
	        | None ->
	            output_string args.ocr (s ^ " is not a supported compiler.\n");
	            flush args.ocr;
	            exit 2
      ),
    "<string> (lv4, lv6, scade)\t Compiler used for the sut [works with --old-mode only!].";

    "--oracle-compiler", Arg.String
      (fun s ->
	      match (string_to_compiler s) with
	          Some comp -> args.oracle_compiler <- comp
	        | None ->
	            output_string args.ocr (s ^ " is not a supported compiler.\n");
	            flush args.ocr;
	            exit 2
      ),
    "<string> (lv4, lv6, or scade)\t Compiler used for the oracle [works with --old-mode only!].";

    "--direct", Arg.Unit (fun () -> args.direct_mode <- true),
    "\tSet the direct mode.\n" ;
    "--old-mode", Arg.Unit (fun () -> args.direct_mode <- false),
    "\tUnset the direct mode.\n" 
]

let rec speclist =
  [
    "--reactive-program", Arg.String (fun str -> parse_rp_string str),
    "<string>.";    
    "-rp", Arg.String (fun str -> parse_rp_string str),
    ("<string> " ^ rp_help);

    "--cov-file", Arg.String (fun s -> args.cov_file <- s),
    "<string>\tfile name coverage info will be put into";

    "--reset-cov-file", Arg.Unit (fun _ -> args.reset_cov_file <- true),
    "";

    "--stop-on-oracle-error", Arg.Unit (fun _ -> args.stop_on_oracle_error <- true),
    "";

    "--test-length", Arg.Int (fun i -> args.step_nb <- i),
    "<int>\tNumber of steps to be done";
    "-l", Arg.Int (fun i -> args.step_nb <- i),
    ("<int>\t\t(currently, " ^ (string_of_int args.step_nb) ^ ").\n");

    (*     "--thick-form", Arg.Int (fun i -> args.formula_nb <- i), *)
    (*        "<int>\tNumber of formula to be drawn at each step"; *)
    (*     "-tf", Arg.Int (fun i -> args.formula_nb <- i),  *)
    (*     ("<int>\t\t(currently, " ^ (string_of_int args.formula_nb) ^ ").\n"); *)


    "--precision", Arg.Int (fun i -> args.precision <- i),
    "<int>\tnumber of digit after the dot used for floating points.\n" ;
    "-p", Arg.Int (fun i -> args.precision <- i), " <int>\n";


    "--fair", Arg.Unit (fun _ -> args.compute_volume <- true),
    ("\t\tCompute the polyhedra volumes before drawing: ");
    "--compute-poly-volume", Arg.Unit (fun _ ->  args.compute_volume <- true), 
    "more fair, but more expensive.\n";

    "--thick-draw", Arg.Int (fun i -> args.draw_nb <- i),
    "<int>\tNumber of draw to be done in each formula ";
    "-td", Arg.Int (fun i -> args.draw_nb <- i),
    ("<int>\t\tat each step (currently, " ^
       (string_of_int args.draw_nb) ^ ").\n");

    "--draw-inside", Arg.Int (fun i -> args.draw_inside <- i),
    "<int>\tDraw on the edges of the convex hull of solutions.";

    "--draw-edges", Arg.Int (fun i -> args.draw_edges <- i),
    "<int>\tDraw on the edges of the convex hull of solutions.";

    "--draw-vertices", Arg.Int (fun i -> args.draw_vertices <- i),
    "<int>\tDraw among the vertices of the convex hull of solutions.\n ";

    "--draw-all-formula", Arg.Unit (fun _ -> args.all_formula <- true),
    "\tTries all the formula reachable from the current state." ;
    "--draw-all-vertices", Arg.Unit (fun _ -> args.all_vertices <- true),
    "\tTries all the polyhedra vertices.\n" ;

    "--dbg", Arg.Unit (fun () -> args.debug_ltop <- true), " debug mode (to debug lurettetop)\n";
    "-ldbg", Arg.Unit (fun () -> args.ldbg <- true), " use the lurette debugger \n";

    (* This option is not meant to be available to end-users...  *)
    (*     "--make-opt", Arg.String (fun s -> args.make_opt <- s),  *)
    (*     ("<string>\tOptions to call make with when building \n" ^  *)
    (*      "\t\t\tlurette (currently, \"" ^ args.make_opt ^ "\").\n"); *)

    "--output", Arg.String (fun s -> args.output <- s),
    ("<string>\tSet the output file name (currently,  \"" ^
       args.output ^ "\").");
    "-o", Arg.String (fun s -> args.output <- s), "<string>\n";

    "--overwrite-output", Arg.Unit (fun () -> args.overwrite_output <- true),
    ("\tOverwrite \"" ^
       args.output ^ "\" if it exists without tring to invent a new name");
    "-oo", Arg.Unit (fun () -> args.overwrite_output <- true), "\n";

    "--batch", Arg.Unit (fun () -> args.go <- true),
    "\t\t\tStart the testing process directly, without prompting.";
    "--go", Arg.Unit (fun () -> args.go <- true), "\n";
    "-go", Arg.Unit (fun () -> args.go <- true), "\n";

    (*     "--restore", Arg.String (fun s -> args.restore <- Some s), *)
    (*        "<string>\tFile name of the package containing" *)
    (*        ^ "\n\t\t\tthe temporarily files to be restored (cf the pack command).\n"; *)

    "--step", Arg.Int (fun i -> args.step_by_step <- Some i), "\t\tRun lurette step by step." ;
    "-s", Arg.Int (fun i -> args.step_by_step <- Some i), "\n";

    "--socket-inet-addr", Arg.String (fun i -> args.socket_inet_addr <- Some i),
    "\t\tSet the socket address.\n" ;

    "--socket-io-port", Arg.Int (fun i -> args.socket_port <- Some i),
    "\t\tSet the socket io port.\n" ;

    "--socket-err-port", Arg.Int (fun i -> args.socket_err_port <- Some i),
    "\t\tSet the socket error port.\n" ;

    "--show-step", Arg.Unit (fun () -> args.show_step <- true),
    "\t\tSet on the show step mode.";

    "--do-not-show-step", Arg.Unit (fun () -> args.show_step <- false),
    "\tSet off the show step mode.\n";

    "--verbose", Arg.Int (fun i -> args.verbose <- i),
    "\t\tSet the verbose level.";
    "-v", Arg.Int( fun i -> args.verbose <- i),"\n";

    "--reactive", Arg.Unit (fun () -> args.reactive <- true),
    "\t\tSet on the reactive mode.";
    "-r", Arg.Unit (fun () -> args.reactive <- true),"\n";

    "--prompt", Arg.String (fun s -> args.prompt <- Some s),
    "\t\tReplace the default prompt.\n";

    "--extra-source-files", Arg.String
      (fun s ->
	      Unix.putenv "EXTRA_SOURCE_FILES" (String.escaped (get_full_path (Sys.getcwd ()) s));
	      args.extra_cfiles <- Some (get_full_path (Sys.getcwd ()) s)),
	 
    "\t\tSet the EXTRA_SOURCE_FILES environment variable.\n";

    "--extra-libs", Arg.String
      (fun s ->
	      Unix.putenv "EXTRA_LIBS" (String.escaped s);
	      args.extra_libs <- Some s),
    "\t\tSet the EXTRA_LIBS environment variable.\n";

    "--extra-libdirs", Arg.String
      (fun s ->
	      Unix.putenv "EXTRA_LIBDIRS" (String.escaped s);
	      args.extra_libdirs <- Some s),
    "\tSet the EXTRA_LIBDIRS environment variable.\n";

    "--extra-includedirs", Arg.String
      (fun s ->
	      Unix.putenv "EXTRA_INCLUDEDIRS" (String.escaped s);
	      args.extra_includedirs <- Some s),
    "\tSet the EXTRA_INCLUDEDIRS environment variable.\n";

    "--step-mode", Arg.String
      (fun s -> 
	      let m = 
	        match s with
	          | "Inside" -> Inside
	          | "inside" -> Inside
	          | "Edges" -> Edges
	          | "edges" -> Edges
	          | "Vertices" -> Vertices
	          | "vertices" -> Vertices
	          | _ -> 
		           output_string args.ocr (s ^ " is not a valid step mode.\n");
		           flush args.ocr;
		           exit 2
	      in
	        args.step_mode <- m
      ),
    "\t\tSet the step mode used to perform the step.\n";

    "--delay-env-outputs", Arg.Unit (fun _ -> args.delay_env_outputs <- true),
    "\t Delay the outputs of the environements before transmitting them to the oracles.";

    "--luciole", Arg.Unit (fun _ -> args.luciole_mode <- true),
    "\t Call lurette via luciole.";

    "--pre-processor", Arg.String (fun s -> args.pp <- Some s),
    "\tPre-processor for Lucky files (e.g., cpp).";
    "-pp", Arg.String (fun s -> args.pp <- Some s), "\n";


    "--prefix", Arg.String (fun s -> args.prefix <-  s),
    "\t\tA string to append before the call to lurette (e.g., \"/usr/bin/times \").\n";

    "--tmp-dir",  Arg.String 
      (fun s -> 
	      args.tmp_dir_provided <- Some s;
	      args.tmp_dir <- s;  
	      Unix.putenv "TMPDIR" s
      ),
    "\t\tUse that directory to put temporary files.\n";

    "--log",  Arg.Unit (fun _ -> args.log <- true),
    "\t\tRedirect stdout to a log file (lurette_stdout.log)\n";

    "--gnuplot", Arg.Unit (fun () -> args.display_gnuplot <- true),
    "\t\tCall gnuplot.";

    "--no-gnuplot", Arg.Unit (fun () -> args.display_gnuplot <- false),
    "\tDo not call gnuplot.";
    "-ngp", Arg.Unit (fun () -> args.display_gnuplot <- false), "\n";

    "--sim2chro", Arg.Unit (fun () -> args.display_sim2chro <- true),
    "\t\tCall sim2chro.";

    "--no-sim2chro", Arg.Unit (fun () -> args.display_sim2chro <- false),
    "\tDo not call sim2chro.";
    "-ns2c", Arg.Unit (fun () -> args.display_sim2chro <- false), "\n";

    "--local-var", Arg.Unit (fun () -> args.display_local_var <- true),
    "\t\tDisplay environment local variables in sim2chro (on).";

    "--no-local-var", Arg.Unit (fun () -> args.display_local_var <- false),
    "\tDo not display environment local variables in sim2chro.\n" ;


    "--ocaml-version", Arg.Unit (fun _ -> (print_string (Sys.ocaml_version^"\n") ; exit 0)),
    "\t\tDisplay the version ocaml version lurette was compiled with and exit." ;

    "--version", Arg.Unit (fun _ -> (print_string (Version.str^"-"^Version.sha) ; exit 0)),
    "\t\tDisplay the version and exit." ;
    "-version", Arg.Unit (fun _ -> (print_string (Version.str^"-"^Version.sha) ; exit 0)),
    "" ;
    "-v", Arg.Unit (fun _ -> (print_string (Version.str^"-"^Version.sha) ; exit 0)),
    "" ;


    "--help", Arg.Unit (fun _ -> (usage_out speclist usage ; exit 0)),
    "\t\tDisplay this list of options." ;
    "-help", Arg.Unit (fun _ -> (usage_out speclist usage ; exit 0)),
    "";
    "-h", Arg.Unit (fun _ -> (usage_out speclist usage ; exit 0)),
    ""
  ]

let (explicit_the_file : string -> string) =
  fun s -> 
    if Filename.is_implicit s
    then (Filename.concat args.sut_dir s)
    else s

 
