(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License
**-----------------------------------------------------------------------
**
** File: gen_stubs_scade.mli
** Author: jahier@imag.fr
**
*)

open Gen_stubs_common
open Type

(*
  Flatten structured types into several basic types.

  e.g: ("toto", ArrayT(3, int) is flattened into
  [("toto[0]", int, 1); ("toto[1]", int, 1); ("toto[2]", int, 1)]

  The int in third position denotes the original argument number.
*)
let rec (flatten_lucky_type : (string * Type.t) list ->
	   (string * Type.t * int) list) =
  fun vl ->
    snd(
      List.fold_left
	  (fun (i, acc) (vn, lt) -> (i+1, acc @ (flatten_var vn lt i)))
	  (0, [])
	  vl
    )
and
  (flatten_var : string -> Type.t -> int -> (string * Type.t * int) list) =
  fun vn lt i ->
    match lt with
	BoolT -> [(vn, lt, i)]
      | IntT -> [(vn, lt, i)]
      | FloatT -> [(vn, lt, i)]
      | UT(ut) -> flatten_structured_type vn ut i
and
  (flatten_structured_type : string -> Type.structured -> int ->
     (string * Type.t * int) list) =
    fun vn ut i ->
      match ut with
	| ArrayT(size, t) ->
	    let l = ref [] in
	      for j=0 to size-1 do
		l := !l @ (flatten_var (vn ^ "[" ^ (string_of_int j) ^ "]") t i)
	      done;
	      !l
	| StructT(fl) ->
	    (List.fold_left
	       (fun acc (fn, ft) -> acc @ (flatten_var (vn ^ "." ^ fn) ft i))
	       []
	       fl)
	| EnumT(el) ->
	    (* TODO *)
	    assert false

(* exported *)
let (go: module_name -> string -> typedef list -> vn_ct list -> vn_ct list -> string -> 
      unit) =
  fun mod_name str tdl vi0 vo0 header_file0 ->
    let header_file = Filename.basename header_file0 in
    let oc = open_out (str ^ ".c.new") in
    let put s = output_string oc s in
    let mput s = if mod_name <> "" then output_string oc s in

    let vi1 = List.map (fun (vn, ct) -> (vn, c_type_to_lucky_type tdl ct)) vi0
    and vo1 = List.map (fun (vn, ct) -> (vn, c_type_to_lucky_type tdl ct)) vo0
    in

    let vi = flatten_lucky_type vi1
    and vo = flatten_lucky_type vo1
    in
      (*
       ** Compiler directive
       *)
    let _ =
      put ("// Automatically generated from " ^ mod_name ^
	          ".h by bin/gen_stubs (scade).\n" ^
	          "#include <stdlib.h>\n" ^
	          "#include <stdio.h>\n" ^
	          "#include \"ocaml2c.h\"\n" ^
	          "#include \"config_types.h\"\n");
      mput ("#include \"" ^ header_file ^ "\" \n");
      put   " \n";

      (*
       ** type definition of values
       *)
      put "typedef int boolean;\n" ;
      put "\n" ;


      (*
       ** variable declarations
       *)


      mput ("static _C_" ^ mod_name ^ " Ctx; \n") ;
      mput ("static _C_" ^ mod_name ^ " Ctxcopy; \n") ;

      mput ("static _C_" ^ mod_name ^ " *_C_ = &Ctx; \n") ;
      mput ("static _C_" ^ mod_name ^ " *_C_copy = &Ctxcopy; \n") ;

      mput "\n" ;


      (*
       ** Program state initialisation
       *)
      put "// Program state initialisation \n" ;
      put ("void " ^ str ^ "_init() \n{\n");
      mput ( "  " ^ mod_name ^ "_init(_C_); \n") ;
      mput ("  memcpy(_C_copy, _C_, sizeof(_C_" ^ mod_name ^ ")); ") ;
      put "\n}\n\n" ;

      put "\n" ;


      (*
       ** Save and restore the state  
       *)

      put " // Save and restore the state  \n";
      put ("void " ^ str ^ "_save_state ()\n{\n");
      mput ("  memcpy(_C_copy, _C_, sizeof(_C_" ^ mod_name ^ "));\n") ;
      put "}\n\n";

      put ("void " ^ str ^ "_restore_state ()\n{\n");
      mput ("  memcpy(_C_, _C_copy, sizeof(_C_" ^ mod_name ^ "));\n") ;
      put "}\n\n";


      (*
       ** Set and get int values from caml
       *)
      put "\n" ;
      put "// set int values \n" ;
      put ("void " ^ str ^ "_set_val_int(int arg_nb, int vali)\n{\n");
      put "  switch (arg_nb){\n";
    in
    let _ =
      List.fold_left
	     (fun j (v, lt, i) ->
	        (match lt with
		          IntT ->
		            put ("  case " ^ (string_of_int j) ^ ":\n    ") ;
		            put ("_C_->_I" ^ (string_of_int i) ^ "_" ^ v ^ " = vali;\n");
		            put "    break;\n";
	           | FloatT -> ()
	           | BoolT -> ()
	           | UT xxx ->
		            assert false
	        );
	        (j+1)
	     )
	     0
	     vi
    in
    let _ =
      put "  default : \n    printf(\"Unexpected type in set_val_int.\");\n";
      put "    printf(\" The %i nth input arg is not an integer \", arg_nb);\n";
      put "    exit(2);\n  }\n";
      put "}\n" ;



      put "\n" ;
      put "// get int values \n" ;
      put ("int " ^ str ^ "_get_val_int(int arg_nb)\n{\n");
      put "  switch (arg_nb){\n";
    in
    let _ =
      List.fold_left
	     (fun j (v, lt, i) ->
	        (match lt with
		          IntT ->
		            put ("\n    case " ^ (string_of_int j) ^
		                   ": return _C_->_O" ^ (string_of_int i) ^ "_"^ v ^ ";\n");
		            put "    break;\n";
	           | FloatT -> ()
	           | BoolT -> ()
	           | UT xxx ->
		            assert false
	        );
	        (j+1)
	     )
	     0
	     vo
    in
    let _ =
      put "  default :\n    printf(\"Unexpected type in get_val_int\");\n";
      put "    printf(\". The %i nth output arg is not an integer \", arg_nb);\n";
      put "    exit(2);\n  }\n";
      put "}\n" ;

      (*
       ** Set and get float values from caml
       *)
      put "\n" ;
      put "// set float values \n" ;
      put ("void " ^ str ^ "_set_val_float(int arg_nb, double valf)\n{\n");
      put "  switch (arg_nb){\n";
    in
    let _ =
      List.fold_left
	     (fun j (v, lt, i) ->
	        (match lt with
		          IntT -> ()
	           | FloatT ->
		            put ("\n    case " ^ (string_of_int j) ^ ": ") ;
		            put ("_C_->_I" ^ (string_of_int i) ^ "_" ^ v ^ " = ((real) valf);\n");
		            put "    break;\n";
	           | BoolT -> ()
	           | UT xxx ->
		            assert false
	        );
	        (j+1)
	     )
	     0
	     vi
    in
    let _ =
      put "  default : \n    printf(\"Unexpected type in set_val_float\");\n";
      put "    printf(\". The %i nth input arg is not a float \", arg_nb);\n";
      put "    exit(2);\n  }\n";
      put "}\n" ;



      put "\n" ;
      put "// get float values \n" ;
      put ("double " ^ str ^ "_get_val_float(int arg_nb)\n{\n");
      put "  switch (arg_nb){\n";
    in
    let _ =
      List.fold_left
	     (fun j (v, lt, i) ->
	        (match lt with
		          IntT -> ()
	           | FloatT ->
		            put ("\n    case " ^ (string_of_int j) ^
		                   ": return ((real) _C_->_O" ^ (string_of_int i) ^ "_"^ v ^ ");\n");
		            put "    break;\n";
	           | BoolT -> ()
	           | UT xxx ->
		            assert false
	        );
	        (j+1)
	     )
	     0
	     vo
    in
    let _ =
      put "  default :\n    printf(\"Unexpected type in get_val_float\");\n";
      put "    printf(\". The %i nth output arg is not a float \", arg_nb);\n";
      put "    exit(2);\n  }\n";
      put "}\n" ;

      (*
       ** Set and get bool values from caml
       *)
      put "\n" ;
      put "// set bool values \n" ;
      put ("void " ^ str ^ "_set_val_bool(int arg_nb, boolean valb)\n{\n");
      put "  switch (arg_nb){\n";
    in
    let _ =
      List.fold_left
	     (fun j (v, lt, i) ->
	        (match lt with
		          IntT -> ()
	           | FloatT -> ()
	           | BoolT ->
		            put ("\n    case " ^ (string_of_int j) ^ ": ") ;
		            put ("_C_->_I" ^ (string_of_int i) ^ "_" ^ v ^ " = valb;\n");
		            put "    break;\n";
	           | UT xxx ->
		            assert false
	        );
	        (j+1)
	     )
	     0
	     vi
    in
    let _ =
      put "  default : \n    printf(\"Unexpected type in set_val_bool\");\n";
      put "    printf(\". The %i nth input arg is not a Boolean \", arg_nb);\n";
      put "    exit(2);\n  }\n";
      put "}\n" ;



      put "\n" ;
      put "// get bool values \n" ;
      put ("boolean " ^ str ^ "_get_val_bool(int arg_nb)\n{\n");
      put "  switch (arg_nb){\n";
    in
    let _ =
      List.fold_left
	     (fun j (v, lt, i) ->
	        (match lt with
		          IntT -> ()
	           | FloatT -> ()
	           | BoolT ->
		            put ("\n    case " ^ (string_of_int j) ^
		                   ": return _C_->_O" ^ (string_of_int i) ^ "_"^ v ^ ";\n");
		            put "    break;\n";
	           | UT xxx ->
		            assert false
	        );
	        (j+1)
	     )
	     0
	     vo
    in
    let _ =
      put "  default :\n    printf(\"Unexpected type in get_val_bool\");\n";
      put "    printf(\". The %i nth output arg is not a Boolean \", arg_nb);\n";
      put "    exit(2);\n  }\n";
      put "}\n" ;



      (*
       ** Step
       *)
      put "\n" ;
      put "// Step \n" ;
      (* Function header *)
      put ("void " ^ str ^ "_step(void) \n{\n");
      mput ("  " ^ mod_name ^ "(_C_);\n") ;

      put "}\n" ;


      (*
       ** Variable number
       *)

      put "\n\n// The 2 following functions return variable numbers \n";
      put ("int " ^ str ^ "_input_arg_nb(void) \n{\n") ;
      put ("   return " ^ (string_of_int (List.length vi)) ^ ";\n");
      put "}\n\n" ;

      put ("int " ^ str ^ "_output_arg_nb(void) \n{\n") ;
      put ("   return " ^ (string_of_int (List.length vo)) ^ ";\n");
      put "}\n\n" ;



      (*
       ** Variable names and types lists
       *)
      put "\n\n// The 2 following functions return the list of inputs \n";
      put "// (resp outputs) variable names and types. \n";

      put ("void " ^ str ^ "_input_var_name_and_type_array(") ;
      put ("int n, vnt_type vnta[" ^ (string_of_int (List.length vi)) ^ "])\n{\n");
    in
    let _ =
      List.fold_left
	     (fun i (vn, t, _) ->
	        put ("  vnta[" ^ (string_of_int i) ^ "].var_name = \"" ^ vn ^ "\";\n");
	        put ("  vnta[" ^ (string_of_int i) ^ "].var_type = \"" ^
		            (Type.to_string t) ^ "\";\n");
	        (i+1)
	     )
	     0
	     vi
    in
    let _ =
      put "}\n\n" ;

      put ("void " ^ str ^ "_output_var_name_and_type_array(") ;
      put ("int n, vnt_type vnta[" ^ (string_of_int (List.length vo)) ^ "])\n{\n");
    in
    let _ =
      List.fold_left
	     (fun i (vn, t, _) ->
	        put ("  vnta[" ^ (string_of_int i) ^ "].var_name = \"" ^ vn ^ "\";\n");
	        put ("  vnta[" ^ (string_of_int i) ^ "].var_type = \"" ^
		            (Type.to_string t) ^ "\";\n");
	        (i+1)
	     )
	     0
	     vo
    in
      put "}\n\n" ;
	   
      close_out oc




