(*pp camlp4o *)
(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
** This file may only be copied under the terms of the GNU Library General
** Public License
**-----------------------------------------------------------------------
**
** File: gen_stubs_common.ml
** Main author: jahier@imag.fr
*)


open Type

(* Exported *)
type file = string
type module_name = string
type var_name = string
type fresh_var_name = string


type c_type =
    Simple of string
  | Array of int * c_type
  | Struct of (string * c_type) list
  | Enum of string list
type typedef = string * c_type

			
(* Exported *)
type vn_ct = var_name * c_type
type alias = c_type * c_type

type vn_ct_mlt_fvn = var_name * c_type * Type.t * fresh_var_name


(****************************************************************************)
(* exported *)
let rec (ctype_to_string: c_type -> string) =
  fun ct ->
    match ct with
	Simple(ident) -> ident
      | Array(size, ct) ->
	  ((ctype_to_string ct) ^ " [" ^ (string_of_int size) ^ "]") 
      | Struct sl ->
	  (List.fold_left
	     (fun acc (f,ct') -> acc ^ "\n\t" ^ f ^ ":" ^ (ctype_to_string ct'))
	     ""
	     sl)
      | Enum el ->
	  (List.fold_left
	     (fun acc (e) -> acc ^ "\n\t" ^ e)
	     ""
	     el)



let print_typedef (name, t) =
  print_string (name ^ " : " ^ (ctype_to_string t) ^ "\n");
  flush stdout


(************************************************************************)
(* compiler used to compile sut and oracles *)
type compiler = VerimagV4 | VerimagV6 | Scade | ScadeGUI | Sildex

let (string_to_compiler:string -> compiler) =
  fun s ->
    match s with
      | "verimag" -> VerimagV4
      | "Verimag" -> VerimagV4
      | "lv4" -> VerimagV4
      | "v4" -> VerimagV4
      | "lv6" -> VerimagV6
      | "v6" -> VerimagV6
      | "scade-gui" -> ScadeGUI
      | "scade" ->   Scade
      | "Scade" ->   Scade
      | "sildex" ->   Sildex
      | "Sildex" ->   Sildex
      | other ->
	  print_string ("The compiler '" ^ other ^ "' is not supported sorry.\n");
	  exit 2

let (compiler_to_string : compiler -> string) =
  fun c ->
    match c with
      | VerimagV4 -> "lv4"
      | VerimagV6 -> "lv6"
      | Scade -> "scade"
      | ScadeGUI -> "scade-gui"
      | Sildex -> "sildex"

(************************************************************************)
(**
  [generate_n_var_names "x" 4] generates the list ["x1"; "x2"; "x3"; "x4"]
*)
let rec (generate_n_var_names: string -> int -> fresh_var_name list) =
  fun x n ->
    match n with
	0 -> []
      | _ -> (List.append (generate_n_var_names x (n-1))
		[(x ^ (string_of_int n))])

let _ = assert ((generate_n_var_names "x" 0) = [])
let _ = assert ((generate_n_var_names "x" 4) = ["x1"; "x2"; "x3"; "x4"])



(* XXX should i use the typedef list here also ??? *)
let rec (lucky_type_to_c_type : Type.t -> c_type) =
  fun t -> 
    match t with
	IntT   -> Simple "int"
      | FloatT -> Simple "float"
      | BoolT  -> Simple "bool"
      |	UT(EnumT enum_list) ->  Enum enum_list
      | UT(ArrayT(size, lt)) -> Array (size, lucky_type_to_c_type lt)
      | UT(StructT field_lucky_type_list) ->
	  let field_c_type_list =
	    List.map
	      (fun (f, lt) -> (f, lucky_type_to_c_type lt))
	      field_lucky_type_list
	  in
	    Struct field_c_type_list

  
let rec (c_type_to_lucky_type : typedef list -> c_type -> Type.t) =
  fun tdl ct ->
    (**  *)
    match ct with
      |	Simple "byte" -> IntT
      | Simple "short" -> IntT
      | Simple "short int" -> IntT
      | Simple "int" -> IntT
      | Simple "_int" -> IntT
      | Simple "long" -> IntT
      | Simple "long int" -> IntT

      | Simple "unsigned short" -> IntT
      | Simple "unsigned short int" -> IntT
      | Simple "unsigned int" -> IntT
      | Simple "unsigned long" -> IntT
      | Simple "unsigned long int" -> IntT

      | Simple "float" -> FloatT
      | Simple "real" -> FloatT
      | Simple "double" -> FloatT
      | Simple "long double" -> FloatT

      | Simple "bool" -> BoolT
      | Simple "_bool" -> BoolT
      | Simple "_boolean" -> BoolT
      | Simple "boolean" -> BoolT
      | Simple "lurette__boolean" -> BoolT

      | Simple str ->
	       ( try
	           let ct_def = List.assoc str tdl in
		          c_type_to_lucky_type tdl ct_def (* recursive call *)
	         with
		          Not_found ->
		            print_string ("\""^str ^ "\": Unsupported type.");
		            flush stdout;
		            exit 2
	       )
      | Array (size, c_type) ->
	       UT(ArrayT(size, c_type_to_lucky_type tdl c_type))
      | Struct field_ctype_list ->
	       let field_lucky_type_list =
	         List.map
	           (fun (f, ct) -> (f, c_type_to_lucky_type tdl ct))
	           field_ctype_list
	       in
	         UT(StructT field_lucky_type_list)
      | Enum enum_list ->
	       UT(EnumT enum_list)

let rec (c_type_to_scade_type : typedef list -> c_type -> string) =
  fun tdl ct ->
    (**  *)
    match ct with
      |	Simple "char" -> "char"

      |	Simple "byte" -> "int"
      | Simple "short" -> "int"
      | Simple "short int" -> "int"
      | Simple "int" -> "int"
      | Simple "_int" -> "int"
      | Simple "long" -> "int"
      | Simple "long int" -> "int"

      | Simple "unsigned short" -> "int"
      | Simple "unsigned short int" -> "int"
      | Simple "unsigned int" -> "int"
      | Simple "unsigned long" -> "int"
      | Simple "unsigned long int" -> "int"

      | Simple "float" -> "real"
      | Simple "real" -> "real"
      | Simple "double" -> "real"
      | Simple "long double" -> "real"

      | Simple "bool" -> "bool"
      | Simple "_bool" -> "bool"
      | Simple "_boolean" -> "bool"
      | Simple "boolean" -> "bool"
      | Simple "lurette__boolean" -> "bool"

      | Simple str ->
	       ( try
	           let ct_def = List.assoc str tdl in
		          c_type_to_scade_type tdl ct_def
	         with
		          Not_found ->
		            print_string ("Warning: \""^str ^ "\" is an unknown type.\n");
		            flush stdout;
		            str
	       )
      | Array (size, c_type) ->
	       ((c_type_to_scade_type tdl c_type) ^ "^" ^ (string_of_int size))

      | Struct field_ctype_list ->
	       let f str acc (field, ct) =
	         (acc ^ str ^ field ^ ":" ^ (c_type_to_scade_type tdl ct))
	       in
	       let str =
	         List.fold_left
	           (f ",")
	           (f "" "" (List.hd field_ctype_list))
	           (List.tl field_ctype_list)
	       in
	         ("[" ^ str ^ "]")
      | Enum enum_list ->
	       ( "(" ^
	           (List.fold_left
	              (fun acc e -> acc ^ "," ^ e)
	              (List.hd enum_list)
	              (List.tl enum_list)) ^
	           ")")


	

(****************************************************************************)
(****************************************************************************)
(**
 [get_typedef file] reads [file] (a C header file) and searches
 for typedef C expressions and returns the list of (alias_type, C_type)
 found in [file].
*)

(* XXX Mettre dans un autre module ?? *)

open Lexing
open MyGenlex


let lexer = make_lexer [
  "long"; "short"; "int"; "signed"; "unsigned";
  "float"; "double"; "char";
  "."; ","; "{"; "}"; ";"; ":"; "("; ")"; "["; "]";
  "/*"; "*/"; "#"; "*" ]

(****************************)


(* exported *)
(* let debug = true  *)
let debug = false
let debug2 = false


(* exported *)
let (print_genlex_token : token -> unit) =
  fun tok ->
    match tok with
	     Kwd((s, e), str)    ->
          (* 	  print_int s; print_string ":"; print_int e; *)
	       print_string ("Kwd`" ^ str ^ "' ")
      | Ident((s, e), str)  ->
	       print_string ("Ident`" ^ str ^ "' ")
      | Int((s, e), i)      ->
	       print_string "Int`"; print_int i; print_string "' "
      | Float((s, e), f)    ->
	       print_string "Float`"; print_float f; print_string "' "
      | String((s, e), str) ->
	       print_string ("String`" ^ str ^ "' ")
      | Char((s, e), c)     ->
	       print_string "Char`"; print_char c ; print_string "' "


(* exported *)
let (print_debug : Lexing.lexbuf -> string -> token Stream.t -> unit) =
  fun ic msg tok ->
    if debug then
      (
	(match Stream.peek tok with 
	     None -> print_string "End of file " 
	   | Some token ->  print_genlex_token token
	);
	print_string (
	  string_of_int (ic.lex_curr_pos) ^ "-" ^
	  (string_of_int (Stream.count tok)) ^ ": " ^ msg);
	flush stdout
      )
    else
      ()

let rec (parse_one_struct_field : Lexing.lexbuf -> token Stream.t -> vn_ct) =
  fun ic tok -> 
    (* 
       A struct field has the shape 
         <c type> <name> 
       or 
         <c type> <name> [<int>] 
       for arrays
    *)
    let _ = 
      print_debug ic ("parse_one_struct_field \n") tok ;
      match Stream.npeek 1 tok with  
	  (Ident (_, "extern"))::_ -> Stream.junk tok
	| _ -> ()
    in
    let tok_list = Stream.npeek 7 tok in 
    let (tail, ct) = 
      match tok_list with  

	| (Kwd (_, "unsigned"))::(Kwd(_, "short"))::(Kwd(_, "int"))::tail -> 
	    Stream.junk tok; Stream.junk tok; Stream.junk tok;  
	    tail,  "unsigned short int" 

	| (Kwd (_, "signed"))::(Kwd(_, "short"))::(Kwd(_, "int"))::tail ->  
	    Stream.junk tok; Stream.junk tok; Stream.junk tok;  
	    tail,  "signed short int" 
	| (Kwd (_, "unsigned"))::(Kwd(_, "long"))::(Kwd(_, "int"))::tail ->  
	    Stream.junk tok; Stream.junk tok; Stream.junk tok;  
	    tail,  "unsigned long int" 
	| (Kwd (_, "signed"))::(Kwd(_, "long"))::(Kwd(_, "int"))::tail ->  
	    Stream.junk tok; Stream.junk tok; Stream.junk tok;  
	    tail,  "signed long int" 
 
	| (Kwd (_, "short"))::(Kwd(_, "int") )::tail ->  
	    Stream.junk tok; Stream.junk tok; tail,  "short int" 
	| (Kwd (_, "long"))::(Kwd(_, "int") )::tail ->  
	    Stream.junk tok; Stream.junk tok; tail,  "long int" 
	| (Kwd (_, "long"))::(Kwd(_, "double"))::tail ->  
	    Stream.junk tok; Stream.junk tok; tail,  "long double" 
	| (Kwd (_, "signed"))::(Kwd(_, "int"))::tail ->  
	    Stream.junk tok; Stream.junk tok; tail,  "signed int" 
	| (Kwd (_, "unsigned"))::(Kwd(_, "int"))::tail ->  
	    Stream.junk tok; Stream.junk tok; tail,  "unsigned int" 
	| (Kwd (_, "unsigned"))::(Kwd(_, "char"))::tail ->  
	    Stream.junk tok; Stream.junk tok; tail,  "unsigned char" 
	| (Kwd (_, "unsigned"))::(Kwd(_, "long"))::tail ->  
	    Stream.junk tok; Stream.junk tok; tail,  "unsigned long" 
	| (Kwd (_, "unsigned"))::(Kwd(_, "short"))::tail ->  
	    Stream.junk tok; Stream.junk tok; tail,  "unsigned short" 
	     
	| ( Kwd(_, "int"))::tail -> Stream.junk tok; tail,  "int" 
	| ( Kwd(_, "long"))::tail -> Stream.junk tok; tail,  "long" 
	| ( Kwd(_, "short"))::tail -> Stream.junk tok; tail,  "short"
	| ( Kwd(_, "signed"))::tail -> Stream.junk tok; tail,  "signed" 
	| ( Kwd(_, "unsigned"))::tail -> Stream.junk tok; tail,  "unsigned" 
	| ( Kwd(_, "float"))::tail -> Stream.junk tok; tail,  "float" 
	| ( Kwd(_, "double"))::tail -> Stream.junk tok; tail,  "double" 
	| ( Kwd(_, "char"))::tail -> Stream.junk tok; tail,  "char"  

	| ( Ident (_, id) )::tail -> Stream.junk tok; tail,  id 

	| _  -> 
	    assert false
    in
      match tail with
	| (Ident (_, name))::(Kwd(_, "["))::(Int(_, i))::(Kwd(_, "]"))::_ -> 
	    Stream.junk tok; Stream.junk tok; Stream.junk tok; Stream.junk tok; 
	    (name, Array(i, Simple ct))

	| (Kwd(_, "["))::(Int(_, i))::(Kwd(_, "]"))::(Ident (_, name))::_ -> 
	    Stream.junk tok; Stream.junk tok; Stream.junk tok; Stream.junk tok; 
	    (name, Array(i, Simple ct))

	| (Ident (_, name))::_ -> 
	    Stream.junk tok; 
	    (name, Simple ct)

	| ((Kwd(_, "*")))::(Ident (_, name))::_ ->
	    Stream.junk tok;
	    Stream.junk tok;
	    (name, Simple (ct^"*"))

	| ((Kwd(_, "*")))::((Kwd(_, "*")))::(Ident (_, name))::_ ->
	    Stream.junk tok;
	    Stream.junk tok;
	    Stream.junk tok;
	    (name , Simple (ct ^ "**"))
	    (* XXX I suppose that there is no "type ***"; is it reasonnable? *)
	| _  -> 
	    assert false
	    

let rec  (find_typedef_list: Lexing.lexbuf -> typedef list -> token Stream.t -> 
	    typedef list) =
  fun ic tdl tok  ->
    let _ = print_debug ic ("find_typedef_list \n") tok in
      match tok with parser
	  [< 'Ident (_, "typedef");  td = find_typedef_list2 ic >] -> 
	    find_typedef_list ic (td::tdl) tok
	    
	| [< 'Ident (_, _) >] -> find_typedef_list ic tdl tok
	| [< 'Kwd (_, _) >] -> find_typedef_list ic tdl tok
	| [< 'Int (_, _) >] -> find_typedef_list ic tdl tok
	| [< 'Float (_, _) >] -> find_typedef_list ic tdl tok
	| [< 'Char (_, _) >] -> find_typedef_list ic tdl tok
	| [< 'String (_, _) >] -> find_typedef_list ic tdl tok
	| [< >]  ->  tdl    

and (find_typedef_list2: Lexing.lexbuf -> token Stream.t -> typedef) =
  fun ic tok ->
    let _ = print_debug ic ("find_typedef_list2 \n") tok in
      match tok with parser
	  [< 
	    'Ident (_, "struct"); 'Kwd (_, "{"); 
	    fl = parse_struct ic [];
	    'Ident (_, struct_name)
	  >] -> (struct_name, Struct(fl))
	| [< 
	    'Ident (_, "enum"); 'Kwd (_, "{");  'Ident (_, e1); 
	    el = parse_enum ic [e1];
	    'Ident (_, enum_name)
	  >] 
	  -> (enum_name, Enum(el))
	| [< vn_ct = parse_one_struct_field ic; 'Kwd (_, ";") >] -> vn_ct

	    
and (parse_struct : Lexing.lexbuf -> vn_ct list -> token Stream.t -> vn_ct list)=
  fun ic acc tok -> 
  let _ = print_debug ic ("parse_struct \n") tok in
    match tok with parser
	[<'Kwd (_, "}") >] -> List.rev acc
      | [<'Kwd (_, "#"); _ = skip_diese ic >] -> parse_struct ic acc tok
      | [<'Kwd (_, "/*"); _ = skip_comment ic >] -> parse_struct ic acc tok
      | [< vn_ct = parse_one_struct_field ic;  'Kwd (_, ";") >] -> 
	  parse_struct ic (vn_ct::acc) tok


and (parse_enum : Lexing.lexbuf -> string list -> token Stream.t -> string list)=
 fun ic acc tok -> 
   let _ = print_debug ic ("parse_enum \n") tok in
     match tok with parser
	 [< 'Kwd (_, "}") >] -> List.rev acc
       | [<'Kwd (_, "/*"); _ = skip_comment ic >] -> parse_enum ic acc tok
       | [<'Kwd (_, "#"); _ = skip_diese ic >] -> parse_enum ic acc tok
       | [< 'Kwd (_, ","); 'Ident (_, en) >] -> parse_enum ic (en::acc) tok

  (* 
     I need to deal with comment there as i look to comment to search
     for the input and output definitions
  *)
and (skip_comment : Lexing.lexbuf -> token Stream.t -> unit) =
  fun ic tok -> 
  let _ = print_debug ic ("skip_comment \n") tok in
    match tok with parser
	[< 'Kwd (_, "*/") >] -> ()
      | [< _ >] -> Stream.junk tok ; skip_comment ic tok
	  
and (skip_diese : Lexing.lexbuf -> token Stream.t -> unit) =
  fun ic tok -> 
  let _ = print_debug ic ("skip_diese \n") tok in
    match tok with parser
	[< 'Ident (_, "ifndef"); 'Ident (_,_) >] -> ()
      | [< 'Ident (_, "ifdef"); 'Ident (_,_) >] -> ()
      | [< 'Ident (_, "else") >] -> ()
      | [< 'Ident (_, "endif") >] -> ()


(* exported *)
let (get_typedef: file -> typedef list) =
  fun file ->
    let file_str = 
      try Util.readfile_rm_crtl_m file 
      with Not_found -> exit 2
    in
    let buff = Lexing.from_string file_str in
    let ic = try open_in file with
	_ ->
	  (
	    print_string ("*** File " ^ file
			  ^ " does not exist. Please check its name.\n");
	    flush stdout;
	    exit 2
	  )
    in
    let tok = (lexer(Stream.of_channel ic)) in
    let tdl = find_typedef_list buff [] tok in
      if debug2 then 
	(
	  print_string "typedef list: \n";
	  List.iter print_typedef tdl; 
	  flush stdout
	);
      tdl


(****************************************************************************)
(* exported *)
let rec (format_string_list: string -> string list -> string) =
  fun str list ->
    match list with
	[] -> ""
      | [e] -> e
      | e::t -> (e ^ str ^ (format_string_list str t))


(****************************************************************************)
(* exported *)
let (update_file : string -> string -> unit) =
  fun new_file old_file ->
  if
    not ((Sys.file_exists old_file)) ||
    ((Util.readfile_rm_crtl_m old_file) <> (Util.readfile_rm_crtl_m new_file))
  then
    (
      output_string stderr (new_file ^ " and " ^ old_file ^
			    " are different, therefore I recompile...\n");
      try Unix.rename new_file old_file
      with _ ->
	prerr_endline ("*** Error: " ^ new_file ^ " does not exist.\n");
	exit 2
    )

(****************************************************************************)

module Tbl = Util.StringMap


let rm_square_braces s = 
  (* 
     scade generates struct types this "{[ ... ]}" instead of like that
     "{ ... }". Therefore, i remove the square braquet there.

     nb: it is easier to remove it here rather than handling the 2 cases in the
     stream-based parser. 
  *)
  (Str.replace_first (Str.regexp "^{\[") "{"  
     (Str.replace_first (Str.regexp "\]}$") "}" s))


let (genlex_token_to_string : token -> string) =
  fun tok ->
    match tok with
	Kwd((s, e), str)   -> ("`" ^ str ^ "' ")
      | Ident((s, e), str) -> ("`" ^ str ^ "' ")
      | Int((s, e), i) -> "`" ^ (string_of_int i) ^ "' "
      | Float((s, e), f) -> "`" ^ (string_of_float f) ^ "' "
      | String((s, e), str) -> ("`" ^ str ^ "' ")
      | Char((s, e), c)     -> "`" ^ (Char.escaped c)^"' "


(* Used to parse the type definition that are coming from the Scade tcl gui. *)
let (typedef_to_type : (string * string) list -> (string * Type.t) list) =
  fun tdl -> 
    let rec get_types acc tok =
      (* 	let _ = print_string ("get_types \n") in *) (* XXX remove me !! *)
      (match tok with parser
	  [< 'Kwd(_, "{"); res = get_struct_field_types acc >] -> res
	| [< 'Ident(_,id);  _ = skip_until_kwd "," >] ->  (* ident or array *)
	    (match id with
		 (* do not add predefined types *)
		 "bool" -> acc
	       | "real" -> acc
	       | "string" -> acc
	       | "char" -> acc
	       | "int" -> acc
	       | _ -> 
		   (if List.mem id acc then acc else (id::acc))
	    )
	| [< 'Kwd(_, "(");  _ = skip_until_kwd ")" >] -> acc (* enum type *)
	| [< >] -> assert false 
      )    
    and skip_until_kwd str tok =
      let _ = 
	if debug2 then (
	let tok_list = Stream.npeek 1 tok in 
	let tok_str = if tok_list = [] then "... None!" else 
	    genlex_token_to_string (List.hd tok_list)
	in
	  print_string ("skip_until_kwd " ^ str ^ " -> "^ tok_str ^" \n")
	);
      in
	(match tok with parser
	   | [< 'Kwd(_, k) >] -> if str = k then () else skip_until_kwd str tok
	   | [< 'Ident(_,id) >] -> skip_until_kwd str tok
	   | [< 'Int(_,i) >] -> skip_until_kwd str tok
	   | [< 'Float(_,f) >] -> skip_until_kwd str tok
	   | [< 'String(_,str) >] -> skip_until_kwd str tok
	   | [< 'Char(_,c) >] -> skip_until_kwd str tok
	   | [< >] -> () 
      	)
    and get_struct_field_types acc tok =
      (* 	let _ = print_string ("get_struct_field_types \n") in *)
      match tok with parser
	  [<
	    'Ident(_, _fn); 'Kwd(_, ":"); acc2 = get_types acc; 
	    res = get_struct_field_types acc2
	  >]
	  -> res
	| [< 'Kwd(_, ",") ; res = get_struct_field_types acc >] -> res
	| [< 'Kwd(_, ";") ; res = get_struct_field_types acc >] -> res
	| [< 'Kwd(_, "}") >] -> acc
	| [< >] -> acc
	    
    and (get_sub_types : string -> string list) =
      fun str0 -> 
	(* Extract from str the (sub-)types that appear in it.
	   Two cases migth occur:
	   - structures, which are of the form "{f1:t1, f2:t2, ...}",
	   and for which we return [t1; t2; ...]
	   - arrays which are fo the form "t^n", and we return [t]
	   Otherwise, there is no sub-type, and we return the empty list
	*)
	let str = rm_square_braces str0 in
	let lexer =
	  MyGenlex.make_lexer ["^";"(";")";"{";":";",";";";"}"] (Stream.of_string str)
	in
	let _tok_list = Stream.npeek 10 lexer in
	let _tok_list2 = Stream.npeek 10 lexer in
	  (* 	    print_string ("\nGetting types of " ^str ^ "\n"); *)
	  get_types [] lexer
	    
    in
    let succ_table = 
      List.fold_left 
	(fun acc (t,td) -> 
	   let st = (get_sub_types td) in
	     Tbl.add t st acc)
	Tbl.empty 
	tdl 
    in
    let (type_list_sorted:string list) = 
      let type_list = fst (List.split tdl) in
	(GraphUtil.top_sort type_list succ_table) 
    in
    let typedef_tbl = 
      List.fold_left (fun acc (t,tdef) -> Tbl.add t tdef acc) Tbl.empty tdl 
    in
      List.fold_left
	(fun acc t -> 
	   let tdef_str = rm_square_braces (Tbl.find t typedef_tbl) in
	   let tdef = LucParse.parse_typedef_string tdef_str acc in
	     (t,tdef)::acc
	)
	[]
	type_list_sorted
	

(* exported *)
let (parse_scade_lurette_arg : string array -> 
       typedef list * vn_ct list * vn_ct list * string * string) =
  fun arg ->
    let _ = assert (arg.(1) = "-inputs") in
    let rec parse_io stop_str i acc =
      if arg.(i) = stop_str then (List.rev acc, i+1) else
	parse_io stop_str (i+2) ((arg.(i), arg.(i+1))::acc)
    in
    let rec get_typedef stop_str i acc =
      if arg.(i) = stop_str then (acc, i+1) else
	get_typedef stop_str (i+2) ((arg.(i), arg.(i+1))::acc)
    in
    let (vi_str, i0)  = parse_io "-outputs" 2 [] in
    let (vo_str, i1)  = parse_io "-typedef" i0 [] in

    let (tdl0, i2) = 
      print_string "get_typedef: \n";
      flush stdout;
      get_typedef "-name" i1 [] in
    let (tdl1: (string * Type.t) list) = 
      typedef_to_type tdl0 in
    let (tdl: (string * c_type) list) = 
      List.map (fun (x,y) -> (x,lucky_type_to_c_type y)) tdl1 
    in

    let ((vi,vo): (string * c_type) list * (string * c_type) list) =
      List.map (fun (x,y) -> (x,Simple y)) vi_str,
      List.map (fun (x,y) -> (x,Simple y)) vo_str
    in
      if debug2 then 
	(
	  print_string "typedef list: \n";
	  List.iter print_typedef tdl; 
	  flush stdout
	);
      (tdl, vi, vo, arg.(i2), arg.(i2+1))

(****************************************************************************)
