(*pp camlp4o *)
(*-----------------------------------------------------------------------
** Copyright (C) - Verimag.
**
** File: lurettetop.ml
** Main author: jahier@imag.fr
*)

open LtopArg

let usage = "  Type i to get test parameters info, h for help, m for a small manual.
"

let man = "
Once lurettetop has been launched, a prompt is printed waiting for
user queries. One first need at least to set the sut (system under
test) and the environment fields like that:

    [your shell prompt] lurettetop
    <lurette> add_rp \"sut:v6:heater_control.lus:heater_control\" 
    <lurette> add_rp \"env:lutin:env.lut:main\"
    <lurette> add_rp \"oracle:v6:heater_control.lus:not_a_sauna\"
or
    <lurette> add_rp \"oracle:ec:not_a_sauna.ec:\"

And then the testing process can start:

    <lurette> run
       ... [... testing ...]

Equivalently, you can directly set values at the command line:

    [your shell prompt] lurettetop -rp \"sut:v6:heater_control.lus:heater_control\" 
                                   -rp \"oracle:v6:heater_control.lus:not_a_sauna\"
                                   -rp \"env:lutin:env.lut:main\"
    <lurette> run
       ... [... testing ...]

You migth also want to try the info command (i for short) to get the values 
of the test parameters, and the h (help) command to obtain the list of 
possible commands.
"

 
let check_rif_name () =
  if (Sys.file_exists args.output && not args.overwrite_output) then
    let rec find_free_name b i =
      let f = Printf.sprintf "%s-%d.rif" b i in
        if Sys.file_exists f then find_free_name b (i+1) else f
    in
    let base_name = 
      try
        let i = Str.search_forward (Str.regexp "-[0-9]+\.rif") args.output 0 in
          String.sub args.output 0 i
      with Not_found -> 
        try Filename.chop_extension args.output
        with _ -> args.output
    in
      args.output <- find_free_name base_name 1
  
  
let (info : unit -> unit) =
  fun _ ->
    check_rif_name ();
    let msg =  "The current test parameters are:
     sut: "^ (String.concat "," (List.map reactive_program_to_string args.suts)) ^ "
     env: "^ (String.concat "," (List.map reactive_program_to_string args.envs)) ^ "
     oracle: "^ (String.concat "," (List.map reactive_program_to_string args.oracles)) ^ "
     test length: " ^  (string_of_int args.step_nb) ^ "
     precision: " ^ (string_of_int args.precision) ^ "
     seed:  " ^ (match args.seed with
			            None ->  "chosen randomly"
			          | Some i -> (string_of_int i))  ^ "
     verbosity level: " ^ (string_of_int (args.verbose)) ^ "
     rif file name: " ^ args.output ^ "
     overwrite rif file? "
      ^ (if args.overwrite_output then "yes" else "no") ^ "
     coverage file name: " ^ args.cov_file ^ "
     do we stop  when an oracle returns false? " 
      ^ (if args.stop_on_oracle_error then "yes" else "no") ^
     (match args.extra_cfiles with
		    None ->  ""
		  | Some str -> ("
     extra_source_files: " ^ str)) ^ 
     (match args.extra_libs with
	       None ->  ""
	     | Some str -> ("
     extra_libs: " ^ str)) ^
     (match args.extra_libdirs with
	       None ->  ""
	     | Some str -> ("
     extra_libdirs: " ^ str)) ^
     (match args.extra_includedirs with
	       None ->  ""
	     | Some str -> ("
     extra_includedirs: " ^ str))  ^ "
     display local var? " ^ (if (args.display_local_var) then "yes" else "no") ^ "

"
    in
      output_string args.ocr msg;
      flush args.ocr

let (display : unit -> unit) =
  fun _ ->
    let msg =  "The commands are:

run, r
     Start the testing process. 

quit q, bye
     Quit the lurette top level

help, h, ?
     Display this list of commands

info, i
     Display a sum-up of the test parameters

batch
    Generate a lurette.batch file with current test parameters
    
man
     Display a small user manual

man_lurette
man_lutin
tuto_lutin
    Launch pdf documentation

add_rp "^rp_help^"
     The current value of the rp fields are 
      sut: "^ (String.concat "," (List.map reactive_program_to_string args.suts)) ^ "
      env: "^ (String.concat "," (List.map reactive_program_to_string args.envs)) ^ "
      oracle: "^ (String.concat "," (List.map reactive_program_to_string args.oracles)) ^ "

reset_rp
     Reset the sut, env, and oracle fields

stl <integer>, set_test_length  <integer>
     Set the test length. Its current value is \"" ^
	       (string_of_int args.step_nb) ^ "\"

set_precision <int>
    Set the number of digit after the dot used for real computation.
    current precision is " ^ (string_of_int args.precision) ^ "

set_seed  <integer>
     Set the seed the random engine is initialised with.
     Its current value is " ^ (match args.seed with
			       None ->  "chose randomly"
			     | Some i -> ("\"" ^ (string_of_int i) ^ "\""))  ^ "
set_seed_randomly
     Let the system set a seed randomly.

set_verbose (0,1,2)
     Set on and off a verbose mode. Its current value is "
	       ^ (string_of_int (args.verbose)) ^ "

set_rif <string>, set_output  <string>
     Set the name of the file the (rif) output of the test is
     put into. Its current value is \"" ^ args.output ^ "\"

set_overwrite_output <bool> 
    Set the overwrite_output mode

sim2chro, s
    Call sim2chro to visualize (rif) data

gnuplot, g
    Call gnuplot (> 3.7) to visualize (rif) data.

set_cov_file <string>
    Set the coverage file name. Its current value is \"" ^ args.cov_file ^ "\"

reset_cov_file <bool>
    Reset the coverage info in the coverage file before the next run.
    Its current value is \"" ^ (if args.reset_cov_file then "true" else "false") ^ "\"

stop_on_oracle_error <bool>
    Set a boolean flag stating what to do when an oracle returns false    
    Its current value is \"" ^ (if args.stop_on_oracle_error then "true" else "false") ^ "\"

more
    Display more commands.

"
    in
      output_string args.ocr msg;
      flush args.ocr

let (display2 : unit -> unit) =
  fun _ ->
    let msg =  "The advanced commands are:

set_draw_nb  <integer>
     Set the number of draw to be done in each formula at each step
     Its current value is \"" ^ (string_of_int args.draw_nb) ^ "\"

set_draw_inside <integer>
set_draw_edges <integer>
set_draw_vertices <integer>
     Set the number of draw to be made:
       - inside: i.e., draw inside the convex hull of solutions.
       - edges : i.e., draw on the edges of the convex hull of solutions.
       - vertices : i.e.,  draw among the vertices of the convex hull of solutions.

     Their current values are
         inside: \"" ^ (string_of_int args.draw_inside ) ^ "\"
         edges: \"" ^ (string_of_int args.draw_edges ) ^ "\"
         vertices: \"" ^ (string_of_int args.draw_vertices ) ^ "\"

set_step_mode {inside|edges|vertices}
    Set the mode used to perform the step

set_draw_all_formula <Boolean>
     Set a Boolean that indicates whether lurette should tries
     one or all the formula reachable from the current step.
     Its current value is " 	
	       ^ (if (args.all_formula)
		  then "\"true\""
		  else "\"false\"") ^ "

set_draw_all_vertices <Boolean>
     Set a Boolean that indicates whether lurette should tries
     all polyhedra vertices.
     Its current value is " 	
	       ^ (if (args.all_vertices)
		  then "\"true\""
		  else "\"false\"") ^ "

set_compute_volume <Boolean>
     Set a Boolean that indicates whether lurette should compute
     polyhedra volumes (achieve fairness, but more expensive).
     Its current value is " 	
	       ^ (if (args.compute_volume)
		  then "\"true\""
		  else "\"false\"") ^ "

set_extra_source_files <string>
     Set the EXTRA_SOURCE_FILES environment variable.
     Its current value is " ^
	       (match args.extra_cfiles with
		    None ->  "unset"
		  | Some str -> ("\"" ^ str ^ "\""))  ^ "

set_extra_libs <string>
     Set the EXTRA_LIBS environment variable.
     Its current value is " ^
	       (match args.extra_libs with
		    None ->  "unset"
		  | Some str -> ("\"" ^ str ^ "\""))  ^ "

set_extra_libdirs <string>
     Set the EXTRA_LIBDIRS environment variable.
     Its current value is " ^
	       (match args.extra_libdirs with
		    None ->  "unset"
		  | Some str -> ("\"" ^ str ^ "\""))  ^ "

set_extra_includedirs <string>
     Set the EXTRA_INCLUDEDIRS environment variable.
     Its current value is " ^
	       (match args.extra_includedirs with
		    None ->  "unset"
		  | Some str -> ("\"" ^ str ^ "\""))  ^ "
set_display_local_var  <boolean>
     Set a flag saying whether or not the local var be displayed in
     sim2chro. Its current value is " ^ (if (args.display_local_var)
			   then "\"true\""
			   else "\"false\"") ^ "

set_prefix  <string>
     Set the string that is appended before the call to lurette.
     This useful, e.g., for timings.
     Its current value is \"" ^ args.prefix ^ "\"

set_step_by_step <int>
    Set the step_by_step mode on. 

set_step_by_step_off
    Set the step_by_step mode off.  

change_dir <string>
    Change the current directory.
    The current dir is \"" ^ args.sut_dir ^ "\"

set_tmp_dir <string>
    Change the temporary directory.
    The current dir is \"" ^ args.tmp_dir ^ "\"
"
    in
      output_string args.ocr msg;
      flush args.ocr

(***********************************************************************)

(* XXX Ca sert pas à grand chose de passer par ca, si ? *)

type t =
    Sut of string * string
  | Oracle of string * string
(*   | MakeOpt of string *)
  | Env of string
  | StepNb of int
  | DrawNb of int
  | DisplaySim2chro of bool
  | DisplayLocalVar of bool
  | StepByStep of int option
  | Seed of int
  | Precision of int
  | RandomSeed
  | Verbose of verbose_level
  | Reactive of bool
  | ComputeVolume of bool
  | ShowStep of bool
  | Output of string
  | OverwriteOutput of bool
  | SetSutCompiler of compiler
  | SetOracleCompiler of compiler
  | CallSim2chro
  | CallGnuplot
  | CallGnuplotPs
  | Build
  | BuildEnv
  | ChDir of string
  | ChTmpDir of string
  | Clean
  | Log
  | Run
  | Batch of string
  | DrawInside of int
  | DrawEdges of int
  | DrawVertices of int
  | AllFormula of bool
  | StepMode of string
  | AllVertices of bool
  | Quit
  | Save
  | Help
  | HelpMore
  | Man
  | LurettePdfMan
  | LutinPdfMan
  | LutinPdfTuto
  | Prompt of string
  | Prefix of string
  | Pack of string
  | ExtraCFiles of string
  | ExtraLibs of string
  | ExtraLibDirs of string
  | ExtraIncludeDirs of string
  | HelpSimple
  | Error of string
  | SutCmd of string
  | OracleCmd of string
  | StdinMode of bool
  | PP of string option
  | CGeneratorFlag of string
  | RootNode of string 
  | ResetCovFile of bool
  | CovFile of string
  | StopOnOracleError of bool
  | Info
  | Nop

let lexer = MyGenlex.make_lexer []

type tok = MyGenlex.token Stream.t


let rec (parse_tok : tok -> t) =
  fun tok ->
    match tok with parser
      | [< 'MyGenlex.Ident(_, "run") >] -> Run
      | [< 'MyGenlex.Ident(_, "r") >] -> Run
      | [< 'MyGenlex.Ident(_, "b") >] -> Build
      | [< 'MyGenlex.Ident(_, "info") >] -> Info
      | [< 'MyGenlex.Ident(_, "i") >] -> Info
      | [< 'MyGenlex.Ident(_, "build") >] -> Build
      | [< 'MyGenlex.Ident(_, "build_env") >] -> BuildEnv
      | [< 'MyGenlex.Ident(_, "change_dir") ; 'MyGenlex.String(_, dir) >] -> ChDir(dir)
      | [< 'MyGenlex.Ident(_, "set_tmp_dir") ; 'MyGenlex.String(_, dir) >] -> ChTmpDir(dir)

      | [< 'MyGenlex.Ident(_, "set_step_mode") ; str = parse_ident_or_string >] -> StepMode(str)
      | [< 'MyGenlex.Ident(_, "set_draw_inside") ; 'MyGenlex.Int(_, i) >] -> DrawInside(i)
      | [< 'MyGenlex.Ident(_, "set_draw_edges") ; 'MyGenlex.Int(_, i) >] -> DrawEdges(i)
      | [< 'MyGenlex.Ident(_, "set_draw_vertices") ; 'MyGenlex.Int(_, i) >] -> DrawVertices(i)

      | [< 'MyGenlex.Ident(_, "set_draw_all_formula"); str = parse_ident_or_string
	     >] ->
	       if List.mem str ["t";"true"]
	       then AllFormula(true)
	       else AllFormula(false)
      | [< 'MyGenlex.Ident(_, "set_draw_all_vertices"); str = parse_ident_or_string
	     >] ->
	       if List.mem str ["t";"true"]
	       then AllVertices(true)
	       else AllVertices(false)

      | [< 'MyGenlex.Ident(_, "set_direct_mode") >] -> args.direct_mode <- true; Nop
      | [< 'MyGenlex.Ident(_, "set_old_mode") >] -> args.direct_mode <- false; Nop
      | [< 'MyGenlex.Ident(_, "clean") >] -> Clean
      | [< 'MyGenlex.Ident(_, "log") >] -> Log
      | [< 'MyGenlex.Ident(_, "set_prompt"); str = parse_ident_or_string >] -> Prompt(str)
      | [< 'MyGenlex.Ident(_, "set_prefix"); str = parse_ident_or_string >] -> Prefix(str)
      | [< 'MyGenlex.Ident(_, "set_env") ; str = parse_ident_or_string >] -> Env(str)
      | [< 'MyGenlex.Ident(_, "set_dbg_on")  >] -> args.debug_ltop <- true; Nop
      | [< 'MyGenlex.Ident(_, "set_dbg_off")  >] -> args.debug_ltop <- false; Nop
          (*       | [< 'MyGenlex.Ident(_, "set_env") ; str = parse_env >] -> Env(str) *)


      | [< 'MyGenlex.Ident(_, "add_rp") ; 'MyGenlex.String(_, str)  >] -> 
          (try parse_rp_string str; Nop 
           with Failure(msg) -> Error (msg)
          )     
      | [< 'MyGenlex.Ident(_, "reset_rp")   >] -> 
          args.oracles <- []; args.envs <- [] ; args.suts <- []; Nop


      | [< 'MyGenlex.Ident(_, "set_root_node") ; node = parse_node >] -> RootNode(node)
      | [< 'MyGenlex.Ident(_, "set_sut") ; str = parse_file_name ; node = parse_node >] -> Sut(str, node)
      | [< 'MyGenlex.Ident(_, "set_sut_cmd") ; 
	        str = parse_ident_or_string 
	     >] -> SutCmd(str)
      | [< 'MyGenlex.Ident(_, "set_oracle_cmd") ; 
	        str = parse_ident_or_string >] 
	     -> OracleCmd(str)
      | [< 'MyGenlex.Ident(_, "set_oracle") ; str = parse_file_name ; node = parse_node>] -> Oracle(str, node)
          (*       | [< 'MyGenlex.Ident(_, "set_make_opt") ; str = parse_ident_or_string >] -> MakeOpt(str) *)
      | [< 'MyGenlex.Ident(_, "stl") ; 'MyGenlex.Int(_, i) >] -> StepNb(i)
      | [< 'MyGenlex.Ident(_, "set_test_length") ; 'MyGenlex.Int(_, i) >] -> StepNb(i)
      | [< 'MyGenlex.Ident(_, "set_draw_nb") ; 'MyGenlex.Int(_, i) >] -> DrawNb(i)
      | [< 'MyGenlex.Ident(_, "set_seed") ; 'MyGenlex.Int(_, i) >] -> Seed(i)
      | [< 'MyGenlex.Ident(_, "set_precision") ; 'MyGenlex.Int(_, i) >] -> Precision(i)
      | [< 'MyGenlex.Ident(_, "set_seed_randomly") >] -> RandomSeed

      | [< 'MyGenlex.Ident(_, "set_preprocessor"); pp = parse_ident_or_string>] ->  PP(Some pp)

      | [< 'MyGenlex.Ident(_, "set_sut_compiler"); s = parse_ident_or_string >] ->
	       ( match (string_to_compiler s) with
		          Some comp -> SetSutCompiler(comp)
	           | None -> Error ("'" ^ s ^ "' is not a supported compiler.\n")
	       )
      | [< 'MyGenlex.Ident(_, "set_oracle_compiler"); s = parse_ident_or_string >] ->
	       ( match (string_to_compiler s) with
		          Some comp -> SetOracleCompiler(comp)
	           | None -> Error ("'" ^ s ^ "' is not a supported compiler.\n")
	       )

      | [< 'MyGenlex.Ident(_, "set_extra_cfiles") ; 'MyGenlex.String(_, str) >] ->
	       ExtraCFiles(str)
      | [< 'MyGenlex.Ident(_, "set_extra_source_files") ; 'MyGenlex.String(_, str) >] ->
	       ExtraCFiles(str)

      | [< 'MyGenlex.Ident(_, "set_extra_libs") ; 'MyGenlex.String(_, str) >] ->
	       ExtraLibs(str)

      | [< 'MyGenlex.Ident(_, "set_extra_libdirs") ; 'MyGenlex.String(_, str) >] ->
	       ExtraLibDirs(str)

      | [< 'MyGenlex.Ident(_, "set_extra_includedirs") ; 'MyGenlex.String(_, str) >] ->
	       ExtraIncludeDirs(str)

      | [< 'MyGenlex.Ident(_, "set_stdin_mode") ; str = parse_ident_or_string >] ->
	       if List.mem str ["t";"true"]
	       then StdinMode(true)
	       else StdinMode(false)

      | [< 'MyGenlex.Ident(_, "set_step_by_step") ; 'MyGenlex.Int(_, i) >] ->
	       StepByStep(Some i)

      | [< 'MyGenlex.Ident(_, "set_step_by_step_off")  >] ->
	       StepByStep(None)


      | [< 'MyGenlex.Ident(_, "set_display_sim2chro") ; 
	        str = parse_ident_or_string
	     >] ->
	       if List.mem str ["t";"true"]
	       then DisplaySim2chro(true)
	       else DisplaySim2chro(false)

      | [< 'MyGenlex.Ident(_, "reset_cov_file") ; 
	        str = parse_ident_or_string
	     >] ->
	       if List.mem str ["t";"true"]
	       then ResetCovFile(true)
	       else ResetCovFile(false)

      | [< 'MyGenlex.Ident(_, "set_cov_file") ;'MyGenlex.String(_, str) >] ->
	       CovFile(str)


      | [< 'MyGenlex.Ident(_, "stop_on_oracle_error") ; 
	        str = parse_ident_or_string
	     >] ->
	       if List.mem str ["t";"true"]
	       then StopOnOracleError(true)
	       else StopOnOracleError(false)

      | [< 'MyGenlex.Ident(_, "set_display_local_var") ; 
	        str = parse_ident_or_string >] ->
	       if List.mem str ["t";"true"]
	       then DisplayLocalVar(true)
	       else DisplayLocalVar(false)

      | [< 'MyGenlex.Ident(_, "set_compute_volume") ; 
	        str = parse_ident_or_string >] ->
	       if List.mem str ["t";"true"]
	       then ComputeVolume(true)
	       else ComputeVolume(false)

      | [< 'MyGenlex.Ident(_, "set_verbose") ; 'MyGenlex.Int(_, i)  >] ->
	       Verbose(i)
	         
      | [< 'MyGenlex.Ident(_, "set_reactive") ; str = parse_ident_or_string >] ->
	       if List.mem str ["t";"true"]
	       then Reactive(true)
	       else Reactive(false)

      | [< 'MyGenlex.Ident(_, "set_show_step") ; str = parse_ident_or_string >] ->
	       if List.mem str ["t";"true"]
	       then ShowStep(true)
	       else ShowStep(false)

      | [< 'MyGenlex.Ident(_, "set_rif") ; str = parse_file_name>] -> Output(str)
      | [< 'MyGenlex.Ident(_, "set_output") ; str = parse_file_name>] -> Output(str)
      | [< 'MyGenlex.Ident(_, "set_overwrite_output") ; str = parse_ident_or_string>] ->
          OverwriteOutput(List.mem str ["t";"true"])

      | [< 'MyGenlex.Ident(_, "batch" ) ; str = parse_ident_or_string >] -> Batch(str)
      | [< 'MyGenlex.Ident(_, "sim2chro") >] -> CallSim2chro
      | [< 'MyGenlex.Ident(_, "s") >] -> CallSim2chro
      | [< 'MyGenlex.Ident(_, "gen_fake_lucky") >] -> BuildEnv
      | [< 'MyGenlex.Ident(_, "g") >] -> CallGnuplot
      | [< 'MyGenlex.Ident(_, "gnuplot") >] -> CallGnuplot
      | [< 'MyGenlex.Ident(_, "gnuplot_ps") >] -> CallGnuplotPs

      | [< 'MyGenlex.Ident(_, "save")  >] -> Save

      | [< 'MyGenlex.Ident(_, "quit")  >] -> Quit
      | [< 'MyGenlex.Ident(_, "q")  >] -> Quit
      | [< 'MyGenlex.Ident(_, "bye")  >] -> Quit
      | [< 'MyGenlex.Ident(_, "exit")  >] -> Quit

      | [< 'MyGenlex.Ident(_, "m")  >] -> Man
      | [< 'MyGenlex.Ident(_, "man")  >] -> Man

      | [< 'MyGenlex.Ident(_, "man_lurette")  >] -> LurettePdfMan
      | [< 'MyGenlex.Ident(_, "man_lutin")  >] -> LutinPdfMan
      | [< 'MyGenlex.Ident(_, "tuto")  >] -> LutinPdfTuto
      | [< 'MyGenlex.Ident(_, "tuto_lutin")  >] -> LutinPdfTuto

      | [< 'MyGenlex.Ident(_, "pack")  ; file = parse_ident_or_string >] -> Pack(file)

      | [< 'MyGenlex.Ident(_, "set_c_generator_flag"); 
	        file = parse_ident_or_string >] -> CGeneratorFlag(file)

      | [< 'MyGenlex.Ident(_, "more")  >] -> HelpMore
      | [< 'MyGenlex.Ident(_, "help")  >] -> Help
      | [< 'MyGenlex.Ident(_, "h")  >] -> Help
      | [< 'MyGenlex.Ident(_, "?")  >] -> Help
      | [< 'MyGenlex.Ident(_, cmd )>]  -> Error (cmd ^ ": unknown command.\n")
      | [<   >] -> HelpSimple


and
    (parse_ident_or_string : tok -> string) =
  fun tok ->
    try
      match tok with parser
	     | [< 'MyGenlex.Ident(_, id ) >] -> id
	     | [< 'MyGenlex.String(_, id )>] -> id
	     | [<   >] -> ""
    with _ ->
	   output_string args.ocr
	     "*** parse error.\n";
	   flush args.ocr;
	   ""
and
    (parse_node : tok -> string) =
  fun tok ->
    try
      match tok with parser
	     | [< 'MyGenlex.Ident(_, id ) >] -> id
	     | [< 'MyGenlex.String(_, id )>] -> id
	     | [<   >] -> ""
    with _ ->
	   output_string args.ocr
	     "*** parse error: cannot parse that node name.\n";
	   flush args.ocr;
	   ""
and
    (parse_file_name : tok -> string) =
  fun tok ->
    try
      match tok with parser
	     | [<  'MyGenlex.String(_, str )>] -> str
	     | [<  'MyGenlex.Ident(_, id ) >] -> id
	     | [<  >]  ->  ""
    with _ ->
	   output_string args.ocr
	     "*** parse error: cannot parse that file name.\n";
	   flush args.ocr;
	   ""
and
    (parse_env : tok -> string) =
  fun tok ->
    try
      (
	     match tok with parser
            (* 	  | [<  'MyGenlex.Ident(_, "x") ; tail = parse_env >] -> (" x " ^ tail) *)
	       | [<  'MyGenlex.String(_, str ); tail = parse_env >] -> (str ^ " " ^tail)
              (* 	  | [<  'MyGenlex.Ident(_, id ) ; tail = parse_env >] -> (id ^ ".luc " ^ tail) *)
	       | [< _ >]  ->  ""
      )
    with e ->
      output_string args.ocr (Printexc.to_string e);
      output_string args.ocr
        "*** Error when parsing the environment field.\n";
      flush args.ocr;
      ""

let (parse : string  -> t) =
  fun str -> 
    parse_tok (lexer (Stream.of_string (str)))

let dot_exe = ExtTools.dot_exe


let remove file = if Sys.file_exists file then Sys.remove file

(* Remove some generated files (lurette_exe in particular) if they
   appear to be outdated; this will force some rebuild to occur. 

   That job should definitely be done by Makefile.lurette
*)
let (remove_outdated_files : unit -> unit) = 
  fun () ->  
    let lurette_exe = Filename.concat args.tmp_dir ("lurette"^dot_exe) in
      if not (Sys.file_exists lurette_exe) then (
        output_string args.ecr (lurette_exe ^ " does not exist. It needs to be build.\n");
        flush args.ecr
      )
      else 
        (* lurette_exe exists, but it migth be outdated. *)
        if (Filename.check_suffix args.sut ".c") then
          (* The SUT is a C file : In that case, we need to rebuild iff
             the sut is newer than the executable lurette.  *)
          if
            let stat1 = Unix.stat (Filename.concat args.sut_dir args.sut)
            and stat2 = Unix.stat lurette_exe
            in
	      (
	        (stat1.Unix.st_mtime > stat2.Unix.st_mtime)
	        ||
	          args.scade_gui (* do not need to check the oracle in that 
			            case since the oracle node is in the 
			            same .etp as the sut *)
	        ||
	          ( match args.oracle with
		      | None -> false
		      | Some oracle ->
		          let stat3 = Unix.stat (Filename.concat args.sut_dir oracle) in
		            (stat3.Unix.st_mtime > stat2.Unix.st_mtime)
	          )
	      )
          then (
            output_string args.ecr 
	      (args.sut^" is newer than "^lurette_exe^"; it needs to be build again.\n");
	    flush args.ecr;
            remove lurette_exe
          ) else ()
        else
          (* The SUT is not a C file :
             We also need to rebuild if the generated c files are older
             than args.sut
          *)
          (
            let gen_c_file = (Filename.concat args.tmp_dir (args.sut_node ^ ".c"))
            and gen_h_file = (Filename.concat args.tmp_dir (args.sut_node ^ ".h"))
            and gen_c_file_oracle = (Filename.concat args.tmp_dir (args.oracle_node ^ ".c"))
            and gen_h_file_oracle = (Filename.concat args.tmp_dir (args.oracle_node ^ ".h"))
            in
	      if
	        not args.scade_gui 
	          (* in the scade gui mode, the user is asked to explicitely
	             compile things.
	          *)
	      then
	        if
	          let stat1 = Unix.stat (Filename.concat args.sut_dir args.sut) 
	          and stat2 = Unix.stat gen_c_file
	          in
	            (stat1.Unix.st_mtime > stat2.Unix.st_mtime)
	        then
	          (
	            remove gen_c_file;
	            remove gen_h_file;
	            remove lurette_exe;
	            output_string args.ecr 
		      ((args.sut) ^ " is newer than " ^ gen_c_file ^ "; " ^
		         lurette_exe ^ " needs to be build again.\n");
	            flush args.ecr
	          );
	      (* do the same for the oracle*)
	      (match args.oracle with
	         | None -> ()
	         | Some oracle ->
	             if 
		       not args.scade_gui 
		         (* ditto, the user ougth to compile it explicitely *)
	             then
		       if
		         let stat1 = Unix.stat (Filename.concat args.sut_dir oracle)
		         and stat2 = Unix.stat gen_c_file_oracle
		         in
		           (stat1.Unix.st_mtime > stat2.Unix.st_mtime)
		       then
		         (
		           output_string args.ecr 
		             ((Filename.concat args.sut_dir oracle) ^ " is newer than " ^
			        gen_c_file_oracle^"; " ^
			        "lurette" ^ dot_exe ^ " needs to be build again.\n");
		           flush args.ecr;
		           remove gen_c_file_oracle;
		           remove gen_h_file_oracle;
		           remove lurette_exe
		         )
	      )
          )



(* exported *)
let (read : string -> bool) =
  fun str ->
    try
	   if args.verbose > 1 then 
        (output_string args.ocr("ltop : " ^ str ^ "\n"); flush args.ocr);
	   (match (parse str) with
        | Nop -> true
	     | Sut(sut0, node) ->
	       let sut =
		      if Filename.is_implicit sut0 then
		        sut0
		      else
		        (
		          args.sut_dir <- (Filename.dirname sut0);
		          Filename.basename sut0
		        )
	       in
	       let old_sut = args.sut
	       and old_node = args.sut_node
	       in
		    if (sut <> old_sut || node <> old_node) then
		      (
		        args.sut <- sut ;
		        args.sut_node <- node
                 (* 		    output_string args.ecr ("The sut node has changed.\n "); *)
                 (* 		    flush args.ecr; *)
		      );
		    true
		      
	     | Oracle(oracle, node) ->
	       let old_oracle = args.oracle
	       and old_node = args.oracle_node
	       in
	       let oracle2 = match oracle with "" -> None | x -> Some x in
		    if (oracle2 <> old_oracle || node <> old_node) then
		      (
		        if oracle = "" then (
		          args.oracle <- None ;
		          args.oracle_node <- ""
		        )
		        else
		          (
			         args.oracle <- Some oracle ;
			         args.oracle_node <- node
		          )
		      );
	       true
        | ResetCovFile(b) -> args.reset_cov_file <- b; true 
        | CovFile(f) -> args.cov_file <- f; true
        | Info -> info(); true
        | StopOnOracleError(b) -> args.stop_on_oracle_error <- b ; true

	     | RootNode(node) -> args.root_node <- node; true
	     | PP(pp) -> args.pp <- pp ; true

	     | SutCmd(cmd) -> args.sut_cmd <- cmd ; true
	     | OracleCmd(cmd) -> args.oracle_cmd <- cmd ; true
	     | StepMode(str) ->
	       let step_mode = string_to_step_mode str in
		    args.step_mode <- step_mode; true
         (* 	  | MakeOpt(str) ->  *)
         (* 	      args.make_opt <- str; true *)
	     | Env(str)  -> args.env <- LtopArg.explicit_the_file str; true
	     | StepNb(i) -> args.step_nb <- i; true
	     | DrawNb(i) -> args.draw_nb <- i; true

	     | DisplaySim2chro(b) -> args.display_sim2chro <- b; true
	     | DisplayLocalVar(b) -> args.display_local_var <- b; true
	     | SetSutCompiler(comp) -> args.sut_compiler <- comp; true
	     | SetOracleCompiler(comp) -> args.oracle_compiler <- comp; true

	     | StdinMode(b)  -> if b then args.sut_compiler <- Stdin; true
	     | StepByStep(b) -> args.step_by_step <- b; true
	     | Seed(i)       -> args.seed <- Some i; true
	     | Precision(i)  -> args.precision <-  i; true
	     | ComputeVolume(b) -> args.compute_volume <- b; true
	     | RandomSeed    -> args.seed <- None; true
	     | Verbose(b)    -> args.verbose <- b; true
	     | Reactive(b)   -> args.reactive <- b; true
	     | ShowStep(b)   -> args.show_step <- b; true
	     | Prompt(p)     -> args.prompt <- Some p; true
	     | Prefix(p)     -> args.prefix <-  p; true
	     | DrawInside i  -> args.draw_inside <- i; true
	     | DrawEdges i   -> args.draw_edges <- i; true
	     | DrawVertices i-> args.draw_vertices <- i; true
	     | AllFormula b  -> args.all_formula <- b; true
	     | AllVertices b -> args.all_vertices <- b; true
	     | Output(str)   -> args.output <- str; true
	     | OverwriteOutput(b)   -> args.overwrite_output <- b; true
	     | Log           -> args.log <- true; true
	     | Clean ->
	          (* XXX Not portable !! *)
	          (* Clean up intermediary files *)
          if args.scade_gui then (
            let make = ExtTools.make in
            let makefile = (Filename.concat args.tmp_dir "Makefile")  in
            let make_arg_list = (List.tl make) @ [ "-f"; makefile; "lurette_clean"] in
            Unix.chdir args.tmp_dir;
            Unix.putenv "SUT_DIR" args.tmp_dir;
            Unix.putenv "SUT" args.sut_node;
            Unix.putenv "USER_TESTING_DIR" (args.sut_dir);
            Unix.putenv "LURETTE_TMP_DIR" (args.tmp_dir);
            if args.luciole_mode then Unix.putenv "LURETTE_DRO" "lurette.dro";
            ignore (Util.my_create_process ~std_out:(Unix.descr_of_out_channel args.ocr) 
                      ~std_err:(Unix.descr_of_out_channel args.ecr)
                      (List.hd make) make_arg_list)
          )
          else
               (* XXX comme pour scade_gui ci dessus, je devrais faire ca via le makefile *)
            (Util2.del (Filename.concat args.tmp_dir ("lurette")) args.ocr args.ecr;
             Util2.del (Filename.concat args.tmp_dir ("*.*")) args.ocr args.ecr)
          ;
	       true
		      
	     | Batch(file) -> LtopArg.gen_batch file; true

	     | ChDir(dir) ->
	       output_string args.ocr
	         (" The current directory for lurette is now " ^ dir ^ 
                  (* 	       "\n cygpath_dir = " ^ (Util.cygpath dir) ^  *)
	             "\n");
	       flush args.ocr ;
	       args.sut_dir <-  dir;
	       true

	     | ChTmpDir(dir) ->
	       output_string args.ecr
	         (" The current temporary directory for lurette is now "^dir^"\n");
	       flush args.ecr ;
	       args.tmp_dir <-  dir;
	       true

	     | BuildEnv ->
	       true

	     | Build ->
	       let build_ok = Build.f args in
		    if not build_ok then (
		      output_string args.ocr "\n*** Cannot build lurette, sorry.\n \n \n";
		      flush args.ocr
		    );
		    true
	     | Run -> (
          check_rif_name ();
          if not args.direct_mode then remove_outdated_files ();
          if ( args.direct_mode || Build.f args) then ( 
            Unix.chdir  args.sut_dir;
            let result = Run.f () in
            if result <> 0 
            then (
              output_string args.ocr "\n*** lurette has terminated abnormally.\n \n";
              RunDirect.clean_terminate()
            )
            else output_string args.ocr "\nLurette has terminated normally.\n"
          )
          else
            output_string args.ocr "\n*** Cannot build lurette, sorry.\n \n \n";
          flush args.ocr;
          Unix.chdir args.tmp_dir;
          args.reset_cov_file <- false;
          true
	     )  
	     | CallSim2chro ->
	       let sdir = Unix.getcwd () in
	       Unix.chdir args.sut_dir;
	       ignore (Util2.sim2chro args.output);
	       Unix.chdir sdir;
	       true
	     | CallGnuplot ->
	       let sdir = Unix.getcwd () in
	       Unix.chdir args.sut_dir;
	       Util2.gnuplot (args.verbose>1) args.output;
	       Unix.chdir sdir;
	       true
	     | CallGnuplotPs ->
	       let sdir = Unix.getcwd () in
	       Unix.chdir args.sut_dir;
	       ignore (Util2.gnuplot_ps args.output);
	       Unix.chdir sdir;
	       true
	     | Quit ->
          RunDirect.clean_terminate();
          false
	     | Save -> LtopArg.gen_lurette_rc ();true
	     | HelpSimple -> 
          output_string args.ocr usage; flush args.ocr; true
	     | Help -> display (); true
	     | HelpMore -> display2 (); true

	     | Man -> output_string args.ocr man ; flush args.ocr; true
	     | LurettePdfMan -> Util2.lurette_man (); true
	     | LutinPdfMan   -> Util2.lutin_man (); true
	     | LutinPdfTuto  -> Util2.lutin_tuto (); true

	     | CGeneratorFlag(str) -> 
	       args.c_generator <- str; true
	         
	     | Pack(file) -> (* not working *)
	       assert false
         (* 	      let *)
         (* 		cmd = ("mv " ^ args.tmp_dir ^ " /tmp/" ^ file ^ *)
         (* 		       "; cd " ^ args.sut_dir ^ "; tar cvfz " ^ *)
         (* 		       file ^ ".tgz /tmp/" ^ file ^ *)
         (* 		       " > tar.log; mv /tmp/" ^ *)
         (* 		       file ^ " " ^ args.tmp_dir ^ " >> tar.log " ) *)
         (* 	      in *)
         (* 	      let tar_res = *)
         (* 		output_string args.ecr (cmd ^ "\n") ; *)
         (* 		flush args.ecr; *)
         (* 		Sys.command cmd *)
         (* 	      in	 *)
         (* 		if tar_res <> 0 *)
         (* 		then *)
         (* 		  ( *)
         (* 		    output_string args.ocr ("*** <<" ^ cmd ^ *)
         (* 				  ">> failed. Is gnu-tar in your path ?\n"); *)
         (* 		    flush args.ocr; *)
         (* 		    true *)
         (* 		  ) *)
         (* 		else *)
         (* 		  true *)

	     | ExtraCFiles(str) ->
	       Unix.putenv "EXTRA_SOURCE_FILES" (String.escaped (get_full_path args.sut_dir str));
	       let cfiles = get_full_path args.sut_dir str in
	       args.extra_cfiles <- if cfiles = "" then None else Some cfiles;
	       true
	     | ExtraLibs(str) ->
	       Unix.putenv "EXTRA_LIBS" (String.escaped str);
	       args.extra_libs <- if str = "" then None else Some str;
	       true
	     | ExtraLibDirs(str) ->
	       Unix.putenv "EXTRA_LIBDIRS" (String.escaped str);
	       args.extra_libdirs <- if str = "" then None else Some str;
	       true
	     | ExtraIncludeDirs(str) ->
	       Unix.putenv "EXTRA_INCLUDEDIRS" (String.escaped str);
	       args.extra_includedirs <- if str = "" then None else Some str;
	       true

	     | Error(errmsg) ->
	       output_string args.ocr errmsg;
	       flush args.ocr;
	       output_string args.ocr usage;
	       flush args.ocr;
	       true
      )
    with
        e ->
          RunDirect.clean_terminate();
          output_string args.ocr ("Bad lurette command: " 
			                         ^ (Printexc.to_string e) ^ " ("^str^") \n") ;
          output_string args.ocr usage ;
          flush args.ocr;
          true
