
(*
  Se plugger sur l'API c de l'oracle ?

  lire le rif sur stdin ?
*)

open Format

(**************************************************************************)
type argT = {
  mutable rif : string option;
  mutable ec : string;
  mutable cov : string;
  mutable debug : bool;
  mutable reinit_cov : bool;
  mutable stop_at_error : bool;
}
 
let arg = {
  rif = None;
  ec = "";
  cov = "";
  debug = false;
  reinit_cov = false;
  stop_at_error = false;
}

(**************************************************************************)
(* Cloned from the OCaml stdlib Arg module: I want it on stdout! (scrogneugneu) *)
let usage_out speclist errmsg =
  Printf.printf "%s" (Arg.usage_string speclist errmsg)

let usage = "Usage: \n\t" ^ 
  Sys.argv.(0) ^ " [options]* -ec <file>.ec <Rif File name to check>

 Performs post-mortem oracle checking using ecexe.

 The set  of oracle Inputs  should be included  in the set of  the RIF
 file inputs/outputs.

 At the first  run, the coverage information is  stored/updated in the
 coverage file  (cf the  -cov option to  set its name).  The variables
 declared in  this file should be  a subset of the  oracle outputs. If
 the coverage  file does not  exist, one is  is created using  all the
 oracle outputs. If  not all those outputs are  meaningfull to compute
 the coverage rate, one just need to delete corresponding lines in the
 coverage file.  The format of  the coverage file  is straightforward,
 but deserves respect.
 
 Options are:
" 

let rec speclist =
  [
    "-ec", Arg.String
      (fun str -> arg.ec <- str),
    "<string>\tec file name containing the RIF file checker (a.k.a., the oracle)" ;
    "-cov", Arg.String
      (fun str -> arg.cov <- str),
    "<string>\tOverride the default coverage file name (<oracle name>.cov by default).";

    "-reset-cov", Arg.Unit (fun _ -> (arg.reinit_cov <- true)), 
    "\treset the coverage rate (to 0%) before running";

    "-stop-at-error", Arg.Unit (fun _ -> (arg.stop_at_error <- true)), 
    "\tStop processing when the oracle returns false";

    "-debug", Arg.Unit (fun _ -> (arg.debug <- true)), 
    "\tset on the debug mode";

    "--help", Arg.Unit (fun _ -> (usage_out speclist usage ; exit 0)),
    "\tDisplay this list of options." ;
    "-help", Arg.Unit (fun _ -> (usage_out speclist usage ; exit 0)),
    "";
    "-h", Arg.Unit (fun _ -> (usage_out speclist usage ; exit 0)),
    ""
  ]

let _ =
  ( try Arg.parse speclist (fun str -> arg.rif <- Some str) usage
    with
	Failure(e) ->
	  output_string stderr e;
	  flush stderr;
	  exit 2
      | e ->
	  output_string stderr (Printexc.to_string e);
	  flush stderr;
	  exit 2
  );
  if arg.ec = "" then (
    output_string stderr "*** It is mandatory to set an oracle (cf -ec option)\n";
    Arg.usage speclist usage;
    flush stderr;
    exit 2)

(**************************************************************************)
let rif_ic : in_channel = 
  match arg.rif with
      Some f -> open_in f
    | None -> stdin

let rif_in, rif_out = RifIO.read_interface ~label:"check_rif " rif_ic  None
let rif_all = rif_in @ rif_out

(**************************************************************************)
(* Oracle launching *)
open RdbgPlugin
let oracle_in, oracle_out, kill_oracle, step_oracle, step_oracle_dbg = 
  let plugin = LustreRun.make_ec  arg.ec in
  plugin.inputs,plugin.outputs,plugin.kill,plugin.step,plugin.step_dbg

(**************************************************************************)
(* Check that the set of oracle Inputs is included in the set of the
   RIF file inputs/outputs.  *)
let _ =
  List.iter 
    (fun (n,t) -> 
       if not (List.mem (n,t) rif_all) then (
         let t = Data.type_to_string t in
         print_string ("The oracle input variable '"^n^"' (of type "^(t)^
                         ") is not present in the RIF data\n");
         print_string "Rif Inputs:\n";
         List.iter (fun (n,t) -> print_string ("\t"^n^":"^(Data.type_to_string t)^"\n")) rif_in;
         print_string "Rif Outputs:\n";
         List.iter (fun (n,t) -> print_string ("\t"^n^":"^(Data.type_to_string t)^"\n")) rif_out;
         flush stdout;
         exit 2
       )
    )
    oracle_in 

let rif_all_names = fst (List.split rif_all)
let oracle_names =  fst (List.split oracle_out)

(**************************************************************************)
(* Coverage stuff initilisation *)

let cov_init  = 
  let cov_file = if arg.cov = "" then 
    ((Filename.chop_extension arg.ec)^".cov")
  else 
    arg.cov
  in
    Coverage.init (List.tl oracle_names) cov_file arg.reinit_cov
        

(**************************************************************************)

let main () =
  let ok = ref true in
  let t = ref (Sys.time ()) in
  let i = ref 0 in
  let oracle_vals = ref [] in
  let cov = ref cov_init in
    
  let print_cov cov =
    let (to_cov, covered, cov_rate)= Coverage.compute_stat cov in
      if to_cov > 0 then (
        if cov_rate=100.0 then printf "\b";
        printf "\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b%6i   The coverage rate is %.1f%s" 
          !i  cov_rate "%"
      )
      else 
        printf "\b\b\b\b\b\b%6i" !i;
      flush stdout;  
  in
  let finish str = 
    flush_all();
    decr i;      
    print_cov !cov;
    Coverage.dump arg.ec (match arg.rif with Some f -> f | None -> "stdin")  !cov;

    if !ok then
      printf 
        "\nData in %s are ok with respect to %s\n"
        (match arg.rif with Some f -> f | None -> "stdin") arg.ec;
    close_in rif_ic; 
    kill_oracle str;
    print_string str;
    flush_all();
  in
    try
      printf "\nAnalysed step number:       " ;
      let rec loop () =
        let ti =  (Sys.time ()) in
          incr i;
          if ti -. !t > 0.2  then (t:=ti; print_cov !cov);
          (* lire le rif ... *)
          let in_vals : Data.subst list = RifIO.read ~pragma:["outs"] rif_ic None rif_all in
            (* ... et l'envoyer a l'oracle *)
            oracle_vals := step_oracle in_vals ;
            cov := Coverage.update_cov !oracle_vals !cov;
            match !oracle_vals with
              | [] -> assert false
              | (_, Data.B true)::_ -> loop ()
              | (_, Data.B false)::tail -> 
                  let msg = Coverage.dump_oracle_io in_vals tail !cov in
                  let msg = sprintf "\n*** The oracle returned false at step %i\n%s" !i msg in
                    ok := false;
                    if arg.stop_at_error then (
                      finish (msg)
                    ) else (
                      print_string msg;
                      loop ()
                    )
              | _ -> 
                  ok := false;
                  finish "*** Error: the oracle first output ougth to be a Boolean\n";
      in
        loop ()
    with 
      | End_of_file -> finish ""
      | RifIO.Bye ->  finish ""
      | e  ->
          finish ("\n"^(Printexc.to_string e)^"\n")


let _ = 
  main ()
