open GMain
open GEdit
open GtkEdit
open GtkList


(**************************************************************************)
(* Global vars *)

let debug = ref false
(* true    *)

let dbg_print msg =
  if !debug then
    (
      print_string msg;
      flush stdout
    )


let pid = ref 0
let lpid = ref 0
let opid = ref 0
let luc_pid = ref 0
let spid = ref 0
let gp_pid = ref 0

let rif_file = ref ""
let next_rif_file = ref ""

			   
(* Flag telling if any saved package need to be restored *)
let restore = ref false

(**************************************************************************)

(* use to perform system calls *)
let (my_create_process : string -> string list -> bool) =
  fun prog args -> 
    try
      let pid = 
	List.iter (fun x -> output_string stderr (x ^ " ")) (prog::args);
	output_string stderr "\n";
	flush stderr;
	Unix.create_process 
	  prog
	  (Array.of_list (prog::args))
	  (Unix.stdin)
	  (Unix.stdout)
	  (Unix.stderr)
      in
      let (_,status) = (Unix.waitpid [Unix.WUNTRACED] pid) in
	( match status with 
	      Unix.WEXITED i -> 
		if i = 0 || i = 1 then
		  (
		    output_string stderr ("     ... " ^ prog ^ " exited normally.\n");
		    flush stderr;
		    true
		  )
		else
		  (
		    output_string stderr ("*** Error: " ^ prog ^ " exited abnormally.\n");
		    flush stderr;
		    false
		  )
	    | _ -> 
	      output_string stdout ("*** " ^ prog ^ " exited ABnormally!\n");
		flush stdout;
		false
	)

    with e -> 
      output_string stdout (Printexc.to_string e);
      flush stdout;
      false

(************************************************************************)
(* nb: We use inet socket instead of pipes in order to remove the dependance
   with cygwin.
*)
let sock_io  = Unix.socket Unix.PF_INET Unix.SOCK_STREAM 0
let sock_err = Unix.socket Unix.PF_INET Unix.SOCK_STREAM 0

let sock_addr =
  let my_name = "127.0.0.1" in
(*   let my_name = Unix.gethostname() in *)
  let my_entrie_byname =
    try 
      Unix.gethostbyname my_name 
    with Not_found -> 
      print_string "host not found (gethostbyname failed).\n";
      assert false
  in
      my_entrie_byname.Unix.h_addr_list.(0)
 
(* 
   Regarder Unix.open_connection, Unix.establish_server, Unix.shutdown_connection
   et eventuellement s'en inspirer...
*)

let sock_addr_str = Unix.string_of_inet_addr sock_addr

(* let _ =  *)
(*   print_string ("sockadr = " ^ sock_addr_str ^ ".\n"); *)
(*   flush stdout *)

let (socket_port_io, socket_port_err) = 
  let rec (bind_with_free_port : Unix.file_descr -> Unix.inet_addr -> int -> int
	     -> int) =
    fun s saddr port maxtry -> 
      if maxtry = 0 then 
	failwith "xlurette: socket bind failure"
      else
	try
	  Unix.bind s (Unix.ADDR_INET(saddr, port));
	  port
	with _ -> 
	  bind_with_free_port s saddr (port+1) (maxtry-1) 
  in
  let port_io  = bind_with_free_port sock_io  sock_addr 2000 10000 in
  let port_err = bind_with_free_port sock_err sock_addr (port_io+1) 10000 in
    (port_io, port_err) 
    
let hosttype = try Unix.getenv "HOSTTYPE" with _ -> "" 

let _ =
  let lurettetop = 
    if 
      hosttype = "win32" || hosttype = "cygwin" 
    then 
      "lurettetop_exe" 
    else 
      "lurettetop" 
  in
  let args =  [lurettetop;
	       "--socket-inet-addr"; sock_addr_str;
	       "--socket-io-port"; string_of_int socket_port_io;
	       "--socket-err-port"; string_of_int socket_port_err
	      ] 
  in
    (
      if !debug then
	(
	  List.iter 
	   (fun x -> output_string stdout (x ^ " ")) 
	   (args);
	  output_string stdout "\n";
	  flush stdout;
	);
      try
	pid := Unix.create_process (lurettetop) (Array.of_list args)
	  Unix.stdin Unix.stdout Unix.stderr
      with _ -> 
	output_string stdout (
	  "*** can not find lurettetop. Is your path set properly?\n" ^ 
	  "*** It should contain the lurette/<arch>/bin/ directory\n");
	exit 1
    )

let (connected_io_sock, _) = 
  Unix.listen sock_io 1;
  Unix.accept sock_io (* bloquant *)

let ic = Unix.in_channel_of_descr connected_io_sock
let oc = Unix.out_channel_of_descr connected_io_sock

(* let _ = *)
(*   (* so that lurettetop can issue a connect on sock_err *) *)
(*   let res = input_line ic in *)
(*     if res = "hello!" then  *)
(*       ( *)
(* 	print_string "xlurette: connecting...\n"; *)
(* 	flush stdout *)
(*       )  *)
(*     else *)
(*       ( *)
(* 	print_string "xlurette: socket connection error.\n"; *)
(* 	flush stdout; *)
(* 	exit 1 *)
(*       ) *)

let _ = dbg_print  "xlurette: socket connecting...!\n"

let (connected_err_sock, _) = 
  output_string oc "hello.\n";
  flush oc;
  Unix.listen sock_err 1;
  Unix.accept sock_err (* bloquant *)

let _ = dbg_print "xlurette: socket connected!\n"

(* So that socket reading is non-blocking 
   has no effect under win32, why ?
*)
let _ = 
  try 
    Unix.setsockopt_float connected_io_sock  Unix.SO_RCVTIMEO 0.00001;
    Unix.setsockopt_float connected_err_sock Unix.SO_RCVTIMEO 0.00001
  with e -> 
    print_string ((Printexc.to_string e) ^ "\n"); 
    flush stdout
    
let _ = dbg_print "xlurette: Unix.setsockopt_float \n"

let ec = Unix.in_channel_of_descr connected_err_sock

exception Niet

(* Non-blocking socket reading *)
(* XXX does not work very well... *)
let (read_sock : string -> Unix.file_descr -> in_channel -> string) = 
  fun label sock ic -> 
  if 
    hosttype = "win32" ||  hosttype = "sparc-sun"
  then
    match Unix.select [sock] [] [] 0.001 with 
      | [],_,_ -> raise Niet  
      |  _,_,_ -> 
	   let str = 
	     try input_line ic
	     with e -> 
	       print_string ("\n!!! " ^(Printexc.to_string e) ^"!!!\n"); 
	       flush stdout;
(* 	       raise Niet   *)
	       assert false 
	   in
(* 	     print_string (label ^ " xlurette sock: " ^ str ^ "\n"); *)
(* 	     flush stdout; *)
	     str
  else
     try input_line ic with _ -> raise Niet  
  

let nb_read_ltop () = read_sock "stdout" connected_io_sock ic 
let nb_read_err_ltop () = read_sock "stderr" connected_err_sock ec   

(**************************************************************************)
      
type running_modeT = 
    Compile of string * string * string * string * string 
  | Pipe of string * string * string
  | NoMode

let running_mode = ref NoMode
  
let sut_node_present = ref true
let oracle_node_present = ref true
let oracle_compiler_present = ref true
let user_seed_present = ref true
let vertices_nb_present = ref true
let user_seed = ref 0

(**************************************************************************)

let remove_control_m str =
  Str.global_replace (Str.regexp "\013") "" str

let (readfile: string -> string) = 
  fun name -> 
    let chanin = open_in_bin name in 
    let len = 1024 in 
    let s = String.create len in 
    let buf = Buffer.create len in 
    let rec iter () = 
      try 
	let n = input chanin s 0 len in 
	  if n = 0 then 
            () 
	  else 
            ( 
              Buffer.add_substring buf s 0 n; 
              iter () 
            ) 
      with 
	  End_of_file -> () 
    in 
    let str =
      iter (); 
      close_in chanin; 
      Buffer.contents buf 
    in
      remove_control_m str

(* let (readfile: string -> string) =  *)
(*   fun file -> *)
(*     try *)
(*       let rec (readfile_ic : in_channel -> string -> string) =  *)
(* 	fun ic acc -> *)
(* 	  let line =  *)
(* 	    try (input_line ic)  *)
(* 	    with End_of_file -> (close_in ic) ; "end_of_file"  *)
(* 	  in *)
(* 	    if (line = "end_of_file")  *)
(* 	    then acc  *)
(* 	    else (readfile_ic ic (acc ^ "\n" ^ line)) *)
(*       in  *)
(*       let ic = (open_in file) in *)
(*       let str = readfile_ic ic "" in *)
(* 	close_in ic; *)
(* 	str *)
(*     with *)
(* 	_ ->  *)
(* 	  ( *)
(* 	    output_string stderr ( *)
(* 	      "File " ^ file ^ " could not be opened.\n"); *)
(* 	    raise Not_found *)
(* 	  ) *)


let (cygpath_w2u : string -> string) =
  (* try to convert a windows path to a unix one. Do nothing if it fails *)
  fun str0 ->
    let str = String.escaped str0 in
      try
	let i0 = Str.search_forward (Str.regexp ":") str 0 in
	let l = String.length str in
	let str_head = String.sub str 0 i0 in
	let str_tail = String.sub str (i0+1) (l-i0-1) in
	let str_tail_new = Str.global_replace (Str.regexp "\\\\") "/" str_tail in
(* 	  print_string ("str_head = " ^ str_head ^ "\n"); *)
(* 	  print_string ("str_tail = " ^ str_tail ^ "\n"); *)
(* 	  print_string ("str_tail_new = " ^ str_tail_new ^ "\n"); *)
	  ("/cygdrive/" ^ str_head ^ str_tail_new) 
      with  _ -> 
(* 	output_string stderr ("cygpath_w2u " ^ str0 ^ " failed.\n"); *)
	str0

type tok = Genlex.token Stream.t

let ok_str = "   OK!
"


(**************************************************************************)
let chop_ext file = 
  try Filename.chop_extension file 
  with _ -> file



let (give_fresh_file_name : string -> string -> string) =
  fun prefix suffix -> 
    let rec (give_fresh_file_name_aux : int -> string) =
      fun i -> 
	let file = (prefix ^ (string_of_int i) ^ suffix) in
	  if 
	    try Sys.file_exists file with _ -> assert false
	  then 
	    give_fresh_file_name_aux (i+1)
	  else
	    file
    in
      give_fresh_file_name_aux 1

let (update_next_rif_file_name : string -> string -> string -> string -> unit) =
  fun rif_base_name0 sut_str0 env_str0 l_str0 ->
    let rif_base_name = 
      if 
	rif_base_name0 <> ""
      then
	rif_base_name0
      else 
	(* 
	   invent a fresh file name, using the provided in rif_base_name0 if not 
	   empty, the 3 other strings otherwise 
	*)
	let str_list0 = Str.split (Str.regexp " ") env_str0 in
	let str_list  = List.map (Filename.basename) str_list0 in
	let env_str = 
	  (* when several env are involved, this string contains blank char *)
	  String.concat "-" (List.map (chop_ext) str_list)
	in
	let l_str = Filename.basename l_str0 in
	let sut_str = Filename.basename sut_str0 in
	  (* We invent a name if no one is provided *)
	  Filename.concat "Data" (sut_str ^ "-" ^ env_str ^ "-" ^ l_str ^ "-")
    in
      next_rif_file := (give_fresh_file_name rif_base_name ".rif")



(**************************************************************************)

let pre_sut_name = ref ""
let pre_sut_node = ref ""
let pre_oracle_name = ref ""
let pre_oracle_node = ref ""

let (remove_extension : string -> string) =
  fun str -> 
    let file_ext = Filename.basename str
    and dir  = Filename.dirname str in
    let file = try Filename.chop_extension file_ext with _ -> file_ext in
      if dir = "." then file else (Filename.concat dir file)


(**************************************************************************)

(** returns the list of lines in ic *)
let rec (readfile_line : in_channel -> string list -> string list) = 
  fun ic acc ->
    let line = 
      try (input_line ic) 
      with End_of_file -> (close_in ic) ; "end_of_file" 
    in
      if (line = "end_of_file") 
      then acc 
      else (readfile_line ic (line::acc))


	  
(** returns the list of files specified by the filter *)
let (get_files_list_filtered : string list -> string list -> string list) =
  fun filters init -> 
    let rec get_all_files dh acc =
      try
	let f = Unix.readdir dh in
	  get_all_files dh (f::acc)
      with End_of_file -> 
	Unix.closedir dh;
	acc
    in
    let dh = Unix.opendir (Unix.getcwd ()) in
    let curdir_files = get_all_files dh [] in
      (* in order to also show the rif files in the Data dir *)
    let dh = 
      try
	Unix.opendir (Filename.concat (Unix.getcwd ()) "Data") 
      with
	  _  -> 
	    Unix.mkdir (Filename.concat (Unix.getcwd ()) "Data") 0o755 ;
	    Unix.opendir (Filename.concat (Unix.getcwd ()) "Data")
    in
    let datadir_files = (List.map (Filename.concat "Data") (get_all_files dh [])) in
    let all_files = curdir_files @ datadir_files in
    let is_matching str filter =
      let sl = Str.split (Str.regexp (Str.quote "*")) filter in
	try
	  let _ = 
	    List.fold_left 
	      (fun i s -> Str.search_forward (Str.regexp s) str i) 
	      0 
	      sl
	  in
	    true
	with 
	    Not_found -> false
    in
    let one_is_matching str = List.exists (is_matching str) filters in
      (List.filter (one_is_matching) all_files) @ init


(** returns the extension of a filename *)
let (get_extension : string -> string) =
  fun file -> 
    let base = Filename.basename file in
      try
	let i = String.index base '.' in
	  String.sub base (i) ((String.length base) - i)
      with Not_found -> ""
	
let _ = assert ((get_extension "/home/jahier/toto.ml") = ".ml")

(**************************************************************************)
(**************************************************************************)
class customized_callbacks = object(self)
  inherit Xlurette_glade_callbacks.default_callbacks
    

  method put str = 
    self#top_xlurette#output_window#insert str;
    self#top_xlurette#tdm_output_window#insert str
  method put_misc str = 
    self#top_xlurette#error_window#insert str;
    self#top_xlurette#tdm_error_window#insert str

  (** returns the list of nodes of a lustre or saofdm file *)
  method nodes_info file =
(* (nodes_info : string -> string list) = *)
    if hosttype = "win32" then 
      (* XXX *)
      [] 
    else
      let file_info = Filename.temp_file (Filename.basename file) "nodes_info" in 
      let files =
	try
	  if 
	    Filename.check_suffix file ".saofdm"
	  then
	    let cmd = ("getsaonodes " ^ file ^ " > " ^ file_info) in
	      if 
		try Sys.file_exists file  with _ -> assert false
	      then
		(
		  if 
		    try ((Sys.command cmd) <> 0) with _ -> assert false
		  then 
		    (
		      self#put_misc cmd ;
		      self#put_misc ("\n*** getsaonodes failed.\n" ^ cmd ^ "\n");
		      []
		    )
		  else
		    (
		    let ic = open_in file_info in
		    let nodes = readfile_line ic [] in
		      close_in ic;
		      nodes
		    )
		)
	      else 
		[]
	  else if 
	    Filename.check_suffix file ".lus"
	  then
	    let cmd = ("lusinfo " ^ file ^ " nodes > " ^ file_info) in
	      if 
		try Sys.file_exists file  with _ -> assert false
	      then
		(
		  if 
		    try ((Sys.command cmd) <> 0) with _ -> assert false
		  then 
		    (
		      self#put_misc 
		       ("Warning: lusinfo failed.\n  " ^ cmd ^ "\n" ^
			"  Probably this lustre file is not an academic " ^ 
			"lustre file, but a lustre file generated by scade.\n");
		      flush stderr;
		      []
		    )
		  else
		    (
		    let ic = open_in file_info in
		    let nodes = readfile_line ic [] in
		      close_in ic;
		      nodes
		    )
		)
	      else 
		[]
	  else
	    []
	with 
	    _ -> []
      in
	if 
	  try Sys.file_exists file_info  with _ -> assert false
	then 
	  (try Sys.remove file_info with _ -> assert false);
	files

  method read_ltop_stdout () =
    let length = 
      try
	float_of_string self#top_set_parameters_window#test_length#text 
      with _ -> 
	(* such a failure happens when the user removes the whole 
	   test length (before typing a new one). *)
	1.0
    in
    let 
      display = self#put
    and 
      display_misc = self#put_misc
    and
      progress =
      try
      ( fun f ->
	  let p = f /. length in 
	    (* 	  let pi = int_of_float (p0 *. 100.) in *)
	    self#top_xlurette#step_number_label#set_text
	      (
		(* 		(string_of_int pi) ^ " % " ^ *)
	        " \n(" 
		^ (string_of_int (int_of_float f))
		^ "/" ^ (string_of_int (int_of_float length)) ^ ")");
	    self#top_xlurette#progressbar#set_percentage p )
      with _ -> assert false
    in

      ( try    
	  while true do
	    let str = remove_control_m (nb_read_ltop ()) in
              dbg_print (str ^ "\n");
	      
	      if 
		str = "" 
	      then 
		()
	      else if
		str = " " 
	      then 
		display "\n"
	      else if 
		str = "One more loop ? [type 's' to stop, `CR' to continue, or an integer to change the number of steps to skip.]"
	      then
		display (
		  "One more loop ?
      click on the step (resp. stop) button to continue (resp. stop)\n")
	      else if 
		str = "  Type h for help, or man for a small user manual."
	      then
		()
(* 		display "No program is currently running.\n" *)

	      else if 
		String.length str > 22 
		&& try String.sub str 0 22 = "The oracle Pid is     " with _ -> assert false
	      then
		let oracle_pid = 
		  try String.sub str 22 ((String.length str) - 22) with _ -> assert false
		in
		  try
		    opid := int_of_string oracle_pid;
		    ()
		  with _ -> display "Bad oracle PID.\n"


	      else if 
		String.length str > 17 
		&& try String.sub str 0 17 = "Launching gnuplot" with _ -> assert false
	      then
		display_misc (str ^ "\n")

	      else if 
		String.length str > 22 
		&& try String.sub str 0 22 = "The Pid of lurette is "with _ -> assert false
	      then
		let lurette_pid = 
		  try String.sub str 22 ((String.length str) - 22) with _ -> assert false
		in
		  try		   
		    lpid := int_of_string lurette_pid;
		    ()
		  with _ -> display "Bad lurette PID.\n"

		  
	      else if 
		String.length str > 22 
		&& try String.sub str 0 22 = "The Pid of lucky is   " with _ -> assert false
	      then
		let lucky_pid = 
		  try String.sub str 22 ((String.length str) - 22) with _ -> assert false
		in
		  try
		    luc_pid := int_of_string lucky_pid 
		  with _ -> display "Bad lucky PID.\n"


	      else if 
		String.length str > 43 
		&& 
		String.sub str 0 43 = "The generated data can be found in the file" 
	      then
		let data_file = String.sub str 44  ((String.length str) - 44) in
		  rif_file := data_file;
		  self#top_xlurette#rif_file#entry#set_text data_file
		  


	      else if 
		String.length str > 22 
		&& try String.sub str 0 22 = "The Pid of the sut is " with _ -> assert false
	      then
		let sut_pid = 
		  try String.sub str 22 ((String.length str) - 22) with _ -> assert false
		in
		  try
		    spid := int_of_string sut_pid
		  with _ -> display "Bad sut PID.\n"
		  
	      else if 
		String.length str > 36 
		&& 
		try String.sub str 0 34 = "The random engine was initialized "
		with _ -> assert false
	      then
		let seed_str = 
		  try String.sub str 48 ((String.length str) - 48) with _ -> assert false
		in
		  try
		    self#set_user_seed (int_of_string seed_str);
		    if
		      not self#top_set_parameters_window#radiobutton_verbose_0#active
		    then
		      display (str ^ "\n");

		  with e -> 
		    display (Printexc.to_string e);
		    display ("\n" ^ seed_str ^ " is a Bad seed.\n")

	      else if
		str = " ==> The test completed; no property has been violated."
	      then
		display (ok_str ^ str ^ "\n")
	      else if 
		String.length str > 7 && try String.sub str 0 8 = "<lurette" with _ -> assert false
	      then
		()
	      else if 
		String.length str > 18 && 
		try String.sub str 0 19 = "warning unused code" with _ -> assert false
	      then
		display_misc (str ^ "\n")
	      else if 
		((String.length str > 9) && 
		 try ((String.sub str 0 9) = "--- step ") with _ -> assert false
		)
	      then
		(
		  let dot_index = String.index str ':' in
		  let step_str = 
		    try String.sub str 9 (dot_index-9) with _ -> assert false
		  in
		  let stepf = float_of_string step_str in
		    progress stepf;
		    if self#top_set_parameters_window#radiobutton_verbose_3#active then
		      display (str ^ "\n")
		)
	      else
		display (str ^ "\n")
	  done
	with
	    Niet -> ()
	  | Sys_error _ -> assert false
	  | e -> 
	      print_string (Printexc.to_string e);
	      flush stdout;
	      assert false
      ); 
      true  


  method read_ltop_stderr () =
    ( try    
	while true do
	  let str = nb_read_err_ltop () in	    
	    self#top_xlurette#tdm_error_window#insert (str ^ "\n");
	    self#top_xlurette#error_window#insert (str ^ "\n")
	done
      with 
	  Niet -> ()
    ); 
    true  


  method show_step_window () = 
    self#top_step_by_step_window#step_by_step_window#show ()
      

(**************************************************************************)
  method add_sut_node () =
    if not !sut_node_present then
      (
	self#top_xlurette#sut_hbox#add 
	 self#top_xlurette#sut_node#coerce;
	sut_node_present := true
      )


  method rm_sut_node () =
    if !sut_node_present then
      (
	self#top_xlurette#sut_hbox#remove 
	 self#top_xlurette#sut_node#coerce;
	sut_node_present := false
      )


  method add_oracle_node () =
    if not !oracle_node_present then
      (
	self#top_xlurette#oracle_hbox#add 
	 self#top_xlurette#oracle_node#coerce;
	oracle_node_present := true
      )

  method rm_oracle_node () =
    if !oracle_node_present then
      (
	self#top_xlurette#oracle_hbox#remove 
	 self#top_xlurette#oracle_node#coerce;
	oracle_node_present := false
      )

  method add_oracle_compiler () =
    if not !oracle_compiler_present then
      (
	self#top_xlurette#oracle_hbox#add 
	 self#top_xlurette#oracle_compiler#coerce;
	oracle_compiler_present := true
      )

  method rm_oracle_compiler () =
    if !oracle_compiler_present then
      (
	self#top_xlurette#oracle_hbox#remove 
	 self#top_xlurette#oracle_compiler#coerce;
	oracle_compiler_present := false
      )

  method add_user_seed () =
    if not !user_seed_present then
      (
	self#top_set_parameters_window#random_seed_hbox#add 
	 self#top_set_parameters_window#user_seed#coerce;
	user_seed_present := true ;
	self#set_user_seed !user_seed
      )

  method rm_user_seed () =
    if !user_seed_present then
      (
	self#top_set_parameters_window#random_seed_hbox#remove 
	 self#top_set_parameters_window#user_seed#coerce;
	user_seed_present := false
      )

  method add_vertices_nb () =
    if not !vertices_nb_present then
      (
	self#top_set_parameters_window#dnv_hbox#add 
	 self#top_set_parameters_window#vertices_nb#coerce;
	vertices_nb_present := true
      )

  method rm_vertices_nb () =
    if !vertices_nb_present then
      (
	self#top_set_parameters_window#dnv_hbox#remove 
	 self#top_set_parameters_window#vertices_nb#coerce;
	vertices_nb_present := false
      )

  (**************************************************************************)


  method set_pipe_mode () =
    match !running_mode with
      | Compile(sname, snode, oname, onode, ename) -> 
	  (
	    let sut_name = self#get_sut () 
	    and oracle_name = self#get_oracle ()
	    and env_name = self#get_env ()
	    in
	      running_mode := Pipe(sut_name, oracle_name, env_name);
	      self#rm_sut_node ();
	      self#rm_oracle_compiler ();
	      self#rm_oracle_node ();

	      (* restore the previous values *)
	      self#set_sut sname ;
	      self#set_sut_node snode ;
	      self#set_env ename ;
	      self#set_oracle oname ;
	      self#set_oracle_node onode  ;
	      self#update_sut_and_oracle_combo ()
	  )
      | Pipe(_,_,_) -> ()
      | NoMode ->
	  let sut_name = self#get_sut () 
	  and oracle_name = self#get_oracle ()
	  and env_name = self#get_env ()
	  in
	    running_mode := Pipe(sut_name, oracle_name, env_name);
	    self#rm_sut_node ();
	    self#rm_oracle_compiler ();
	    self#rm_oracle_node (); 
	    self#update_sut_and_oracle_combo ()

  method set_compiler_mode () =
    match !running_mode with
      | Pipe(sname, oname, ename) -> 
	  (
	    let sut_name = self#get_sut () 
	    and sut_node = self#get_sut_node ()
	    and oracle_name = self#get_oracle ()
	    and oracle_node = self#get_oracle_node ()
	    and env_name = self#get_env ()
	    in
	      running_mode := Compile(sut_name, sut_node, oracle_name, oracle_node, env_name);
	      self#top_xlurette#sut_hbox#remove self#top_xlurette#sut_compiler#coerce;
	      
	      self#add_sut_node ();
	      self#top_xlurette#sut_hbox#add self#top_xlurette#sut_compiler#coerce;
	      self#add_oracle_node ();
	      self#add_oracle_compiler ();

	      (* restore the previous values *)
	      self#set_sut sname ;
	      self#set_oracle oname ;
	      self#set_env ename ;
	      self#update_sut_and_oracle_combo ()

	  )
      | Compile(_,_,_,_,_) -> ()
      | NoMode -> 
	  let sut_name = self#get_sut () 
	  and sut_node = self#get_sut_node ()
	  and oracle_name = self#get_oracle ()
	  and oracle_node = self#get_oracle_node ()
	  and env_name = self#get_env ()
	  in
	    running_mode := 
	    Compile(sut_name, sut_node, oracle_name, oracle_node, env_name);
	    self#top_xlurette#sut_hbox#remove self#top_xlurette#sut_compiler#coerce;
	    
	    self#add_sut_node ();
	    self#top_xlurette#sut_hbox#add self#top_xlurette#sut_compiler#coerce;
	    self#add_oracle_node ();
	    self#add_oracle_compiler ();
	    self#update_sut_and_oracle_combo ()
	      


  method on_sut_compiler_entry_changed () = 
    if (self#get_sut_compiler ()) = "stdin/stdout" then
      self#set_pipe_mode ()
    else
      self#set_compiler_mode ();
    self#update_sut_and_oracle_combo ()

  method on_oracle_compiler_entry_changed () = 
    self#update_sut_and_oracle_combo ()
	

  method on_user_seed_changed () =
    user_seed := 
    try
      int_of_string self#top_set_parameters_window#user_seed#text
    with _ -> 1

  method on_radiobutton_draw_all_vertices_toggled () =
    if self#top_set_parameters_window#radiobutton_draw_all_vertices#active then
      self#rm_vertices_nb ()
    else
      self#add_vertices_nb ()


  method on_radiobutton_user_seed_clicked () =
    if self#top_set_parameters_window#radiobutton_user_seed#active then
      self#add_user_seed ()
    else
      self#rm_user_seed ()
	
  method on_radiobutton_verbose_3_clicked () =
    if self#top_set_parameters_window#radiobutton_verbose_3#active then
      debug := true
    else
      debug := false
      
  method on_clean_up_button_clicked () = 
    output_string oc ("clean \n");
    flush oc;
    self#top_xlurette#output_window#insert "Clean-up temporary files ...\n"

  method on_help_button_clicked () = 
    let pdf_viewer = 
      try Sys.getenv "PDF_VIEWER"  with _ -> assert false
    in
    let lpath = 
      try Sys.getenv "LURETTE_PATH"  with _ -> assert false
    in 
    let cmd = (pdf_viewer ^ " " ^ lpath ^ "/doc/lurette-man.pdf") in 
      if 
	try Sys.command cmd <> 0  with _ -> assert false
      then  
	self#top_xlurette#output_window#insert (cmd ^ " failed\n") 



  method on_lucky_help_button_clicked () = 
    let lpath = Sys.getenv "LURETTE_PATH" in
    let pdf_viewer = Sys.getenv "PDF_VIEWER" in
    let cmd = (pdf_viewer ^ " " ^ lpath ^ "/doc/lucky-man.pdf") in
      if Sys.command cmd <> 0 then 
	self#top_xlurette#output_window#insert (cmd ^ " failed\n")



  (* set parameters window *)
  method on_set_parameters_clicked () = 
    self#top_set_parameters_window#set_parameters_window#show ();

  method on_close_parameters_window_clicked () = 
    self#top_set_parameters_window#set_parameters_window#misc#hide ()

  (* sut help window *)
  method on_sut_help_button_clicked () = 
    self#top_sut_help_window#sut_help_window#show ()

  method ok_sut_help_clicked () = 
    self#top_sut_help_window#sut_help_window#misc#hide ()

  (* env help window *) 
  method on_env_help_button_clicked () =  
    self#top_env_help_window#env_help_window#show () 
      
  method ok_env_help_clicked () =  
    self#top_env_help_window#env_help_window#misc#hide () 
      
  (* oracle help window *) 
  method on_oracle_help_button_clicked () =  
    self#top_oracle_help_window#oracle_help_window#show () 
      
  method ok_oracle_help_clicked () =  
    self#top_oracle_help_window#oracle_help_window#misc#hide () 

  (* test_thickness help window *) 
  method on_test_thickness_help_button_clicked () =  
    self#top_test_thickness_help_window#test_thickness_help_window#show () 

  method ok_test_thickness_help_clicked () =  
    self#top_test_thickness_help_window#test_thickness_help_window#misc#hide () 



  (* set env var window *) 
  method on_extra_var_button_clicked () =
    self#top_extra_env_var_window#extra_env_var_window#misc#show () 
  method on_quit_extra_var_window_clicked () =
    self#top_extra_env_var_window#extra_env_var_window#misc#hide () 


  (* sut file selection window *)
  method browse_sut_clicked () = 
    self#top_fileselection_sut#fileselection_sut#show ()


  method change_dir dir =
    let cmd = ("change_dir \"" ^ (String.escaped dir) ^ "\"\n") in
      if !debug then (
	(output_string stderr cmd);
	flush stderr
      );
      output_string oc cmd;
      flush oc;
      (try Sys.chdir dir  with _ -> assert false);
      self#put ("The current directory for lurette is now " ^ dir ^ ".\n");
      self#update_sut_and_oracle_combo (); 
      self#update_lucky_files_combo ()




  method ok_sut_clicked () = 
    let sut_file = cygpath_w2u self#top_fileselection_sut#fileselection_sut#get_filename in
    let dir = Filename.dirname sut_file 
    and basename = Filename.basename sut_file in 
      if !debug then (
	output_string stderr ("sut_file = " ^ sut_file ^ "\n");
	output_string stderr ("dir = " ^ dir ^ "\n");
	output_string stderr ("basename = " ^ basename ^ "\n");
	flush stderr;
      );
      self#top_xlurette#sut_name#entry#set_text basename ;
      self#change_dir dir;
      self#top_fileselection_sut#fileselection_sut#misc#hide ()

  method cancel_sut_clicked () = 
    self#top_fileselection_sut#fileselection_sut#misc#hide ()


  (* oracle file selection window *)
  method browse_oracle_clicked () = 
    self#top_fileselection_oracle#fileselection_oracle#show ()

  method ok_oracle_clicked () = 
    let oracle_file = self#top_fileselection_oracle#fileselection_oracle#get_filename in
      self#top_xlurette#oracle_name#entry#set_text oracle_file ;
      self#top_fileselection_oracle#fileselection_oracle#misc#hide ()

  method cancel_oracle_clicked () = 
    self#top_fileselection_oracle#fileselection_oracle#misc#hide ()

  (* env file selection window *)
  method browse_env_clicked () = 
    self#top_fileselection_env#fileselection_env#show ()

  method add_env_clicked () = 
    self#top_fileselection_add_env#fileselection_add_env#show ()

  method ok_env_clicked () = 
    let env_file = self#top_fileselection_env#fileselection_env#get_filename in
      self#top_xlurette#env_name#entry#set_text  env_file ;
      self#top_fileselection_env#fileselection_env#misc#hide ()

  method ok_add_env_clicked () = 
    let env_file = 
      self#top_fileselection_add_env#fileselection_add_env#get_filename 
    in
      self#top_xlurette#env_name#entry#set_text 
	(self#top_xlurette#env_name#entry#text ^ " " ^ env_file) ;
      self#top_fileselection_add_env#fileselection_add_env#misc#hide ()

  method cancel_env_clicked () = 
    self#top_fileselection_env#fileselection_env#misc#hide ()

  method cancel_add_env_clicked () = 
    self#top_fileselection_add_env#fileselection_add_env#misc#hide ()


  method close_step_button_clicked () = 
    self#top_step_by_step_window#step_by_step_window#misc#hide ()


  (* EXTRA_CFILE selection window *)
  method add_extra_cfiles_clicked () = 
    self#top_fileselection_extra_cfiles#fileselection_extra_cfiles#show ()

  method ok_extra_cfiles_clicked () = 
    let cfile = self#top_fileselection_extra_cfiles#fileselection_extra_cfiles#get_filename in
      self#top_extra_env_var_window#extra_cfiles_entry#set_text (cfile ^ "  " ^
								 ((self#top_extra_env_var_window#extra_cfiles_entry#text))) ;
      self#top_fileselection_extra_cfiles#fileselection_extra_cfiles#misc#hide ()

  method cancel_extra_cfiles_clicked () = 
    self#top_fileselection_extra_cfiles#fileselection_extra_cfiles#misc#hide ()

  (* EXTRA_LIBS selection window *)
  method add_extra_libs_clicked () = 
    self#top_fileselection_extra_libs#fileselection_extra_libs#show ()

  method ok_extra_libs_clicked () = 
    let cfile = self#top_fileselection_extra_libs#fileselection_extra_libs#get_filename in
      self#top_extra_env_var_window#extra_libs_entry#set_text (cfile ^ "  -l" ^
							       ((self#top_extra_env_var_window#extra_libs_entry#text)  )) ;
      self#top_fileselection_extra_libs#fileselection_extra_libs#misc#hide ()

  method cancel_extra_libs_clicked () = 
    self#top_fileselection_extra_libs#fileselection_extra_libs#misc#hide ()

  (* EXTRA_LIBDIRS selection window *)
  method add_extra_libdirs_clicked () = 
    self#top_fileselection_extra_libdirs#fileselection_extra_libdirs#show ()

  method ok_extra_libdirs_clicked () = 
    let cfile = self#top_fileselection_extra_libdirs#fileselection_extra_libdirs#get_filename in
      self#top_extra_env_var_window#extra_libdirs_entry#set_text (cfile ^ "    -L" ^ 
								  ((self#top_extra_env_var_window#extra_libdirs_entry#text))) ;
      self#top_fileselection_extra_libdirs#fileselection_extra_libdirs#misc#hide ()

  method cancel_extra_libdirs_clicked () = 
    self#top_fileselection_extra_libdirs#fileselection_extra_libdirs#misc#hide ()

  (* EXTRA_INCLUDEDIRS selection window *)
  method add_extra_includedirs_clicked () = 
    self#top_fileselection_extra_includedirs#fileselection_extra_includedirs#show ()

  method ok_extra_includedirs_clicked () = 
    let cfile = self#top_fileselection_extra_includedirs#fileselection_extra_includedirs#get_filename in
      self#top_extra_env_var_window#extra_includedirs_entry#set_text (cfile ^ "   -I" ^ 
								      ((self#top_extra_env_var_window#extra_includedirs_entry#text))) ;
      self#top_fileselection_extra_includedirs#fileselection_extra_includedirs#misc#hide ()

  method cancel_extra_includedirs_clicked () = 
    self#top_fileselection_extra_includedirs#fileselection_extra_includedirs#misc#hide ()



  (* Show lucky ps *)
  method show_env_button_clicked () = 
    let env = self#top_xlurette#env_name#entry#text in
      if get_extension env = ".lut" 
      then (
        output_string stdout "Cannot display lutin programs.\n"
      )
      else 
        let cmd_show = ("set_env \"" ^ env ^ "\"\n show \n") in
        let all_cmds = (self#get_all_cmds ()) in
          output_string oc all_cmds ;
          output_string oc cmd_show ;
          if !debug then (output_string stderr cmd_show);
          flush oc
    
      
  (* quit window *)

  (* to quit xlurette *)
  method quit () = 
    output_string oc " quit\n"; (* to kill lurettetop *)
    flush oc;
    prerr_endline "xlurette: bye! " ;
    flush stdout;
    
    let (pid',_) = (Unix.waitpid [Unix.WUNTRACED] !pid) in
      if pid' <> 0 then (try Unix.kill (!pid) Sys.sigkill with _ -> () ); 

      (* let's make sure everyone is dead *)
      if !luc_pid <> 0 then (try Unix.kill (!luc_pid) Sys.sigkill with _ -> () );
      if !spid <> 0 then (try Unix.kill (!spid) Sys.sigkill with _ -> () );
      if !lpid <> 0 then (try Unix.kill (!lpid) Sys.sigkill with _ -> () );
      if !opid <> 0 then (try Unix.kill (!opid) Sys.sigkill with _ -> () );
      exit 0 

  method on_quit_xlurette_button_clicked () =
    self#top_quit_window#quit_window#show () 

  method on_quit_yes_clicked () =
    self#quit ()

  method on_quit_no_clicked () =	
    self#top_quit_window#quit_window#misc#hide ()


      
  method step () = 
    if 
      self#top_set_parameters_window#radiobutton_step_yes#active
    then
      (
	let cmd_step = (self#top_step_by_step_window#skip_step_number#text ^ "\n") in
	  if !debug then (output_string stderr cmd_step);
	  output_string oc cmd_step;
	  flush oc
      )
    else
      self#top_xlurette#output_window#insert 
	"Step button unactive because step-by-step mode is off\n"

  method stop_run () = 
    self#top_xlurette#output_window#insert "Stopping the current run.\n";

    if !opid <> 0 then 
      (
	self#top_xlurette#output_window#insert 
	 ("Killing process " ^ string_of_int !opid ^ "\n");
	try 
	  Unix.kill (!opid) Sys.sigint 
	    (* 	  Unix.kill (!opid) Sys.sigkill  *)
	with _ -> () 
      );

    if !lpid <> 0 then 
      (
	self#top_xlurette#output_window#insert 
	 ("Killing process " ^ string_of_int !lpid ^ "\n");
	try 
	  Unix.kill (!lpid) Sys.sigkill ;
	with _ -> () ); 

    if !luc_pid <> 0 then 
      (
	self#top_xlurette#output_window#insert 
	 ("Killing process " ^ string_of_int !luc_pid ^ "\n");
	try 
	  Unix.kill (!luc_pid) Sys.sigint
	    (* 	  Unix.kill (!luc_pid) Sys.sigkill  *)
	with _ -> () 
      );
    if !spid <> 0 then 
      (
	self#top_xlurette#output_window#insert 
	 ("Killing process " ^ string_of_int !spid ^ "\n");
	try Unix.kill (!spid) Sys.sigkill with _ -> () 
      ); 
    opid := 0;
    lpid := 0;
    spid := 0;
    luc_pid := 0
      

  method stop () = 
    if 
      self#top_set_parameters_window#radiobutton_step_yes#active
    then
      (
	let cmd_stop = ("s\n") in
	  if !debug then (output_string stderr cmd_stop; flush stderr);
	  output_string oc cmd_stop;
	  flush oc;    
	  self#top_step_by_step_window#step_by_step_window#misc#hide ()
      )
    else
      self#top_xlurette#output_window#insert 
	"Stop button unactive because step-by-step mode is off\n"
	

  method on_delete_rif_button_clicked () =
    
    (*     Sys.remove *)
    (*     let _ = (self#top_xlurette#list_rif_files#clear_item 1) *)
    (* 	 ~callback: *)
    (* 	      (begin fun li -> *)
    (* 		 let text = (List.hd li#children)#misc#name in *)
    (* 		   print_string text; *)
    (* 	       end *)
    (* 	      ) *)
    (*     in *)
    output_string oc "\nDelete file \n";
    flush oc


  (*************************************************************************)
      (* Timing Diagram Manager stuff  *)


(* 
   o compute the next_rif_file_name
*)

  method compute_next_rif_file_name () =
    if 
      (self#get_sut_compiler () = "stdin/stdout") 
    then
      update_next_rif_file_name
	self#top_set_parameters_window#rif_file_basename#text
	"stdin" 
	(remove_extension self#top_xlurette#env_name_entry#entry#text)
	self#top_set_parameters_window#test_length#text 
    else
      (
	update_next_rif_file_name
	 self#top_set_parameters_window#rif_file_basename#text
	 (remove_extension self#top_xlurette#sut_name_entry#entry#text) 
	 (remove_extension self#top_xlurette#env_name_entry#entry#text)
	 self#top_set_parameters_window#test_length#text 	
      )

  method get_rif_file () =     
    !rif_file


  method tdm_gnuplot () =
    let rif_file = String.escaped self#top_xlurette#rif_file#entry#text in
    let cmd = 
      if rif_file = "" then "" else 
	("set_output \"" ^ (String.escaped rif_file) ^ "\"\n" ^ "gnuplot\n")  
    in
      output_string oc cmd ;
      if !debug then (output_string stderr cmd;flush stderr);
      flush oc

  method tdm_gnuplot_ps () =
    let rif_file = self#top_xlurette#rif_file#entry#text in
    let cmd = 
      ("set_output \"" ^ (String.escaped rif_file) ^ "\"\n" ^ "gnuplot_ps\n") 
    in
      output_string oc cmd ;
      if !debug then (output_string stderr cmd;flush stderr);
      flush oc

  method tdm_sim2chro () =
    let rif_file = self#top_xlurette#rif_file#entry#text in
    let cmd = 
      ("set_output \"" ^ (String.escaped rif_file) ^ "\"\n" ^ "sim2chro\n") 
    in
      output_string oc cmd ;
      if !debug then (output_string stderr cmd;flush stderr);
      flush oc

  method tdm_edit () =
    let rif_file = self#top_xlurette#rif_file#entry#text in
    let cmd = 
      if hosttype = "win32" then
	("%EDITOR% " ^ rif_file ^ "&\n") 
      else
	("$EDITOR " ^ rif_file ^ "&\n") 
    in
    let _ = 
      try Sys.command cmd  with _ -> assert false
    in
      ()
      
  method tdm_delete () =
    let rif_file = self#top_xlurette#rif_file#entry#text in
      try
	let cmd = 
	  ("rm -f " ^ (Filename.chop_extension rif_file) ^ ".*  2>/dev/null \n") 
	in
	let _ = 
	  try Sys.command cmd  with _ -> assert false
	in
	  self#update_rif_combo_delete ()
      with _ -> ()
	

  (*************************************************************************)
	
  method build_env_clicked () = 
    let all_cmds = (self#get_all_cmds ()) 
    in  
      output_string oc all_cmds ; 
      output_string oc ("build_env\n" );
      flush oc;
      self#update_lucky_files_combo ()

  method call_sim2chro_clicked () = 
    let cmd_display = 
      if self#top_xlurette#rif_file#entry#text = "" then "" else
	("set_output \"" ^ (String.escaped self#top_xlurette#rif_file#entry#text)
	 ^ "\"\n" ^ "sim2chro\n") 
    in
      output_string oc cmd_display ;
      if !debug then (output_string stderr cmd_display;flush stderr);
      flush oc
	

  (* returns all the command to be sent to lurettetop in order to run the test *)
  method get_all_cmds () =
    let sut = String.escaped self#top_xlurette#sut_name#entry#text 
    and sut_node = self#get_sut_node () in 
    let cmd_sut = 
      if 
	((self#get_sut_compiler ()) = "stdin/stdout")
      then
	("set_sut_cmd \"" ^ sut ^ "\"\n") 
      else
	("set_sut \"" ^ sut ^ "\" \"" ^ sut_node ^ "\"\n") 
    in

    let env = String.escaped self#top_xlurette#env_name#entry#text in
    let cmd_env = ("set_env \"" ^ env ^ "\"\n") in

    let oracle = String.escaped self#top_xlurette#oracle_name#entry#text 
    and oracle_node = self#get_oracle_node () in
    let cmd_oracle = 
      if 
	((self#get_sut_compiler ()) = "stdin/stdout")
      then
	("set_oracle_cmd \"" ^ oracle ^ "\"\n") 
      else
	("set_oracle \"" ^ oracle ^ "\" \"" ^ oracle_node ^ "\"\n") in

    let cmd_sut_compiler = 
      ("set_sut_compiler \"" ^ (self#get_sut_compiler ()) ^ "\"\n")
    in

    let oracle_compiler = (self#get_oracle_compiler ()) in
    let cmd_oracle_compiler =
      if oracle_compiler = "" then 
        if (self#get_sut_compiler ()) = "ocaml" then "set_oracle_compiler \"ocaml\"" else "" 
      else
	("set_oracle_compiler \"" ^ oracle_compiler ^ "\"\n")
    in

    let test_length = self#top_set_parameters_window#test_length#text in
    let cmd_test_length = ("set_test_length " ^ test_length ^ "\n") in

    let draw_nb = self#top_set_parameters_window#draw_nb#text in
    let cmd_draw_nb = ("set_draw_nb " ^ draw_nb ^ "\n") in

    let pp = self#get_pp_entry () in 
    let pp_cmd = ("set_preprocessor \"" ^ pp  ^ "\"\n") in

    let cmd_step =
      if self#top_set_parameters_window#radiobutton_step_yes#active
      then ("set_step_by_step " ^ self#top_step_by_step_window#skip_step_number#text  ^ "\n") 
      else "set_step_by_step_off\n"
    in

    let cmd_seed =
      if self#top_set_parameters_window#radiobutton_random_seed#active 
      then "set_seed_randomly \n" 
      else ("set_seed " ^ (string_of_int (self#get_user_seed ())) ^ "\n")
    in

    let cmd_step_mode =
      if self#top_set_parameters_window#radiobutton_step_mode_inside#active 
      then "set_step_mode inside\n" else
	if self#top_set_parameters_window#radiobutton_step_mode_edges#active 
	then "set_step_mode edges\n" else
	  if self#top_set_parameters_window#radiobutton_step_mode_vertices#active 
	  then "set_step_mode vertices\n" else assert false
    in

    let cmd_compute_vol =
      if self#top_set_parameters_window#radiobutton_fair_draw#active 
      then "set_compute_volume true\n" 
      else "set_compute_volume false\n" 
    in
      
    let cmd_draw_all_formula = 
      if self#top_set_parameters_window#radiobutton_draw_all_formula#active
      then "set_draw_all_formula true\n" 
      else "set_draw_all_formula false\n"
    in
    let cmd_draw_inside = 
      ("set_draw_inside " ^ self#top_set_parameters_window#inside_nb#text ^ "\n")
    in
    let cmd_draw_edges = 
      ("set_draw_edges " ^ self#top_set_parameters_window#edges_nb#text ^ "\n")
    in
    let cmd_draw_vertices = 
      if 
	self#top_set_parameters_window#radiobutton_draw_all_vertices#active
      then 
	"set_draw_all_vertices true\n"
      else
	("set_draw_all_vertices false" ^ 
	 "\nset_draw_vertices " ^ self#top_set_parameters_window#vertices_nb#text ^ "\n")
    in

    let cmd_call_sim2chro =
      if self#top_set_parameters_window#radiobutton_call_sim2chro_yes#active 
      then "set_display_sim2chro true \n" 
      else "set_display_sim2chro false \n" 
    in

    let cmd_display_local =
      if self#top_set_parameters_window#radiobutton_display_local_yes#active 
      then "set_display_local_var true \n"
      else "set_display_local_var false \n" 
    in

    let cmd_verbose =
      if self#top_set_parameters_window#radiobutton_verbose_0#active 
      then "set_verbose 0\n" 
      else if self#top_set_parameters_window#radiobutton_verbose_1#active 
      then "set_verbose 1\n" 
      else if self#top_set_parameters_window#radiobutton_verbose_2#active 
      then "set_verbose 2\n" 
      else if self#top_set_parameters_window#radiobutton_verbose_3#active 
      then "set_verbose 3\n" 
      else ""
    in
    let cmd_reactive =
      if self#top_set_parameters_window#radiobutton_reactive_on#active 
      then "set_reactive true \n" 
      else "set_reactive false \n" 
    in
    let cmd_precision = 
      ("set_precision " ^ self#top_set_parameters_window#precision#text ^ "\n")
    in

    let cmd_show_step =
      (*       if (self#top_set_parameters_window#radiobutton_show_step_on#active  *)
      (* 	  || self#top_set_parameters_window#radiobutton_verbose_on#active ) *)
      (* 	  then  *)
      "set_show_step true \n" 
	(* 	  else *)
	(* 	    "set_show_step false \n"  *)
    in      

    let extra_cfiles = 
      ("set_extra_cfiles \"" ^ 
       (String.escaped self#top_extra_env_var_window#extra_cfiles_entry#text) ^ "\"\n")
    and extra_libs = 
      ("set_extra_libs \"" ^ 
       (String.escaped self#top_extra_env_var_window#extra_libs_entry#text) ^ "\"\n")
    and extra_libdirs =
      ("set_extra_libdirs \"" ^ 
       (String.escaped self#top_extra_env_var_window#extra_libdirs_entry#text) ^ "\"\n")
    and extra_includedirs = 
      ("set_extra_includedirs \"" ^ 
       (String.escaped self#top_extra_env_var_window#extra_includedirs_entry#text) ^ "\"\n")
    and c_generator_flag = 
      ("set_c_generator_flag \"" ^ 
       (String.escaped self#top_extra_env_var_window#c_generator_flag_entry#text) ^ "\"\n")
    in      

      cmd_test_length ^ cmd_draw_nb ^ cmd_compute_vol 
      ^ cmd_sut ^ cmd_env ^ cmd_oracle ^ cmd_step ^ cmd_seed
      ^ cmd_draw_inside ^ cmd_draw_edges ^ cmd_draw_vertices ^ cmd_step_mode
      ^ cmd_draw_all_formula ^ cmd_call_sim2chro ^ cmd_precision
      ^ cmd_display_local ^ cmd_verbose ^ cmd_reactive ^ cmd_show_step ^ cmd_sut_compiler 
      ^ cmd_oracle_compiler ^ extra_cfiles ^ extra_libs ^ extra_libdirs 
      ^ extra_includedirs ^ c_generator_flag ^ pp_cmd 

 	
  method update_rif_combo () =
    let rif_files = (get_files_list_filtered ["*\.rif$"] [""]) in
      self#top_xlurette#rif_file#set_popdown_strings rif_files;
      (* set_popdown_strings changes the value of the entry, 
	 hence i need to restore it *)
      self#top_xlurette#rif_file#entry#set_text !rif_file

  method update_rif_combo_delete () =
    let rif_files = (get_files_list_filtered ["*\.rif$"] [""]) in
      self#top_xlurette#rif_file#set_popdown_strings rif_files;
      (* When a rif file is deleted, i need to set a new rif_file. I use the one
	 that is set automatically after a set_popdown_strings *)
      rif_file := self#top_xlurette#rif_file#entry#text 



  method update_all_combo_boxes () =
    self#update_rif_combo ();
    self#update_lucky_files_combo ();
    self#update_sut_and_oracle_combo ()
      
  method on_sut_name_changed () =
    (* Update the sut node Combo list *)
    let sut = self#top_xlurette#sut_name#entry#text  in
      if !sut_node_present && sut <> !pre_sut_name then
	(
	  pre_sut_name := sut;
	  if 
	    sut = "" 
	  then
	    self#set_sut_node ""
	  else
	    let nodes = (self#nodes_info sut) in
	      if nodes <> [] then
		(
		  self#top_xlurette#sut_node#set_popdown_strings nodes;
		)
	)
	

  method on_sut_node_changed () =
    let sut_node = self#get_sut_node () in
      if 
	sut_node <> !pre_sut_node && !pre_sut_node <> ""
      then
	(
	  pre_sut_node := sut_node
	)


  method on_oracle_name_changed () =
    let oracle = self#top_xlurette#oracle_name#entry#text in
      if !oracle_node_present && oracle <> !pre_oracle_name then
	(
	  pre_oracle_name := oracle;
	  if oracle = "" then
	    self#set_oracle_node ""
	  else
	    let nodes = (self#nodes_info oracle) in
	      if nodes <> [] then 
		self#top_xlurette#oracle_node#set_popdown_strings nodes;
	)


  method on_oracle_node_changed () =
    if !oracle_node_present then
      let oracle_node = self#get_oracle_node () in
	if 
	  oracle_node <> !pre_oracle_node
	then
	  (
	    pre_oracle_node := oracle_node
	  )


  method update_sut_and_oracle_combo () =
    let files_lus = (get_files_list_filtered ["*\.lus$"; "*\.c$"] [""]) in
    let files_sao = (get_files_list_filtered ["*\.etp"; "*\.vsp"; "*\.saofdm$"; "*\.lus$"; "*\.c$"] [""]) in
    let files_c = (get_files_list_filtered ["*\.c$"] [""]) in
    let files_ec = (get_files_list_filtered ["*\.ec$"] [""]) in
    let files_ml = (get_files_list_filtered ["*\.ml$"] [""]) in
    let sut = self#top_xlurette#sut_name#entry#text
    and oracle = self#top_xlurette#oracle_name#entry#text
    and sut_node = self#get_sut_node ()
    and oracle_node = self#get_oracle_node ()
    and env_name = self#get_env ()
    in
      (
	match (self#get_oracle_compiler ()) with
	  | "lv6"
	  | "lv4"
	  | "verimag" -> 
              self#add_oracle_node ();
	      if files_lus <> [] then 
		self#top_xlurette#oracle_name#set_popdown_strings files_lus ;
	  | "scade" -> 
              self#add_oracle_node ();
	      if files_sao <> [] then 
		self#top_xlurette#oracle_name#set_popdown_strings files_sao ;
	      
	  | "sildex" -> 
              self#add_oracle_node ();
	      if files_c <> [] then 
		self#top_xlurette#oracle_name#set_popdown_strings files_c ;
	      
	  | _ -> ()
      );

      (
	match (self#get_sut_compiler ()) with
	  | "verimag" 
	  | "lv4" 
	  | "lv6" -> 
              self#add_sut_node ();
              self#add_oracle_node ();
              self#add_oracle_compiler ();
	      if files_lus <> [] then 
		(
		  self#top_xlurette#sut_name#set_popdown_strings files_lus ;
		)
		
	  | "ocaml" -> 
	      if files_ml <> [] then 
		(
		  self#top_xlurette#sut_name#set_popdown_strings files_ml ;
		  self#top_xlurette#oracle_name#set_popdown_strings files_ml ;
                  self#rm_oracle_compiler ();
                  self#rm_oracle_node ();
                  self#rm_sut_node ()
		)
	  | "scade" -> 
              self#add_oracle_node ();
              self#add_oracle_compiler ();
              self#add_sut_node ();
	      if files_sao <> [] then 
		(
		  self#top_xlurette#sut_name#set_popdown_strings files_sao ;
		)
		
	  | "sildex" -> 
              self#add_oracle_node ();
              self#add_oracle_compiler ();
              self#add_sut_node ();
	      if files_c <> [] then 
		(
		  self#top_xlurette#sut_name#set_popdown_strings files_c ;
		)
		
	  | "stdin/stdout" -> 
	      let msgl = 
		[ "% ZZZ experimental and undocumented      ";
		  "% Enter a command that launch an app that";
		  "% reads/writes RIF on stdin/stdout, e.g.,"; 
		  "./a.out"; 
		] 
		@ (List.map (fun x -> if x ="" then "" else "ecexe "^ x) files_ec)
	      in
		(
		  self#top_xlurette#oracle_name#set_popdown_strings msgl;
		  self#top_xlurette#sut_name#set_popdown_strings msgl ;
		)   
	      	
	  | str ->
	      print_string str; 
	      print_string "\nUnsupported compiler\n";
	      flush stdout;
	      assert false
      );
      
      (* 
	 XXX set_poddown_strings changes the entries !! 
	 I therefore need to save and restore them. 
      *)
      self#top_xlurette#sut_name#entry#set_text sut;
      self#top_xlurette#sut_node_entry#entry#set_text sut_node;

      self#top_xlurette#oracle_name#entry#set_text oracle;
      self#top_xlurette#oracle_node_entry#entry#set_text oracle_node;
      self#set_env env_name

	

  method update_lucky_files_combo () =
    let env = self#top_xlurette#env_name#entry#text in
    let files = (get_files_list_filtered ["*\.lut$"; "*\.luc$"] [""]) in
      if files <> [] then 
	(
	  self#top_xlurette#env_name#set_popdown_strings files ;
	  self#top_xlurette#env_name#entry#set_text env
	)

	
  method run_lurette () = 
    
    self#top_xlurette#progressbar#set_percentage 0.0;

    (* .saofdm must be compiled with scade.*)
    if
      (
	Filename.check_suffix self#top_xlurette#sut_name_entry#entry#text ".etp" ||
	Filename.check_suffix self#top_xlurette#sut_name_entry#entry#text ".vsp" ||
	Filename.check_suffix self#top_xlurette#sut_name_entry#entry#text ".saofdm"
      )
      && (List.mem (self#get_sut_compiler ()) ["verimag"; "lv4"; "lv6";"ocaml"])
    then
      self#set_sut_compiler "scade";
    if 
      (
	Filename.check_suffix self#top_xlurette#sut_name_entry#entry#text ".etp" ||
	Filename.check_suffix self#top_xlurette#sut_name_entry#entry#text ".vsp" ||
	Filename.check_suffix self#top_xlurette#sut_name_entry#entry#text ".saofdm" 
      )
      && (List.mem (self#get_oracle_compiler ()) ["verimag"; "lv4"; "lv6"])
    then
      self#set_oracle_compiler "scade";
    
    let all_cmds = (self#get_all_cmds ()) in

    let cmd_set_output_file = 
      self#compute_next_rif_file_name ();
      if !next_rif_file = "" then "" else 
	("set_output \"" ^ (String.escaped !next_rif_file) ^ "\"\n") 
    in
      
      if self#top_set_parameters_window#radiobutton_step_yes#active
      then self#top_step_by_step_window#step_by_step_window#show ();
      
      output_string oc all_cmds ;
      flush oc;
      output_string oc cmd_set_output_file ;
      flush oc;
      output_string oc "run\n" ;
      flush oc;
      if !debug then 
	(
	  output_string stderr (all_cmds ^ "\n" ^ cmd_set_output_file ^ "run\n"); 
	  flush stderr
	) ;
      flush oc;
      if 
	self#top_xlurette#env_name#entry#text = ""
      then
	(* Generate an environment for the current sut *)
	(
	  let node = self#top_xlurette#sut_node_entry#entry#text in
	    (* a fake env ougth to have been generated by lurettetop *)
	    if 
	      try Sys.file_exists (node ^ "_env.luc") with _ -> assert false
	    then
	      self#top_xlurette#env_name#entry#set_text (node ^ "_env.luc");
	    (* Fill the combo boxes again, since more files may have been created *)
	    self#update_lucky_files_combo ()
	);
      
      self#update_all_combo_boxes ()


  method on_env_name_entry_changed () = 
    self#update_lucky_files_combo () 

  method on_create_batch_clicked () =
    let batch_file = ((self#get_sut_node ()) ^ ".batch") in 
      self#save_parameters ();
      output_string oc ("batch \"" ^ batch_file ^ "\"\n"); 
      flush oc
	


  method save_parameters () =
    let msg = "Saving current parameters in .lurette_rc \n"
    and cmds = self#get_all_cmds ()
    and file = open_out ".lurette_rc"
    in
      output_string file cmds;
      close_out file;
      self#top_xlurette#tdm_output_window#insert msg;
      self#top_xlurette#output_window#insert msg


  (*   method save_session () = *)
	(*     output_string oc ("pack " ^ self#top_xlurette#saved_session_file#text ^ " \n"); *)
	(*     flush oc; *)
	(*     prerr_endline "save session ..."; *)
	(*     () *)


  method get_env () =
    self#top_xlurette#env_name_entry#entry#text
  method set_env str =
    self#top_xlurette#env_name_entry#entry#set_text str

  method get_sut () =
    self#top_xlurette#sut_name#entry#text
  method set_sut str =
    self#top_xlurette#sut_name#entry#set_text str

  method get_sut_node () =
    if !sut_node_present then
      self#top_xlurette#sut_node_entry#entry#text
    else
      ""
  method set_sut_node str =
    if !sut_node_present then
      self#top_xlurette#sut_node#entry#set_text str ;

  method get_sut_compiler () =
    self#top_xlurette#sut_compiler_entry#entry#text
  method set_sut_compiler str =
    self#top_xlurette#sut_compiler#entry#set_text str

  method get_oracle_compiler () =
    if !oracle_compiler_present then
      self#top_xlurette#oracle_compiler_entry#entry#text
    else
      ""
  method set_oracle_compiler str =
    if !oracle_compiler_present then
      self#top_xlurette#oracle_compiler#entry#set_text str ;


  method get_oracle () =
    self#top_xlurette#oracle_name#entry#text
  method set_oracle str =
    self#top_xlurette#oracle_name#entry#set_text str

  method get_oracle_node () =
    if !oracle_node_present then
      self#top_xlurette#oracle_node#entry#text
    else
      ""
  method set_oracle_node str =
    if !oracle_node_present then
      self#top_xlurette#oracle_node#entry#set_text str ;


  method get_test_length () =
    try
      int_of_string self#top_set_parameters_window#test_length#text
    with _ -> 
      10
  method set_test_length i =
    self#top_set_parameters_window#test_length#set_text (string_of_int i)



  method get_draw_nb () =
    try
      int_of_string self#top_set_parameters_window#draw_nb#text
    with _ -> 1
  method set_draw_nb i =
    self#top_set_parameters_window#draw_nb#set_text (string_of_int i)


  method get_draw_inside_nb () =
    try
      int_of_string self#top_set_parameters_window#inside_nb#text
    with _ -> 0

  method set_draw_inside_nb i =
    self#top_set_parameters_window#inside_nb#set_text (string_of_int i)

  method get_draw_edges_nb () =
    try
      int_of_string self#top_set_parameters_window#edges_nb#text
    with _ -> 0

  method set_draw_edges_nb i =
    self#top_set_parameters_window#edges_nb#set_text (string_of_int i)

  method get_draw_vertices_nb () =
    try
      if !vertices_nb_present then
	int_of_string self#top_set_parameters_window#vertices_nb#text
      else
	0
    with _ -> 0

  method set_draw_vertices_nb i =
    if !vertices_nb_present then
      self#top_set_parameters_window#vertices_nb#set_text (string_of_int i)


  method get_precision () =
    try
      int_of_string self#top_set_parameters_window#precision#text
    with _ -> 2

  method set_precision i =
    self#top_set_parameters_window#precision#set_text (string_of_int i)


  method get_user_seed () =
    !user_seed
  method set_user_seed i =
    user_seed := i;
    if !user_seed_present then
      self#top_set_parameters_window#user_seed#set_text (string_of_int i)



  method get_pp_entry () =
    self#top_set_parameters_window#pp_entry#text
  method set_pp_entry str =
    self#top_set_parameters_window#pp_entry#set_text str

  method get_rif_file_basename () =
    self#top_set_parameters_window#rif_file_basename#text
  method set_rif_file_basename str =
    self#top_set_parameters_window#rif_file_basename#set_text str

  (*   method get_restore () = *)
      (*     self#top_set_parameters_window#saved_session_file#text *)
      (*   method set_restore str = *)
      (*     self#top_set_parameters_window#saved_session_file#set_text str *)


  method read_commands line =
    let lexer = Genlex.make_lexer [] in
    let (parse_ident_or_string : tok -> string) = 
      fun tok -> 
	try
	  match tok with parser 
	    | [<  'Genlex.String str >] -> str
	    | [<  'Genlex.Ident str >] -> str
	    | [<  >]  ->  "" 
	  with _ -> 
	    print_string 
	    "*** parse error: cannot parse that file name.\n";
	    flush stdout;
	    ""
    in
    let rec read_cmd tok =
      (* : 'a -> tok -> unit) = *)

      ( match tok with parser
	    
	  | [< 'Genlex.Ident "set_env" ; 'Genlex.String str >] ->
	      self#set_env str
	      
	  | [< 'Genlex.Ident "set_output" ; 'Genlex.String rif_file_str >] ->
	      let remove_the_end x =
		try
		  let idx = 
		    Str.search_backward (Str.regexp "[0-9]") x (String.length x) 
		  in
		  let _ = 
		    Str.search_backward (Str.regexp "[^0-9]") x idx
		  in
		    String.sub x 0 (Str.match_end ()) 
		with 
		    Not_found -> x
		  | _ -> assert false
	      in
	      let rif_file_str' = (remove_the_end rif_file_str) in
		self#set_rif_file_basename rif_file_str';
		rif_file := rif_file_str'
		  
		  
	  | [< 
	      'Genlex.Ident "set_sut_cmd" ;  'Genlex.String str
	    >] ->
	      pre_sut_name := str;
	      pre_sut_node := "";
	      self#set_sut str;
	      self#set_sut_node ""

	  | [< 
	      'Genlex.Ident "set_oracle_cmd" ;  'Genlex.String str
	    >] ->
	      pre_oracle_name := str;
	      pre_oracle_node := "";
	      self#set_oracle str;
	      self#set_oracle_node ""
		
	  | [< 
	      'Genlex.Ident "set_sut" ; 
	      str = parse_ident_or_string  ;
	      node = parse_node 
	    >] ->
	      pre_sut_name := str;
	      pre_sut_node := node;
	      self#set_sut str;
	      self#set_sut_node node
		
	  | [< 
	      'Genlex.Ident "set_oracle" ; 
	      str = parse_ident_or_string  ; 
	      node = parse_node 
	    >] ->
	      pre_oracle_node := node;
	      pre_oracle_name := str;
	      self#set_oracle str;
	      self#set_oracle_node node
		
	  | [< 'Genlex.Ident "set_sut_compiler" ; str = parse_ident_or_string >] ->
	      self#set_sut_compiler str
	      
	  | [< 
	      'Genlex.Ident "set_oracle_compiler" ; 
	      str = parse_ident_or_string 
	    >] ->
	      self#set_oracle_compiler str

	  | [< 'Genlex.Ident "set_step_mode" ; str = parse_ident_or_string >] ->
	      ( match str with
		  | "vertices" ->
		      self#top_set_parameters_window#radiobutton_step_mode_vertices#set_active true
		  | "edges" -> 
		      self#top_set_parameters_window#radiobutton_step_mode_edges#set_active true
		  | "inside" -> 
		      self#top_set_parameters_window#radiobutton_step_mode_inside#set_active true
		  | _ -> 
		      self#top_set_parameters_window#radiobutton_step_mode_inside#set_active  true
	      );
	      
	  | [< 'Genlex.Ident "set_test_length" ; 'Genlex.Int i >] ->
	      self#set_test_length i
	      
	  | [< 'Genlex.Ident "set_draw_inside" ; 'Genlex.Int i >] ->
	      self#set_draw_inside_nb i
	  | [< 'Genlex.Ident "set_draw_edges" ; 'Genlex.Int i >] ->
	      self#set_draw_edges_nb i
	  | [< 'Genlex.Ident "set_draw_vertices" ; 'Genlex.Int i >] ->
	      self#set_draw_vertices_nb i

	  | [< 'Genlex.Ident "set_precision" ; 'Genlex.Int i >] ->
	      self#set_precision i

	  | [< 'Genlex.Ident "set_preprocessor" ;  pp = parse_ident_or_string >] ->
	      self#set_pp_entry pp

	  | [< 'Genlex.Ident "set_draw_nb" ; 'Genlex.Int i >] -> 
	      self#set_draw_nb i
	      
	  | [< 'Genlex.Ident "set_seed" ; 'Genlex.Int i >] -> 
	      self#top_set_parameters_window#radiobutton_user_seed#set_active true ;
	      self#set_user_seed i

	  | [< 'Genlex.Ident "set_compute_volume"; 'Genlex.Ident id >] -> 
	      if id = "true" 
	      then self#top_set_parameters_window#radiobutton_fair_draw#set_active true 
	      else self#top_set_parameters_window#radiobutton_efficient_draw#set_active true 

	  | [< 'Genlex.Ident "set_draw_all_formula"; 'Genlex.Ident id >] -> 
	      if id = "true" 
	      then self#top_set_parameters_window#radiobutton_draw_all_formula#set_active true 
	      else self#top_set_parameters_window#radiobutton_draw_one_formula#set_active true 

	  | [< 'Genlex.Ident "set_draw_all_vertices"; 'Genlex.Ident id >] -> 
	      if id = "true" 
	      then self#top_set_parameters_window#radiobutton_draw_all_vertices#set_active true 
	      else self#top_set_parameters_window#radiobutton_draw_some_vertices#set_active true 



	  | [< 'Genlex.Ident "set_seed_randomly" >] -> 
	      self#top_set_parameters_window#radiobutton_random_seed#set_active true 

	  | [< 'Genlex.Ident "set_step_by_step" ; 'Genlex.Int i >] -> 
	      self#top_set_parameters_window#radiobutton_step_yes#set_active true;
	      self#top_step_by_step_window#skip_step_number#set_text (string_of_int i) 

	  | [< 'Genlex.Ident "set_step_by_step_off" >] -> 
	      self#top_set_parameters_window#radiobutton_step_no#set_active true 

	  | [< 'Genlex.Ident "set_display_sim2chro" ; 'Genlex.Ident id >] -> 
	      if id = "true" 
	      then self#top_set_parameters_window#radiobutton_call_sim2chro_yes#set_active true 
	      else self#top_set_parameters_window#radiobutton_call_sim2chro_no#set_active true 

	  | [< 'Genlex.Ident "set_display_local_var" ; 'Genlex.Ident id >] -> 
	      if id = "true" 
	      then self#top_set_parameters_window#radiobutton_display_local_yes#set_active true 
	      else self#top_set_parameters_window#radiobutton_display_local_no#set_active true 

	  | [< 'Genlex.Ident "set_verbose" ; 'Genlex.Int vlevel >] -> (
	      match vlevel with
		  0 -> 
		    self#top_set_parameters_window#radiobutton_verbose_0#set_active true ;
		    debug := false
		| 1 -> 
		    self#top_set_parameters_window#radiobutton_verbose_1#set_active true ;
		    debug := false
		| 2 -> 
		    self#top_set_parameters_window#radiobutton_verbose_2#set_active true ;
		    debug := false
		| 3 -> 
		    self#top_set_parameters_window#radiobutton_verbose_3#set_active true ;
		    debug := true
		| _ ->  
		    self#top_set_parameters_window#radiobutton_verbose_3#set_active true ;
		    debug := true
	    )
	  | [< 'Genlex.Ident "set_reactive" ; 'Genlex.Ident id >] -> 
	      if id = "true" 
	      then self#top_set_parameters_window#radiobutton_reactive_on#set_active true 
	      else self#top_set_parameters_window#radiobutton_reactive_off#set_active true 

	  | [< 'Genlex.Ident "set_show_step" ; 'Genlex.Ident id >] -> 
	      ()
	      (* 	      if id = "true"  *)
	      (* 	      then self#top_set_parameters_window#radiobutton_show_step_on#set_active true  *)
	      (* 	      else self#top_set_parameters_window#radiobutton_show_step_off#set_active true  *)

	  | [< 'Genlex.Ident "set_extra_cfiles" ; 'Genlex.String str >] -> 
	      self#top_extra_env_var_window#extra_cfiles_entry#set_text str 

	  | [< 'Genlex.Ident "set_extra_libs" ; 'Genlex.String str >] -> 
	      self#top_extra_env_var_window#extra_libs_entry#set_text str 

	  | [< 'Genlex.Ident "set_extra_libdirs" ; 'Genlex.String str >] -> 
	      self#top_extra_env_var_window#extra_libdirs_entry#set_text str 

	  | [< 'Genlex.Ident "set_extra_includedirs" ; 'Genlex.String str >] -> 
	      self#top_extra_env_var_window#extra_includedirs_entry#set_text str 

	  | [< 'Genlex.Ident "set_c_generator_flag" ; 'Genlex.String str >] -> 
	      self#top_extra_env_var_window#c_generator_flag_entry#set_text str 

	  | [< _ >] -> ()
      )

    and  
      (parse_node : tok -> string) = 
      fun tok -> 
	let str =
	  try
	    match tok with parser 
	      | [< 'Genlex.String id  >] -> id
	      | [< 'Genlex.Ident id  >] -> id
	      | [<   >] -> ""
	    with _ -> 
	      ""
	in
	let base = Filename.basename str in
	  try Filename.chop_extension base 
	  with _ -> base
	    
    in
      try
	read_cmd (lexer (Stream.of_string line))
      with _ -> 
	print_string (
	  "*** Warning: an error occured when parsing the \".lurette_rc\" file;"^
	  " command " ^ line ^" is ignored.\n" 
	);
	  flush stdout;
(* 	  exit 1 *)

end


(**************************************************************************)
let usage = "
usage: xlurette [<options>] [<env ([x] env)*>]

xlurette is a Graphical interface over lurettetop, which lets
one automatically test reactive programs written in lustre.

Command line <options> are:
"

let rec speclist callbacks = 
  [
    "--sut", Arg.String (fun s -> callbacks#set_sut s), 
       "<string>\tFile name of the system under test.";
    "-sut", Arg.String (fun s -> callbacks#set_sut s),
    " <string>\n" ;

    "--oracle", Arg.String (fun s -> callbacks#set_oracle s), 
       "<string>\tFile name of the oracle.";
    "-oracle", Arg.String (fun s -> callbacks#set_oracle s), " <string>\n";

    "--test-length", Arg.Int (fun i -> callbacks#set_test_length i),
       "<int>\tNumber of steps to be done";
    "-l", Arg.Int (fun i -> callbacks#set_test_length i),  
    ("<int>\t\t(default=" ^ (string_of_int (callbacks#get_test_length ())) ^ ").\n");

    "--thick-draw", Arg.Int (fun i -> callbacks#set_draw_nb i),
       "<int>\tNumber of draw to be done in each formula ";
    "-td", Arg.Int (fun i -> callbacks#set_draw_nb i),
    ("<int>\t\tat each step (default=" ^ 
     (string_of_int (callbacks#get_draw_nb ())) ^ ").\n");
    
    "--draw-inside", Arg.Int (fun i -> callbacks#set_draw_inside_nb i),
       "<int>\tDraw on the edges of the convex hull of solutions.";

    "--draw-edges", Arg.Int (fun i -> callbacks#set_draw_edges_nb i),
       "<int>\t\tDraw on the edges of the convex hull of solutions.";

    "--draw-vertices", Arg.Int (fun i -> callbacks#set_draw_vertices_nb i),
       "<int>\tDraw among the vertices of the convex hull of solutions.\n ";


    "--precision", Arg.Int (fun i -> callbacks#set_precision i),
    ("<int>\tNumber of digit after the dot used to perform the " ^ 
     "numeric computations .");


    "--output", Arg.String (fun s -> callbacks#set_rif_file_basename s),
    ("<string>\tSet the output file name (default is \"" ^ 
    (callbacks#get_rif_file_basename ()) ^ "\").");
    "-o", Arg.String (fun s -> callbacks#set_rif_file_basename s), "<string>\n";


(*     "--restore", Arg.String (fun s -> callbacks#set_restore s; restore := true),  *)
(*        "<string>\tFile name (without extension) of the package containing" *)
(*        ^ "\n\t\t\tthe temporarily files to be restored (cf the pack command).\n"; *)


    "--help", Arg.Unit (fun _ -> (Arg.usage (speclist callbacks) usage  ; exit 0)),
       "\t\tDisplay this list of options." ;
    "-help", Arg.Unit (fun _ -> (Arg.usage (speclist callbacks) usage  ; exit 0)),
    "";
    "-h", Arg.Unit (fun _ -> (Arg.usage (speclist callbacks) usage  ; exit 0)),
    ""
]
    

(******************************************************************************)

let main () = 
  let _ =
    (* Check that PIXMAP_DIR env var is set. *)
    try 
      try Sys.getenv "PIXMAP_DIR" with _ -> assert false;
    with Not_found -> 
      (
	print_string "*** Can not run xlurette, sorry. \n***";
	print_string "You need to set the PIXMAP_DIR env variable properly.\n";
	flush stdout;
	exit 2;
      );
  in
  let _ = dbg_print "xlurette: starting main loop.\n" in

(*   let lurette_path = *)
(*     (* Check that LURETTE_PATH env var is set. *) *)
(*     try  *)
(*       try Sys.getenv "LURETTE_PATH" with _ -> assert false; *)
(*     with Not_found ->  *)
(*       ( *)
(* 	print_string "*** Can not run xlurette, sorry. \n***"; *)
(* 	print_string "You need to set the LURETTE_PATH env variable properly.\n"; *)
(* 	flush stdout; *)
(* 	exit 2; *)
(*       ) *)
(*   in *)
  let callbacks = new customized_callbacks in
  let _ = dbg_print "xlurette: starting Xlurette_glade_interface.top_xlurette.\n" in
  let xlurette = new Xlurette_glade_interface.top_xlurette callbacks in
  let _ = dbg_print "xlurette: XXXXX .\n" in

  let _step_by_step = 
    new Xlurette_glade_interface.top_step_by_step_window callbacks 
  in
  let _browse_sut = 
    new Xlurette_glade_interface.top_fileselection_sut callbacks 
  in
  let _browse_oracle = 
    new Xlurette_glade_interface.top_fileselection_oracle callbacks 
  in
  let _browse_env = 
    new Xlurette_glade_interface.top_fileselection_env callbacks 
  in
  let _browse_add_env = 
    new Xlurette_glade_interface.top_fileselection_add_env callbacks 
  in

  let _on_sut_help_button = 
    new Xlurette_glade_interface.top_sut_help_window callbacks 
  in
  let on_set_parameters_button = 
    new Xlurette_glade_interface.top_set_parameters_window callbacks 
  in

  let _ = on_set_parameters_button#set_parameters_window#set_title 
	    " Setting the Test Parameters " ;
    if on_set_parameters_button#radiobutton_random_seed#active then
      (
	on_set_parameters_button#random_seed_hbox#remove 
	 on_set_parameters_button#user_seed#coerce;
	user_seed_present := false
      );

    if on_set_parameters_button#radiobutton_draw_all_vertices#active then
      (
	on_set_parameters_button#dnv_hbox#remove 
	 on_set_parameters_button#vertices_nb#coerce;
	vertices_nb_present := false
      )
  in 

  let _on_env_help_button =  
    new Xlurette_glade_interface.top_env_help_window callbacks  
  in 
  let _on_oracle_help_button =  
    new Xlurette_glade_interface.top_oracle_help_window callbacks  
  in 
  let _on_test_thickness_help_button =  
    new Xlurette_glade_interface.top_test_thickness_help_window callbacks  
  in 

  let _on_extra_env_var_button =  
    new Xlurette_glade_interface.top_extra_env_var_window callbacks  
  in 
  let _browse_extra_cfiles =  
    new Xlurette_glade_interface.top_fileselection_extra_cfiles callbacks  
  in 
  let _browse_extra_libs =  
    new Xlurette_glade_interface.top_fileselection_extra_libs callbacks  
  in 
  let _browse_extra_libdirs =  
    new Xlurette_glade_interface.top_fileselection_extra_libdirs callbacks  
  in 
  let _browse_extra_includedirs =  
    new Xlurette_glade_interface.top_fileselection_extra_includedirs callbacks  
  in 

  let on_quit_xlurette_button =  
    new Xlurette_glade_interface.top_quit_window callbacks  
  in 
  let _ = on_quit_xlurette_button#quit_window#set_title 
	    " Bye? " 
  in 

    (* msg generated by mlglade :
       You should probably remove the next line if you want to use the 
       event masks from glade 
    *)
  let _ = 
    dbg_print "xlurette: starting gtk stuff.\n";

    GtkBase.Widget.add_events xlurette#xlurette#as_widget [`ALL_EVENTS] in

    callbacks#put      " \t\t\t\t\t\t\t\t\t\t\t\t********* Standard Output Window *********\n";
    callbacks#put_misc " \t\t\t\t\t\t\t\t********* Miscellaneous Messages Window (system calls, etc.) *********\n";
    
    (* Fill the combo boxes *)
    callbacks#update_rif_combo ();
    callbacks#update_sut_and_oracle_combo ();
    callbacks#update_lucky_files_combo ();
    
    (* read command in the .lurette_rc file *)
    let lurette_rc =  ".lurette_rc" in 
      (  
	if 
	  try Sys.file_exists lurette_rc  with _ -> assert false
	then 
	  let ic = (open_in lurette_rc) in 
	    try 
	      while true do 
		let _ =  
		  callbacks#read_commands (input_line ic) 
		in 
		  () 
	      done  
	    with End_of_file ->  
	      close_in ic  
      ); 
      

    (* 
       read the lurettetop command line options (that will override the 
       .lurette.=_rc ones) 
    *)
      let env_saved = callbacks#get_env () in
	callbacks#set_env "";
	( try 
	    Arg.parse (speclist callbacks)
		(fun s -> 
		   if s = "x" 
		   then callbacks#set_env ((callbacks#get_env ()) ^ " x ")
		   else callbacks#set_env ((callbacks#get_env ()) ^ " " ^ s )
		)
		usage
	  with 
	      Failure(e) ->
		(
		  print_string e;
		  flush stdout ;
		  flush stderr ;
		  exit 2
		)
	    | _ ->
	      exit 2 
	);
	if (callbacks#get_env () = "") then callbacks#set_env env_saved;
        dbg_print "xlurette: show!\n";

	xlurette#xlurette#show() ;

	let _ = xlurette#xlurette#connect#destroy ~callback: callbacks#quit in	
	let _ = xlurette#xlurette#set_title " Xlurette " in	

	let _ = on_set_parameters_button#set_parameters_window#connect#destroy 
		  ~callback: callbacks#on_close_parameters_window_clicked in
	let _ = Timeout.add ~ms:100 ~callback:callbacks#read_ltop_stdout in  
	let _ = Timeout.add ~ms:100 ~callback:callbacks#read_ltop_stderr in  
	
          dbg_print "xlurette: starting graphics...\n";
	  Main.main ()  
	

(* XXX ougth to work, but does not... 
   it would have been better than what i currently do...
*)
(*
	  GtkThread.async (callbacks#read_ltop_stdout) ();
	  GtkThread.async (callbacks#read_ltop_stderr) ();
	  GtkThread.main ()
*)

let _ =
  if !debug then
    (
      print_string "xlurette: launching the main loop \n";
      flush stdout
    )


let _ = 
  try
    main ()
  with e -> 
    print_string "*** Error in xlurette\n";
    print_string (Printexc.to_string e);
    exit 1
