#############################################################
#
# Scade/Lurette GUI
#
# SYNOPSIS:
#
# AUTHORS:
#    jahier@imag.fr, 
#    Laurent.POUCHAN@esterel-technologies.com
#    
#
# CREATED:
#    May 2005
#
#
#############################################################


# Packages ##################################################
package require commands
package require properties
package require dialogs



global ScadeLuretteVersion
set ScadeLuretteVersion "0.1"

#'############################################################
#' Test parameters 
#'############################################################

##### Lurette Settings vars
global lurette_projectPathName
set lurette_projectPathName ""

global lurette_dir
set lurette_dir ""

global lurette_sut
set lurette_sut ""

global lurette_oracle
set lurette_oracle ""

global lurette_use_oracle
set lurette_use_oracle "false"

global lurette_env
set lurette_env ""

global lurette_step_nb
set lurette_step_nb "100"

##### Lurette Preferences var

global rifbase_filename
set rifbase_filename ""

global dnb
set dnb 1

global dne
set dne 0

global dni
set dni 0

global dnv
set dnv 0

global precision
set precision "2"

global seed
set seed 0

global seed_is_random
set seed_is_random "true"
 
global pp
set pp "lucky_cpp.bat"

global draw_all_formula
set draw_all_formula "true"

global draw_all_vertices
set draw_all_vertices "false"

global step_mode
set step_mode "inside"

global verbose_level
set verbose_level 0

global display_local
set display_local "false"

# input/output and error channels (opened via sockets)
global ioc
global log_oc
global log_ec

# used to know when the sockets are initialised
global socket_state
set socket_state "not_connected"

# use to clean-up processes at quit time or to kill the running lurette process
global lurette_pid
set lurette_pid 0

global oracle_pid
set sut_pid 0

global sut_pid
set oracle_pid 0

# state of executables
global lurettetop_is_running
set lurettetop_is_running false

global lurette_is_running
set lurette_is_running false

#############################################################

global luretteincludedir
global luretteop 
global gen_fake_lucky 

set scadebindir $env(ETBIN)
set lurettedir [file join $scadebindir ".." "lurette"]
set lurettebindir [file join $lurettedir "bin"]
set luretteincludedir [file join $lurettedir "include"]
set lurettetop [file join $lurettebindir "lurettetop"]
set gen_fake_lucky [file join $lurettebindir "gen_fake_lucky"]

set env(LURETTE_PATH) $lurettedir
set env(LURETTE_BIN_PATH) $lurettebindir
set env(HOST_TYPE) win32

#'############################################################
#' Helpers
#'############################################################

proc OutputDebug {str} {
    global verbose_level

    if { $verbose_level > 0 } {
	output $str
    }
}

proc OutputDebugVerb {str} {
    global verbose_level

    if { $verbose_level >= 2 } {
	output $str
    }
}

proc OutputVerb3  {str} {
    global verbose_level

    if { $verbose_level >= 3 } {
	output $str
    }
}

proc slashPath {path} {
    return [string map {\\ \\\\ \  \\\ } $path]
}

proc protectSlashPath {path} {
    return [string map {\\  \\\ } $path]
}
proc antislashPath {path} {
    return [string map {\\ / \  / } $path]
}

# reverse a list
proc reverse { list } {
    set i 0
    set j [expr { [llength $list] - 1 }]
    while { $j > $i } {
	set temp [lindex $list $i]
	lset list $i [lindex $list $j]
	lset list $j $temp
	incr i
	incr j -1
    }
    return $list
}

# returns the index of filename in list, 0 if not in the list
proc FindIndex { filename list } {
    set i 0
    set max [llength $list]
    set cont true

    while { $cont } {
	set name  [Get [lindex $list $i] name]
	if {$name == $filename} {
	    set cont false
	} else {
	    incr i
	}
	if { $i > $max } {
	    set $i 0
	    set cont false
	}
    }
    return $i
}


# -------------------------------------------------------------------------
# CreateFolder -- PROCEDURE
#
# SYNOPSYS:
#   Creates a folder in a project
#
# ARGUMENTS:
#   - project: the project in which to create the folder
#   - name: the name of the folder to create
#   - owner: the owner of the folder
#
# RETURN VALUE:
#   1 if the folder has been found, 0 otherwise.
#
# SIDE EFFECT:
#   Creates a reference to the variable named $folderNameVar from the
#   calling context and modifies it when the folder is found
# -------------------------------------------------------------------------

proc CreateFolder { project name owner} {
    # create a folder in the project file
    set folder [Create $project Folder]
    # link the folder to its owner (project or folder)
    if { [Class $owner] == "Project" } {
	SetRole $folder owner $owner
    } else {
	SetRole $folder folder $owner
    }
    # set the name
    SetAtt $folder name $name

    return $folder
}

# -------------------------------------------------------------------------
# CreateFileReference -- PROCEDURE
#
# SYNOPSYS:
#   Create file references
#
# ARGUMENTS:
#   - project: the project in which create the file references
#   - pathname: the path name to the file reference
#   - owner: the target owner of the reference
#
# RETURN VALUE:
#   The created file reference
#
# SIDE EFFECT:
#   Creates a reference in a project
# -------------------------------------------------------------------------
proc CreateFileReference { project pathname owner} {

    # Create a fileref in the project file
    set fileReference [Create $project FileRef]

    # Link the file to its owner (project or folder)
    if { [Class $owner] == "Project" } {
	SetRole $fileReference owner $owner
    } else {
	SetRole $fileReference folder $owner
    }

    # Set the path (a relative path to the project might is being computed)
    #AM REMOVED Call $fileReference SetPathName $pathname
    SetAtt $fileReference persistAs $pathname
    return $fileReference
}

proc RemoveFileReference { project pathname owner fileReference } {

    # Unlink the file to its owner (project or folder)
    if { [Class $owner] == "Project" } {
	UnsetRole $fileReference owner $owner
    } else {
	UnsetRole $fileReference folder $owner
    }
}

#   Add file references in a folder
#     - project: the project in which to add files
#     - fileList: the list of name of files to add
#     - folder: the folder in which to add files
proc AddFiles {project fileList folder} {
    foreach fileName $fileList {
	CreateFileReference $project $fileName $folder
    }
}
# -------------------------------------------------------------------------
# SearchFolder -- PROCEDURE
#
# SYNOPSYS:
#   Tries to find a folder from its name
#
# ARGUMENTS:
#   - folderNameVar: the name of the variable in which to store the folder
#     when found
#   - folderName: the name of the folder to look for
#   - root: the folder role
#
# RETURN VALUE:
#   1 if tyhe folder has bee found, 0 otherwise.
#
# SIDE EFFECT:
#   Creates a reference to the variable named $folderNameVar from the
#   calling context and modifies it when the folder is found
# -------------------------------------------------------------------------
proc SearchFolder {folderNameVar folderName root} {
    # folder is a reference to the variable called $folderNameVar in the 
    # calling context
    upvar $folderNameVar folder

    if { [string equal [class $root] "Folder"]} { 
	set name [Get $root name]
	if { $folderName == $name } {
	    set folder $root
	    return 0
	}
    }
    return 1
}


############################################################
#
# Persistant Data

proc GetToolProp { project configuration tool name default } {
    set prop [ Call $project FindToolProp $tool $name $configuration ]
    set result $default
    if { "$prop" != "" } {
	set result [Get $prop values]
    }

    return $result
}

proc SetToolProp { project configuration tool name value default } {
    set prop [ Call $project FindToolProp $tool $name $configuration ]
    if { $prop == "" } {
	if { "$value" != "$default" } {
	    set prop [ Create $project "Prop" ]
	    SetAtt $prop "name" "@$tool:$name"
	    SetRole $prop "entity" $project
	    SetRole $prop "configuration" $configuration
	    SetAtt $prop "values" $value
	}
    } else {
	if { "$value" == "$default" } {
	    Delete $prop
	} else {
	    set oldValue [ Get $prop "values" ]
	    if { "$value" != "$oldValue" } {
		SetAtt $prop "values" $value
	    }
	}
    }
}

proc GetBoolToolProp { project configuration tool name default } {
    set value [ GetToolProp $project $configuration $tool $name $default ]
    if { "$value" == "true" } {
	return 1
    } else {
	return 0
    }
}

proc SetBoolToolProp { project configuration tool name value default } {
    if { "$value" == "1" } {
	set value true
    } else {
	set value false
    }
    SetToolProp $project $configuration $tool $name $value $default
}



#'############################################################
#' Build Configuration list
#'############################################################

proc BuildConfigurationListMapProc { resultListConf configuration } {
    upvar $resultListConf configurationList
    #'set confName [Get $configuration "name"]
    #'lappend configurationList $confName 
    lappend configurationList $configuration
    return 1
}

proc BuildConfigurationList {} {
    global project
    set configurationList {}
    if { [llength $project] != 0 } {
	set item [lindex $project 0]
	MapRole $item "configuration" \
	    "BuildConfigurationListMapProc configurationList" 
    }
    return $configurationList
}

#'############################################################
#' Lurette Configuration 
#'############################################################


proc CheckConf { } {
    if { [GetLuretteConfiguration] == "" } { 
        MessageDialog -name "Lurette configuration missing " \
	    -message "Could not find \"Lurette\" configuration" -icon warning 
	CloseCmdDialogLuretteSettings
    }
}

proc GetLuretteConfigurationMapProc { luretteConfigurationRes configuration } {
    upvar $luretteConfigurationRes luretteConfiguration
    set vname [Get $configuration name]
    if {$vname == "Lurette"} {
        set luretteConfiguration $configuration
        return 0
    }
    return 1
}

proc GetLuretteConfiguration {} {
    global project
    set luretteConfiguration ""
    if { [llength $project] != 0 } {
	set item [lindex $project 0]
	MapRole $item "configuration" \
	    "GetLuretteConfigurationMapProc luretteConfiguration" 
    }
    return $luretteConfiguration

}


proc GetRootNode {project} {
    CheckConf
    set conf [GetLuretteConfiguration]
    set root_node [call $project GetToolProp "GENERATOR" "ROOTNODE" $conf]

    return $root_node
}


#'############################################################
#' Build Node list
#'############################################################

proc BuildNodeListMapProc {node} {
    global nodeList
    lappend nodeList $node
    return 1
}

proc ForTheModel {model} {
    global nodeList
    MapRole $model "node" "BuildNodeListMapProc" 
    return 1
}

proc CompareNode { node1 node2 } {
    set name1 [Get $node1 name]
    set name2 [Get $node2 name]
    
    set res [string compare $name1 $name2]
    return $res
}

proc BuildNodeList {} {
    global scade nodeList
    set nodeList {}
    set item [lindex $scade 0]
    set vmodel [GetRole $item model]
    ForTheModel $vmodel
    return [lsort -command CompareNode $nodeList]
}

#'############################################################
#' Get Node by name
#'############################################################

proc GetNodeByName { nodeList searchNodeName } {
    foreach node $nodeList {
	set nodeName [Get $node name]
	if { $searchNodeName == $nodeName} {
	    return $node
	    break;
	}
    }
}

#'############################################################
#' Build Luc list
#'############################################################

proc BuildFileLucListMapProc {fileRef} {
    global lucList
    set fileName [Get $fileRef name]
    set listSplitFileName [split $fileName .]
    set extension [lindex $listSplitFileName 1]
    if {$extension == "luc"} {
    	lappend lucList $fileRef
    }
    return 1
}

proc BuildLucList {} {
    global project lucList
    set lucList {}
    set item [lindex $project 0]
    MapRole $item fileRef BuildFileLucListMapProc
    
    return [lsort -command CompareNode $lucList]
}

#'############################################################
#' Build RIF files list
#'############################################################


proc BuildRifFileListMapProc {fileRef} {
    global rifList
    set fileName [Get $fileRef name]
    set listSplitFileName [split $fileName .]
    set extension [lindex $listSplitFileName 1]
    if {$extension == "rif"} {
	lappend rifList $fileRef
    } 
    return 1
}

proc BuildRifFileList {} {
    global project rifList
    set rifList {}
    set item [lindex $project 0]
    MapRole $item fileRef BuildRifFileListMapProc
    
    return  $rifList
}


#'############################################################
#' Build I/O list from a node
#'############################################################

proc ForEachInput {in} {
    global inputList
    lappend inputList $in
    return 1
}

proc ForEachOutput {out} {
    global outputList
    lappend outputList $out
    return 1
}

proc BuildInputList {node} {
    global inputList
    set inputList {}
    MapRole $node input ForEachInput
    return $inputList
}

proc BuildOutputList {node} {
    global outputList
    set outputList {}
    MapRole $node output ForEachOutput
    return $outputList
}

#######################################################################
# GUI custom code						      #
#######################################################################

#######################################################################
#
# Dialog Window that lets one set the main testing parameters
#

declare dialog -variable "LuretteSettingsDlg" -name "Lurette settings" \
    -w 580 -h 230 -OnBuildDialogBox OnBuildDialogLuretteSettings \
    -OnInitDialogBox OnInitDialogLuretteSettings \
    -OnCloseDialogBox OnCloseDialogLuretteSettings \
    -CloseCmdDialogBox CloseCmdDialogLuretteSettings

proc OnBuildDialogLuretteSettings { adrDialog } {
    global ComboBoxSUT ComboBoxOracle ComboBoxEnv \
	EditTestLength lurette_step_nb RadioButtonOracle RadioButtonNoOracle
    
    Control -class Button -variable "ButtonOK" -parent "$adrDialog" \
	-name "OK" \
	-x 150 -y 180 -w 72 -h 20 \
	-command OnClickButtonOkSettings

    Control -class Label -variable "Label1" -parent "$adrDialog" \
	-name "SUT node:" \
	-x 13 -y 20 -w 100 -h 20
    Control -class ObjComboBox -variable "ComboBoxSUT" -parent "$adrDialog" \
	-x 100 -y 20 -w 300 -h 800 \
	-selected OnSelectedComboBoxSUT \
	-selchangecmd OnSelChangeCmdComboBoxSUT \
	-editchangecmd OnEditChangeCmdComboBoxSUT

    
    Control -class RadioButton -variable "RadioButtonOracle" -parent "$adrDialog" \
	-name "Oracle node:" \
	-command OnClickRadioButtonOracle\
	-x 13 -y 60 -w 85 -h 20
    Control -class ObjComboBox -variable "ComboBoxOracle" \
	-parent "$adrDialog" \
	-x 100 -y 60 -w 300 -h 700 \
	-selected OnSelectedComboBoxOracle \
	-selchangecmd OnSelChangeCmdComboBoxOracle \
	-editchangecmd OnEditChangeCmdComboBoxOracle
    Control -class RadioButton -variable "RadioButtonNoOracle" \
	-parent "$adrDialog" -name "No oracle" \
	-x 410 -y 60 -w 90 -h 20 \
	-command OnClickRadioButtonNoOracle
    
    
    Control -class Label -variable "Label3" -parent "$adrDialog" \
	-name "Environment:" \
	-x 12 -y 100 -w 75 -h 20        
    Control -class ObjComboBox -variable "ComboBoxEnv" \
	-parent "$adrDialog" \
	-x 100 -y 100 -w 300 -h 700 \
	-selected OnSelectedComboBoxEnv \
	-selchangecmd OnSelChangeCmdComboBoxEnv \
	-editchangecmd OnEditChangeCmdComboBoxEnv
    Control -class Button -variable "gen_env_buton" -parent "$adrDialog" \
	-name "gen. corresp. environments" \
	-tooltip "generate environments for the curent node" \
	-imagefile "./Images/env.bmp" \
	-command GenLucSettings \
	-x 410 -y 20 -w 150 -h 20
    
    
    Control -class Label -variable "Label4" -parent "$adrDialog" \
	-name "Test length:" \
	-x 13 -y 140 -w 74 -h 20
    Control -class Edit -variable "EditTestLength" -name "$lurette_step_nb" \
	-parent "$adrDialog" \
	-x 100 -y 140 -w 100 -h 20


    Control -class Button -variable "ButtonCancel" -parent "$adrDialog" \
	-name "Cancel" \
	-x 280 -y 180 -w 72 -h 20 \
	-command OnClickButtonCancel
    
    Group $RadioButtonOracle $RadioButtonNoOracle
}

proc GenLucSettings { } {
    global ComboBoxSUT ComboBoxEnv project

    CheckConf
    set firstProject [lindex $project 0]
    set luretteConf [GetLuretteConfiguration]
    set luretteEnvNodeName \
	[GetToolProp $firstProject $luretteConf "LURETTE" "lurette_env" ""]

    set node [Get [$ComboBoxSUT -selected] name]
    GenLuc $node

    set list [BuildLucList]
    $ComboBoxEnv -modellist $list
    $ComboBoxEnv -selected [GetNodeByName $list $luretteEnvNodeName]
}

# ccc

proc OnInitDialogLuretteSettings { $dummy } {
    global ComboBoxSUT ComboBoxOracle ComboBoxEnv  project lurette_sut \
	configuration lurette_dir lurette_projectPathName EditTestLength \
	RadioButtonOracle RadioButtonNoOracle
    
    CheckConf

    set firstProject [lindex $project 0]
    set luretteConf [GetLuretteConfiguration]
    set luretteSutNodeName \
	[GetToolProp $firstProject $luretteConf "LURETTE" "lurette_sut" ""]
    set luretteOracleNodeName \
	[GetToolProp $firstProject $luretteConf "LURETTE" "lurette_oracle" ""]
    set luretteUseOracle \
	[GetToolProp $firstProject $luretteConf "LURETTE" "lurette_use_oracle" ""]
    set luretteEnvNodeName \
	[GetToolProp $firstProject $luretteConf "LURETTE" "lurette_env" ""]
    set luretteStepNb \
	[GetToolProp $firstProject $luretteConf "LURETTE" "lurette_step_nb" ""]

 

    set lurette_projectPathName [Get [lindex $project 0] pathname]
    set lurette_dir [file dirname $lurette_projectPathName]
    
    set  list [BuildNodeList]
    $ComboBoxSUT -modellist $list
    $ComboBoxSUT -selected [GetNodeByName $list $luretteSutNodeName]

	

    if { $luretteUseOracle == "true" } {
	$RadioButtonOracle -check 1
	$RadioButtonNoOracle -check 0
	$ComboBoxOracle -modellist $list
	$ComboBoxOracle -selected [GetNodeByName $list $luretteOracleNodeName]

    } else {
	$RadioButtonOracle -check 0
	$RadioButtonNoOracle -check 1
 	$ComboBoxOracle -modellist ""
 	$ComboBoxOracle -selected ""
    }

    set list [BuildLucList]
    $ComboBoxEnv -modellist $list
    $ComboBoxEnv -selected [GetNodeByName $list $luretteEnvNodeName]

    $EditTestLength  -name "$luretteStepNb"
}

proc OnCloseDialogLuretteSettings {} {}

proc CloseCmdDialogLuretteSettings {} {}

#' widget function(s)

proc OnClickButtonOkSettings {} {
    global ComboBoxSUT ComboBoxOracle ComboBoxEnv EditTestLength project \
	lurette_projectPathName lurette_sut lurette_oracle lurette_use_oracle \
	lurette_env lurette_step_nb project configuration rifbase_filename \
	RadioButtonOracle RadioButtonNoOracle

    catch { set lurette_sut [Get [$ComboBoxSUT -selected] name] } {
	output "*** software error: set lurette_sut failed! \n"
    }
    catch { set lurette_oracle [Get [$ComboBoxOracle -selected] name] } {
	output "*** software error:  set lurette_oracle failed! \n"
    }
    catch { set lurette_env  [Get [$ComboBoxEnv -selected] name] }  {
	output "*** software error:  set lurette_env failed! \n"
    }

    CheckConf
    set firstProject [lindex $project 0]
    set luretteConf [GetLuretteConfiguration]


    SetToolProp $firstProject $luretteConf "LURETTE" "lurette_sut" $lurette_sut ""
    SetToolProp $firstProject $luretteConf "LURETTE" "lurette_oracle" \
	$lurette_oracle ""

    if {[$RadioButtonOracle -check]} {
	set lurette_use_oracle "true"
    } else {
	set lurette_use_oracle "false"
    }
    SetToolProp $firstProject $luretteConf "LURETTE" "lurette_use_oracle" \
	$lurette_use_oracle ""


    SetToolProp $firstProject $luretteConf "LURETTE" "lurette_env" $lurette_env ""

    if { [catch { set lurette_step_nb [$EditTestLength -name] } ] } {
	set lurette_step_nb 1
    }

    SetToolProp $firstProject $luretteConf "LURETTE" "lurette_step_nb" \
	$lurette_step_nb ""

    CloseCmdDialogLuretteSettings
}

proc OnSelectedComboBoxSUT {} {}
proc OnSelChangeCmdComboBoxSUT {} {}
proc OnEditChangeCmdComboBoxSUT {} {}
proc OnSelectedComboBoxOracle {} {}
proc OnSelChangeCmdComboBoxOracle {} {}
proc OnEditChangeCmdComboBoxOracle {} {}
proc OnSelectedComboBoxEnv {} {}
proc OnSelChangeCmdComboBoxEnv {} {}
proc OnEditChangeCmdComboBoxEnv {} {}
proc OnClickButtonCancel {} {
    CloseCmdDialogLuretteSettings
}

proc OnClickRadioButtonOracle {} { 
    global RadioButtonOracle RadioButtonNoOracle ComboBoxOracle project

    # We need to clean to make sure the executable do contain the
    # oracle functions.
    CleanLuretteTop
    $RadioButtonNoOracle -check 0
    $RadioButtonOracle -check 1

    set proj [lindex $project 0]
    CheckConf
    set luretteConf [GetLuretteConfiguration]
    set luretteOracleNodeName \
	[GetToolProp $proj $luretteConf "LURETTE" "lurette_oracle" ""]
    set list [BuildNodeList]
    
    $ComboBoxOracle -modellist $list
    $ComboBoxOracle -selected [GetNodeByName $list $luretteOracleNodeName]
    
    
}
proc OnClickRadioButtonNoOracle {} { 
    global RadioButtonOracle RadioButtonNoOracle ComboBoxOracle

    # it should not be necessary to clean in that case...
    # CleanLuretteTop
    $RadioButtonNoOracle -check 1
    $RadioButtonOracle -check 0
    $ComboBoxOracle -selected ""
}


##############################################################################
#
# Dialog Window that let one set the auxiliary testing parameters
#

Declare dialog -variable "DialogLurettePref" -name "Lurette Preferences" \
    -w 500 -h 550 -OnBuildDialogBox OnBuildDialogLurettePref \
    -OnInitDialogBox OnInitDialogLurettePref \
    -OnCloseDialogBox OnCloseDialogLurettePref \
    -CloseCmdDialogBox CloseCmdDialogLurettePref

# build function

proc OnBuildDialogLurettePref { adrDialog } {
    global Label1 RadioButtonFormulaNumberOne RadioButtonFormulaNumberAll \
	RadioButtonAllVertices RadioButtonSomeVertices \
	RadioButtonSeed1 RadioButtonSeed2 \
	RadioButtonLocal1 RadioButtonLocal2 \
	RadioButtonVerb1 RadioButtonVerb2 RadioButtonVerb3 RadioButtonVerb4 \
	RadioButtonSM1 RadioButtonSM2 RadioButtonSM3 \
	ButtonOKPref ButtonCancelPref \
	lurette_sut lurette_oracle lurette_use_oracle lurette_env lurette_step_nb \
	EditDNB EditDNI EditDNE EditDNV \
	EditSeed EditPP EditPrecision EditRifbase \
	dnb dne dni dnv seed pp precision rifbase_filename

#	RadioButtonStep1 RadioButtonStep2 \

# Ok button
    Control -class Button -variable "ButtonOKPref" -parent "$adrDialog" \
	-name "OK" -x 126 -y 470 -w 91 -h 22 -command OnClickButtonOkPref

# Cancel button
    Control -class Button -variable "ButtonCancelPref" -parent "$adrDialog" \
	-name "Cancel" -x 243 -y 470 -w 77 -h 20 -command OnClickButtonCancelPref

#### Thickness parameters 
    Control -class Label -variable "Label2" -parent "$adrDialog" \
	-name "Test thickness parameters" -x 15 -y 10 -w 230 -h 20

# Draw Number Formula
    Control -class Label -variable "dnfLabel" -parent "$adrDialog" \
	-name "Draw Formula number:" -x 70 -y 40 -w 122 -h 20

    Control -class RadioButton -variable "RadioButtonFormulaNumberOne" \
	-parent "$adrDialog" -name "One" -x 274 -y 40 -w 57 -h 20 \
	-command OnClickRadioButtonFormulaNumberOne
    Control -class RadioButton -variable "RadioButtonFormulaNumberAll" \
	-parent "$adrDialog" -name "All" -x 347 -y 40 -w 57 -h 20 \
	-command OnClickRadioButtonFormulaNumberAll

# Draw Number Boolean
    Control -class Label -variable "DNBLabel" -parent "$adrDialog" \
	-name "Draw number for Boolean variables:" -x 70 -y 70 -w 180 -h 16
    Control -class Edit -name "$dnb" -variable "EditDNB" -parent "$adrDialog" \
	-x 273 -y 70 -w 130 -h 20

# Draw Number Inside
    Control -class Label -variable "DNILabel" -parent "$adrDialog" \
	-name "Draw number Inside polyhedron:" -x 70 -y 100 -w 180 -h 16
    Control -class Edit -name "$dni" -variable "EditDNI" -parent "$adrDialog" \
	-x 273 -y 100 -w 130 -h 20

# Draw Number Edge
    Control -class Label -variable "DNELabel" -parent "$adrDialog" \
	-name "Draw number at polyhedron edges:" -x 70 -y 130 -w 180 -h 16
    Control -class Edit -name "$dne" -variable "EditDNE" -parent "$adrDialog" \
	-x 273 -y 130 -w 130 -h 20

# Draw Number vertices
    Control -class Label -variable "DNVLabel" -parent "$adrDialog" \
	-name "Draw number at polyhedron vertices:" -x 70 -y 160 -w 180 -h 16

    Control -class RadioButton -variable "RadioButtonAllVertices" \
	-parent "$adrDialog" \
	-name "All" -x 280 -y 160 -w 60 -h 20 -command OnClickRadioButtonAllVertices
    Control -class RadioButton -variable "RadioButtonSomeVertices" \
	-parent "$adrDialog" -name "Some" -x 350 -y 160 -w 60 -h 20 \
	-command OnClickRadioButtonSomeVertices
    Control -class Edit -variable "EditDNV" -name "$dnv" -parent "$adrDialog" \
	-x 420 -y 160 -w 60 -h 20

# Step mode 
    Control -class Label -variable "StepModeLabel" -parent "$adrDialog" \
	-name "Step Mode:" -x 20 -y 200 -w 90 -h 20

    Control -class RadioButton -variable "RadioButtonSM1" -parent "$adrDialog" \
	-name "Inside" -x 150 -y 200 -w 67 -h 20 -command OnClickRadioButtonSM1
    Control -class RadioButton -variable "RadioButtonSM2" -parent "$adrDialog" \
	-name "Edges" -x 230 -y 200 -w 67 -h 20 -command OnClickRadioButtonSM2
    Control -class RadioButton -variable "RadioButtonSM3" -parent "$adrDialog" \
	-name "Vertices" -x 310 -y 200 -w 67 -h 20 -command OnClickRadioButtonSM3

# Seed mode 
    Control -class Label -variable "SeedLabel" -parent "$adrDialog" \
	-name "Random Engine Seed:" -x 20 -y 230 -w 190 -h 20

    Control -class RadioButton -variable "RadioButtonSeed1" -parent "$adrDialog" \
	-name "Random" -x 150 -y 230 -w 67 -h 20 -command OnClickRadioButtonSeed1
    Control -class RadioButton -variable "RadioButtonSeed2" -parent "$adrDialog" \
	-name "Manual" -x 230 -y 230 -w 67 -h 20 -command OnClickRadioButtonSeed2
    Control -class Edit -variable "EditSeed" -name "$seed" -parent "$adrDialog" \
	-x 300 -y 230 -w 100 -h 20

# Step by step  mode 
#     Control -class Label -variable "StepByStepLabel" -parent "$adrDialog" \
# 	-name "Step by step run:" -x 20 -y 260 -w 190 -h 20

#     Control -class RadioButton -variable "RadioButtonStep1" -parent "$adrDialog" \
# 	-name "Yes" -x 150 -y 260 -w 67 -h 20 -command OnClickRadioButtonStep1
#     Control -class RadioButton -variable "RadioButtonStep2" -parent "$adrDialog" \
# 	-name "No" -x 230 -y 260 -w 67 -h 20 -command OnClickRadioButtonStep2

# Local vars 
    Control -class Label -variable "LocalVarLabel" -parent "$adrDialog" \
	-name "show the Env local vars:" -x 20 -y 290 -w 190 -h 20

    Control -class RadioButton -variable "RadioButtonLocal1" -parent "$adrDialog" \
	-name "Yes" -x 150 -y 290 -w 67 -h 20 -command OnClickRadioButtonLocal1
    Control -class RadioButton -variable "RadioButtonLocal2" -parent "$adrDialog" \
	-name "No" -x 230 -y 290 -w 67 -h 20 -command OnClickRadioButtonLocal2

# Verbose mode 
    Control -class Label -variable "VerboseLabel" -parent "$adrDialog" \
	-name "Verbose Mode:" -x 20 -y 320 -w 190 -h 20

    Control -class RadioButton -variable "RadioButtonVerb1" -parent "$adrDialog" \
	-name "Quiet" -x 150 -y 320 -w 67 -h 20 -command OnClickRadioButtonVerb1
    Control -class RadioButton -variable "RadioButtonVerb2" -parent "$adrDialog" \
	-name "Wordy" -x 230 -y 320 -w 67 -h 20 -command OnClickRadioButtonVerb2
    Control -class RadioButton -variable "RadioButtonVerb3" -parent "$adrDialog" \
	-name "Talkative" -x 310 -y 320 -w 67 -h 20 -command OnClickRadioButtonVerb3
    Control -class RadioButton -variable "RadioButtonVerb4" -parent "$adrDialog" \
	-name "Logorrheic" -x 390 -y 320 -w 67 -h 20 -command OnClickRadioButtonVerb4

# Pre-processor
    Control -class Label -variable "PPLabel" -parent "$adrDialog" \
	-name "Pre processor:" -x 20 -y 350 -w 180 -h 20
    Control -class Edit -variable "EditPP" -parent "$adrDialog" \
	-name "$pp" -x 150 -y 350 -w 130 -h 20

# Precision
    Control -class Label -variable "PrecisionLabel" -parent "$adrDialog" \
	-name "Precision:" -x 20 -y 380 -w 180 -h 20
    Control -class Edit -variable "EditPrecision" -parent "$adrDialog" \
	-name "$precision" \
	-x 150 -y 380 -w 130 -h 20
    
# Base rif file name

    set lurette_env_base [file rootname lurette_env]

    Control -class Label -variable "RifBaseLabel" -parent "$adrDialog" \
	-name "Base rif file name:" -x 20 -y 410 -w 180 -h 20
    Control -class Edit -variable "EditRifbase" -parent "$adrDialog" \
	-name "$lurette_sut-$lurette_env_base-$lurette_step_nb-" \
	-x 150 -y 410 -w 230 -h 20


### Grouping Radio buttions
    Group $RadioButtonFormulaNumberOne $RadioButtonFormulaNumberAll
    Group $RadioButtonSM1 $RadioButtonSM2 $RadioButtonSM3
    Group $RadioButtonSeed1 $RadioButtonSeed2
#    Group $RadioButtonStep1 $RadioButtonStep2
    Group $RadioButtonLocal1 $RadioButtonLocal2
    Group $RadioButtonVerb1 $RadioButtonVerb2 $RadioButtonVerb3 $RadioButtonVerb4
    Group $RadioButtonAllVertices $RadioButtonSomeVertices
}

# window function(s)

proc OnInitDialogLurettePref { $dummy } {
# A default initialisation of all preference parameters is done at the beginning
# of this file
    global project \
	RadioButtonFormulaNumberOne RadioButtonFormulaNumberAll \
	RadioButtonAllVertices RadioButtonSomeVertices \
	RadioButtonSeed1 RadioButtonSeed2 \
	RadioButtonLocal1 RadioButtonLocal2 \
	RadioButtonVerb1 RadioButtonVerb2 RadioButtonVerb3 RadioButtonVerb4 \
	RadioButtonSM1 RadioButtonSM2 RadioButtonSM3 \
	draw_all_formula draw_all_vertices \
	dnb dne dni dnv \
	step_mode seed_is_random seed verbose_level display_local \
	EditDNB EditDNI EditDNE EditDNV \
	EditSeed EditPP EditPrecision EditRifbase

    CheckConf
    set firstProject [lindex $project 0]
    set luretteConf [GetLuretteConfiguration]


    ### Retrieving parameters from the Lurette configuration
    set dni \
	[GetToolProp $firstProject $luretteConf "LURETTE" "lurette_dni" ""]
    set dnb \
	[GetToolProp $firstProject $luretteConf "LURETTE" "lurette_dnb" ""]
    set dne \
	[GetToolProp $firstProject $luretteConf "LURETTE" "lurette_dne" ""]
    set dnv \
	[GetToolProp $firstProject $luretteConf "LURETTE" "lurette_dnv" ""]
    set conf_seed \
	[GetToolProp $firstProject $luretteConf "LURETTE" "lurette_seed" ""]
    set lurettePP \
	[GetToolProp $firstProject $luretteConf "LURETTE" "lurette_pp" ""]
    set lurettePrecision \
	[GetToolProp $firstProject $luretteConf "LURETTE" "lurette_precision" ""]

    set draw_all_formula [GetToolProp $firstProject $luretteConf \
				  "LURETTE" "lurette_draw_all_formula" ""]
    set draw_all_vertices [GetToolProp $firstProject $luretteConf \
				  "LURETTE" "lurette_draw_all_vertices" ""]
    set seed_is_random [GetToolProp $firstProject $luretteConf \
				  "LURETTE" "lurette_seed_is_random" ""]
    set verbose_level [GetToolProp $firstProject $luretteConf \
				  "LURETTE" "lurette_verbose_level" ""]


    set display_local [GetToolProp $firstProject $luretteConf \
				  "LURETTE" "lurette_display_local" ""]
    set step_mode [GetToolProp $firstProject $luretteConf \
				  "LURETTE" "lurette_step_mode" ""]

    set rifbase_filename [GetToolProp $firstProject $luretteConf \
				  "LURETTE" "lurette_rifbase_filename" ""]

    $EditDNI  -name "$dni"
    $EditDNB  -name "$dnb"
    $EditDNE  -name "$dne"
    $EditDNV  -name "$dnv"
    $EditPP   -name "$lurettePP"
    $EditPrecision  -name "$lurettePrecision"
    $EditRifbase  -name "$rifbase_filename"

    # Take the seed of the lurette conf for the first run only.
    # Otherwise, we take that last generated seed
    if { $seed == 0 } {
	$EditSeed -name "$conf_seed"
	set seed $conf_seed
    }

    if { $draw_all_formula == "false" } {
	$RadioButtonFormulaNumberOne -check 1
	$RadioButtonFormulaNumberAll -check 0
    } else {
	$RadioButtonFormulaNumberOne -check 0
	$RadioButtonFormulaNumberAll -check 1
    }

    if { $draw_all_vertices == "true" } {
	$RadioButtonAllVertices -check 1
	$RadioButtonSomeVertices -check 0
    } else {
	$RadioButtonAllVertices -check 0
	$RadioButtonSomeVertices -check 1
    }

    if { $seed_is_random == "true" } {
	$RadioButtonSeed1 -check 1
	$RadioButtonSeed2 -check 0
    } else {
	$RadioButtonSeed1 -check 0
	$RadioButtonSeed2 -check 1
	
    }

    if { $verbose_level == 0 } {
	$RadioButtonVerb1 -check 1
	$RadioButtonVerb2 -check 0
	$RadioButtonVerb3 -check 0
	$RadioButtonVerb4 -check 0
    } elseif { $verbose_level == 1  } {
	$RadioButtonVerb1 -check 0
	$RadioButtonVerb2 -check 1
	$RadioButtonVerb3 -check 0
	$RadioButtonVerb4 -check 0
    } elseif { $verbose_level == 2  } {
	$RadioButtonVerb1 -check 0
	$RadioButtonVerb2 -check 0
	$RadioButtonVerb3 -check 1
	$RadioButtonVerb4 -check 0
    } else {
	$RadioButtonVerb1 -check 0
	$RadioButtonVerb2 -check 0
	$RadioButtonVerb3 -check 0
	$RadioButtonVerb4 -check 1
    }

        
    if { $display_local == "true" } {
	$RadioButtonLocal1 -check 1
	$RadioButtonLocal2 -check 0
    } else {
	$RadioButtonLocal1 -check 0
	$RadioButtonLocal2 -check 1
    }
    
    if { $step_mode == "inside" } {
	$RadioButtonSM1 -check 1
	$RadioButtonSM2 -check 0
	$RadioButtonSM3 -check 0
    } elseif { $step_mode == "edges" } {
	$RadioButtonSM1 -check 0
	$RadioButtonSM2 -check 1
	$RadioButtonSM3 -check 0
    } else {
	$RadioButtonSM1 -check 0
	$RadioButtonSM2 -check 0
	$RadioButtonSM3 -check 1
    }



}

proc OnCloseDialogLurettePref {} { }
proc CloseCmdDialogLurettePref {} { }

# widget function(s)
proc OnClickRadioButtonFormulaNumberOne {} { 
    global RadioButtonFormulaNumberOne RadioButtonFormulaNumberAll
    $RadioButtonFormulaNumberOne -check 1
    $RadioButtonFormulaNumberAll -check 0
}
proc OnClickRadioButtonFormulaNumberAll {} { 
    global RadioButtonFormulaNumberOne RadioButtonFormulaNumberAll
    $RadioButtonFormulaNumberOne -check 0
    $RadioButtonFormulaNumberAll -check 1
}

proc OnClickRadioButtonAllVertices {} { 
    global RadioButtonAllVertices RadioButtonSomeVertices
    $RadioButtonAllVertices -check 1
    $RadioButtonSomeVertices -check 0
}
proc OnClickRadioButtonSomeVertices {} { 
    global RadioButtonAllVertices RadioButtonSomeVertices

    $RadioButtonAllVertices -check 0
    $RadioButtonSomeVertices -check 1
}

# proc OnClickRadioButtonStep1 {} { 
#     global RadioButtonStep1 RadioButtonStep2
#     $RadioButtonStep1 -check 1
#     $RadioButtonStep2 -check 0
# }
# proc OnClickRadioButtonStep2 {} { 
#     global RadioButtonStep1 RadioButtonStep2

#     $RadioButtonStep1 -check 0
#     $RadioButtonStep2 -check 1
# }

proc OnClickRadioButtonSeed1 {} { 
    global RadioButtonSeed1 RadioButtonSeed2
    $RadioButtonSeed1 -check 1
    $RadioButtonSeed2 -check 0
}
proc OnClickRadioButtonSeed2 {} { 
    global RadioButtonSeed1 RadioButtonSeed2

    $RadioButtonSeed1 -check 0
    $RadioButtonSeed2 -check 1
}

proc OnClickRadioButtonVerb1 {} { 
    global RadioButtonVerb1 RadioButtonVerb2 RadioButtonVerb3 RadioButtonVerb4
    $RadioButtonVerb1 -check 1
    $RadioButtonVerb2 -check 0
    $RadioButtonVerb3 -check 0
    $RadioButtonVerb4 -check 0
}
proc OnClickRadioButtonVerb2 {} { 
    global RadioButtonVerb1 RadioButtonVerb2 RadioButtonVerb3 RadioButtonVerb4
    $RadioButtonVerb1 -check 0
    $RadioButtonVerb2 -check 1
    $RadioButtonVerb3 -check 0
    $RadioButtonVerb4 -check 0
}
proc OnClickRadioButtonVerb3 {} { 
    global RadioButtonVerb1 RadioButtonVerb2 RadioButtonVerb3 RadioButtonVerb4
    $RadioButtonVerb1 -check 0
    $RadioButtonVerb2 -check 0
    $RadioButtonVerb3 -check 1
    $RadioButtonVerb4 -check 0
}
proc OnClickRadioButtonVerb4 {} { 
    global RadioButtonVerb1 RadioButtonVerb2 RadioButtonVerb3 RadioButtonVerb4
    $RadioButtonVerb1 -check 0
    $RadioButtonVerb2 -check 0
    $RadioButtonVerb3 -check 0
    $RadioButtonVerb4 -check 1
}

proc OnClickRadioButtonLocal1 {} { 
    global RadioButtonLocal1 RadioButtonLocal2
    $RadioButtonLocal1 -check 1
    $RadioButtonLocal2 -check 0
}
proc OnClickRadioButtonLocal2 {} { 
    global RadioButtonLocal1 RadioButtonLocal2
    $RadioButtonLocal1 -check 0
    $RadioButtonLocal2 -check 1
}

proc OnClickRadioButtonSM1 {} { 
    global RadioButtonSM1 RadioButtonSM2 RadioButtonSM3
    
    $RadioButtonSM1 -check 1
    $RadioButtonSM2 -check 0
    $RadioButtonSM3 -check 0
}
proc OnClickRadioButtonSM2 {} { 
    global RadioButtonSM1 RadioButtonSM2 RadioButtonSM3
    $RadioButtonSM1 -check 0
    $RadioButtonSM2 -check 1
    $RadioButtonSM3 -check 0
}
proc OnClickRadioButtonSM3 {} { 
    global RadioButtonSM1 RadioButtonSM2 RadioButtonSM3
    $RadioButtonSM1 -check 0
    $RadioButtonSM2 -check 0
    $RadioButtonSM3 -check 1
}


proc OnClickButtonOkPref {} {
    global project \
	RadioButtonFormulaNumberOne draw_all_formula \
	RadioButtonAllVertices RadioButtonSomeVertices \
	RadioButtonSeed1 RadioButtonSeed2 \
	RadioButtonLocal1 RadioButtonLocal2 \
	RadioButtonVerb1 RadioButtonVerb2 RadioButtonVerb3 RadioButtonVerb4 \
	RadioButtonSM1 RadioButtonSM2 RadioButtonSM3 \
	EditDNB EditDNI EditDNE EditDNV \
	EditSeed EditPP EditPrecision EditRifbase \
	draw_all_vertices \
	dnb dne dni dnv pp precision rifbase_filename \
	step_mode seed_is_random seed verbose_level \
	display_local pp lurette_sut lurette_env lurette_step_nb 
    

    set dni [$EditDNI -name]    
    set dnb [$EditDNB -name]
    set dne [$EditDNE -name]
    set dnv [$EditDNV -name]
    set seed [$EditSeed -name]
    set pp [$EditPP -name]
    set precision [$EditPrecision -name]
    set rifbase_filename [$EditRifbase -name]


    if {[$RadioButtonFormulaNumberOne -check]} {
	set draw_all_formula "false"
    } else {
	set draw_all_formula "true"
    }

    if {[$RadioButtonAllVertices -check]} {
	set draw_all_vertices "true"
    } else {
	set draw_all_vertices "false"
    }

    if {[$RadioButtonSeed1 -check]} {
	set seed_is_random "true"
    } else {
	set seed_is_random "false"
    }

    if {[$RadioButtonLocal1 -check]} {
	set display_local "true"
    } else {
	set display_local "false"
    }

    if {[$RadioButtonSM1 -check]} {
	set step_mode "inside"
    } elseif {[$RadioButtonSM2 -check]} {
	set step_mode "edges"
    } else {
	set step_mode "vertices"
    }


    if {[$RadioButtonVerb1 -check]} {
	set verbose_level "0"
    } elseif {[$RadioButtonVerb2 -check]} {
	set verbose_level "1"
    } elseif {[$RadioButtonVerb3 -check]} {
	set verbose_level "2"
    } else {
	set verbose_level "3"
    }


    $EditDNI -name "$dni"
    $EditDNB -name "$dnb"
    $EditDNE -name "$dne"
    $EditDNV -name "$dnv"
    $EditSeed -name "$seed"
    $EditPP -name "$pp"
    $EditPrecision -name "$precision"
	
     if { $rifbase_filename != "" } {
 	$EditRifbase -name "$rifbase_filename"
     }

    CheckConf
    set prj [lindex $project 0]
    set lConf [GetLuretteConfiguration]

    SetToolProp $prj $lConf "LURETTE" "lurette_dni" $dni ""
    SetToolProp $prj $lConf "LURETTE" "lurette_dnb" $dnb ""
    SetToolProp $prj $lConf "LURETTE" "lurette_dne" $dne ""
    SetToolProp $prj $lConf "LURETTE" "lurette_dnv" $dnv ""
    SetToolProp $prj $lConf "LURETTE" "lurette_seed" $seed ""
    SetToolProp $prj $lConf "LURETTE" "lurette_pp" $pp ""
    SetToolProp $prj $lConf "LURETTE" "lurette_precision" $precision ""

    SetToolProp $prj $lConf "LURETTE" "lurette_draw_all_formula" \
	$draw_all_formula ""
    SetToolProp $prj $lConf "LURETTE" "lurette_draw_all_vertices" \
	$draw_all_vertices ""
    SetToolProp $prj $lConf "LURETTE" "lurette_seed_is_random" $seed_is_random ""
    SetToolProp $prj $lConf "LURETTE" "lurette_verbose_level" $verbose_level ""
    SetToolProp $prj $lConf "LURETTE" "lurette_display_local" $display_local ""
    SetToolProp $prj $lConf "LURETTE" "lurette_step_mode" $step_mode ""
    SetToolProp $prj $lConf "LURETTE" "lurette_rifbase_filename" $rifbase_filename ""



    update

    CloseCmdDialogLurettePref
}

proc OnClickButtonCancelPref {} {
    CloseCmdDialogLurettePref
}



#######################################################################
#
# dialog AboutDlg ######################################################

declare dialog -variable "AboutDlg" -name "About Lurette" \
    -w 350 -h 130 -OnBuildDialogBox OnBuildAboutDlg \
    -OnInitDialogBox OnInitAboutDlg -OnCloseDialogBox OnCloseAboutDlg \
    -CloseCmdDialogBox CloseCmdAboutDlg

# build function

proc OnBuildAboutDlg { adrDialog } {
    global ButtonOk ScriptsDirectory ScadeLuretteVersion
    
    Control -class Label -variable "Label1" -parent "$adrDialog" \
	-name "Lurette/SCADE integration\nVersion : $ScadeLuretteVersion"\
	-x 100 -y 13 -w 290 -h 90

    Control -class Button -variable "ButtonOk" -parent "$adrDialog" -name "OK" \
	-x 250 -y 73 -w 70 -h 22 -command OnClickButtonOk

    set imagePath [file join $ScriptsDirectory ".\\Images\\aboutlurette.bmp"]

    Control -class Image -variable "Image1" -parent "$adrDialog" \
	-x 10 -y 13 -w 80 -h 82 -name "" -imagefile $imagePath
}

#' window function(s)

proc OnInitAboutDlg { $dummy } {}
proc OnCloseAboutDlg {} {}
proc CloseCmdAboutDlg {} {}

proc OnClickButtonOk {} {
    CloseCmdAboutDlg
}

# Command ################################################

proc isProjectLoaded {} {
    global project
    if { [llength $project] != 0 } {	
	return 1
    } else {
	return 0
    }
}

#############

Declare command -variable "OpenLuretteSettingsCmd" \
    -name "Open Lurette Settings..."  -imagefile ".\\Images\\config-params.bmp" \
    -statusmessage "Open Lurette Settings Dialog" \
    -tooltip "Open Lurette Settings Dialog" -accelerator "Ctrl+Shift+L" \
    -OnActivateCommand OnActivateLuretteSettingsCmd \
    -OnEnableCommand OnEnableLuretteSettingsCmd \
    -OnCheckCommand OnCheckLuretteSettingsCmd

proc OnActivateLuretteSettingsCmd { $dummy } {
    Dialog LuretteSettingsDlg
}

proc OnEnableLuretteSettingsCmd { $dummy } {
    return [isProjectLoaded]
}

proc OnCheckLuretteSettingsCmd { $dummy } {}

#############

Declare command -variable "OpenLurettePrefCmd" -name "Open Lurette Preference..." \
    -imagefile ".\\Images\\config-params2.bmp" \
    -statusmessage "Open Lurette Preference Dialog" \
    -tooltip "Open Lurette Preference Dialog" \
    -accelerator "Ctrl+Shift+P" \
    -OnActivateCommand OnActivateLurettePrefCmd \
    -OnEnableCommand OnEnableLurettePrefCmd \
    -OnCheckCommand OnCheckLurettePrefCmd

proc OnActivateLurettePrefCmd { $dummy } {
    Dialog DialogLurettePref
}

proc OnEnableLurettePrefCmd { $dummy } {
    return [isProjectLoaded]
}

proc OnCheckLurettePrefCmd { $dummy } {}

#############

Declare command -variable "AboutCmd" -name "About Lurette..."  \
    -imagefile "" \
    -statusmessage "View information about lurette" \
    -tooltip "View information about Lurette" \
    -accelerator "" \
    -OnActivateCommand OnActivateAboutCmd \
    -OnEnableCommand OnEnableAboutCmd \
    -OnCheckCommand OnCheckAboutCmd

proc OnActivateAboutCmd { $dummy } {
    Dialog AboutDlg
}

proc OnEnableAboutCmd { $dummy } {
    return 1
}
proc OnCheckAboutCmd { $dummy } {}

#############
# build 

Declare command -variable "BuildCmd" -name "Build" \
    -imagefile "./Images/build.bmp" \
    -statusmessage "Build the Lurette executable" \
    -tooltip "Build the Lurette executable" \
    -accelerator "" \
    -OnActivateCommand OnActivateBuildCmd \
    -OnEnableCommand OnEnableBuildCmd \
    -OnCheckCommand OnCheckBuildCmd



proc OnActivateBuildCmd { $dummy } {
   global ioc lurette_sut

    if { [LaunchLuretteTop] } {
	if { $lurette_sut == "" } {
	    Dialog LuretteSettingsDlg
	    Dialog DialogLurettePref
	}
	set build_cmd [BuildLurettetopCmds $lurette_sut]
	if { $build_cmd == "none" } {
	    return 0
	} else {
	    append build_cmd "build\n"
	    
	    puts $ioc "$build_cmd"
	    flush $ioc
	    update
	    return 1
	}
    } else {
	return 0
    }
 }

proc OnEnableBuildCmd { $dummy } { 
    return 1
}
proc OnCheckBuildCmd { $dummy } {}

#############
# rebuild 

Declare command -variable "RebuildCmd" -name "Rebuild" \
    -imagefile "./Images/rebuild.bmp" \
    -statusmessage "Rebuild the Lurette executable" \
    -tooltip "Rebuild the Lurette executable" \
    -accelerator "" \
    -OnActivateCommand OnActivateRebuildCmd \
    -OnEnableCommand OnEnableRebuildCmd \
    -OnCheckCommand OnCheckRebuildCmd


proc CleanLuretteTop { } {
    global ioc lurettetop_is_running lurette_sut

    if { $lurettetop_is_running } {
	if { $lurette_sut == "" } {
	    Dialog LuretteSettingsDlg
	    Dialog DialogLurettePref
	}
	set rebuild_cmd [BuildLurettetopCmds $lurette_sut]
	if {$rebuild_cmd == "none" } {
	    return 0
	} else {
	    append rebuild_cmd "\nclean\nbuild\n"
	    
	    puts $ioc "$rebuild_cmd"
	    flush $ioc
	    update
	    return 1
	}
    }
}

proc OnActivateRebuildCmd { $dummy } {
    CleanLuretteTop
 }
proc OnEnableRebuildCmd { $dummy } { 
    return 1
}
proc OnCheckRebuildCmd { $dummy } {}



#############
# stop_run 

Declare command -variable "StopRunCmd" -name "StopRun" \
    -imagefile "./Images/stop.bmp" \
    -statusmessage "Stop the current run (by killing the process)" \
    -tooltip "Stop the current run" \
    -accelerator "" \
    -OnActivateCommand OnActivateStopRunCmd \
    -OnEnableCommand OnEnableStopRunCmd \
    -OnCheckCommand OnCheckStopRunCmd


proc OnActivateStopRunCmd { $dummy } {
    LuretteStop
}

proc LuretteStop { } {
    global lurette_pid lurette_is_running

    if { $lurette_pid != 0 && $lurette_is_running } {
	# set kill_cmd "exec tskill $lurette_pid"
	set kill_cmd "exec tskill lurette"
	output "$kill_cmd\n"
	
	set res 1
	if { [ catch $kill_cmd result] } {
	    output "kill lurette process failed: $result\n"
	    set res 0
	}
	set lurette_is_running false
	set $lurette_pid 0
	return $res
    } else {
	output "Cannot kill PID $lurette_pid...\n"
	return 1
    }
}

proc OnEnableStopRunCmd { $dummy } { 
    global lurette_is_running
    
    return $lurette_is_running
}
proc OnCheckStopRunCmd { $dummy } {}


#############
# update 

# XXX to turn around a bug (in filevent ?)
Declare command -variable "UpdateCmd" -name "Update" \
    -imagefile "./Images/lupref.bmp" \
    -statusmessage "update" \
    -tooltip "Update the display" \
    -accelerator "" \
    -OnActivateCommand OnActivateUpdateCmd \
    -OnEnableCommand OnEnableUpdateCmd \
    -OnCheckCommand OnCheckUpdateCmd



proc OnActivateUpdateCmd { $dummy } {
    update
}
proc OnEnableUpdateCmd { $dummy } { 
    return 1
}
proc OnCheckUpdateCmd { $dummy } {}


#############
# quit 

proc LuretteQuit {} {
    global ioc log_oc log_ec lurettetop_is_running

    LuretteStop
    output "lurette: bye!\n"
    puts $ioc "\nquit\n"
    flush $ioc
    
    close $ioc
    close $log_oc
    close $log_ec
    return 1
}


#######################################################################
#
# Dialog Window for managing RIF files (view, delete, ...)
#

declare dialog -variable "RifManagementDlg" -name "RIF files management" \
    -w 700 -h 600 -OnBuildDialogBox OnBuildDialogRifManagement \
    -OnInitDialogBox OnInitDialogRifManagement \
    -OnCloseDialogBox OnCloseDialogRifManagement \
    -CloseCmdDialogBox CloseCmdDialogRifManagement

# build function

proc OnBuildDialogRifManagement { adrDialog } {
 global RifFilesListBox 

    Control -class Button -variable "GnuplotButton" -parent "$adrDialog" \
	-name " View Selected file with Gnuplot " \
     -x 450 -y 100 -w 170 -h 20 \
	-command OnClickButtonGnuplot

    Control -class ObjListBox -variable "RifFilesListBox" \
	-parent "$adrDialog" \
	-x 10 -y 10 -w 430 -h 500 \
	-selected OnSelectedCmdRif \
	-selchangecmd OnSelChangeCmdRif \
	-editchangecmd OnEditChangeCmdRif


    Control -class Button -variable "ButtonCancel" -parent "$adrDialog" \
	-name "Close window" \
	-x 450 -y 30 -w 72 -h 20 \
	-command OnClickButtonCancelGnuplot

#     Control -class Button -variable "ButtonDelete" -parent "$adrDialog" \
# 	-name "Delete file" \
# 	-x 450 -y 150 -w 72 -h 20 \
# 	-command OnClickButtonDeleteRif
}

proc OnSelectedCmdRif {} {}
proc OnSelChangeCmdRif  {} {}
proc OnEditChangeCmdRif {} {}

proc OnInitDialogRifManagement { $dummy } {
 global RifFilesListBox lurette_sut

    LaunchLuretteTop
    if {$lurette_sut == ""} {
	Dialog LuretteSettingsDlg
    }

    set rif_files_list [BuildRifFileList]    
    $RifFilesListBox -modellist $rif_files_list

    if { [catch { 
	set i [FindIndex [GetLastRifFileName] $rif_files_list]
	set lrfn [lindex $rif_files_list $i]
	$RifFilesListBox -selected $lrfn
    }] } {
	$RifFilesListBox -selected [lindex $rif_files_list 0]
    }    
}

proc OnCloseDialogRifManagement {} {}

proc CloseCmdDialogRifManagement {} {}

proc OnClickButtonCancelGnuplot {} {
    CloseCmdDialogRifManagement
}


# proc OnClickButtonDeleteRif {} {
#     global  RifFilesListBox
#     if { [catch {set selected_rif_file_ref \
# 		     [$RifFilesListBox -selected] } ] } {
# 	output "No file was selected\n"
#     } else {
# 	set selected_rif_file [Get $selected_rif_file_ref name]
# 	OutputDebug "Delete $selected_rif_file \n"
# 	file delete $selected_rif_file
# 	set rif_files_list [BuildRifFileList]    
# 	$RifFilesListBox -modellist $rif_files_list
#     }
# }

proc OnClickButtonGnuplot {} {
    global  RifFilesListBox

    LaunchLuretteTop

    if { [catch {set selected_rif_file [Get [$RifFilesListBox -selected] name] } ] } {
	output "No file was selected\n"
    } else {
	OutputDebug "Launch gnuplot with $selected_rif_file \n"
	LaunchGnuplot $selected_rif_file
    }
}

#############
# gnuplot 

Declare command -variable "GnuPlotCmd" -name "DisplayGnuplot" \
    -imagefile "./Images/gnuplot.bmp" \
    -statusmessage "Display generated data with Gnuplot" \
    -tooltip "Display generated data with Gnuplot" \
    -accelerator "" \
    -OnActivateCommand OnActivateGnuPlotCmd \
    -OnEnableCommand OnEnableGnuPlotCmd \
    -OnCheckCommand OnCheckGnuPlotCmd


# If rifbase_filename is empty, we compute a default rif name.
# In any cases, we make it sure it is unique using a counter
# that we increment if necessary.
proc GetBaseName { rifbase_filename } {
    global lurette_sut lurette_env lurette_step_nb

    if { $rifbase_filename == "" } {
	
	set listSplitFileName [split $lurette_env .]
	set lenv  [lindex $listSplitFileName 0]
    
	catch { set rifbase "$lurette_sut-$lenv-$lurette_step_nb-" } {
	    set rifbase "lurette"
	} 
    } else {
	set rifbase $rifbase_filename
    }

    return $rifbase
}

proc GetNewRifFileName  { } {
    global rifbase_filename 

    set rifbase [GetBaseName $rifbase_filename]
    set i 1
    while {[file exists "$rifbase$i.rif"] } {
	incr i
    }
    
    OutputDebugVerb "rif file name: $rifbase$i.rif ($rifbase_filename)\n"
    return "$rifbase$i.rif"
}

proc GetLastRifFileName  { } {
    global rifbase_filename 

    set rifbase [GetBaseName $rifbase_filename]
    set i 1
    while {[file exists "$rifbase$i.rif"] } {
	incr i
    }
    
    set j [expr $i-1]
    set last_rif "$rifbase$j.rif"

    return $last_rif
}

proc LaunchGnuplot { rif_file } {
    global ioc rifbase_filename lurette_env

    if { [file exists $rif_file] } {
	set cmd "\
set_output \"$rif_file\"\n\
set_env $lurette_env\n\
gnuplot\n\
"
	OutputDebugVerb "$cmd"
	puts $ioc "$cmd"
	flush $ioc
    } else {
	output "File $rif_file does not exist.\n"
    }
    update
   
}

proc OnActivateGnuPlotCmd { $dummy } {
    Dialog RifManagementDlg
#     set rif_file [GetLastRifFileName] 
#     LaunchGnuplot rif_file        
}

proc OnEnableGnuPlotCmd { $dummy } {
    global lurettetop_is_running
    
#    return $lurettetop_is_running
    return 1
}
proc OnCheckGnuPlotCmd { $dummy } {}

#############

Declare command -variable "RunCmd" -name "Run Lurette..."  \
    -imagefile "./Images/run.bmp" -statusmessage "Run Lurette" \
    -tooltip "Run Lurette" -accelerator "" \
    -OnActivateCommand OnActivateRunCmd \
    -OnEnableCommand OnEnableRunCmd \
    -OnCheckCommand OnCheckRunCmd


# build the lurettetop command that sets all the test parameters
proc BuildLurettetopCmds { lurette_sut } {
    global lurette_projectPathName project \
	lurette_oracle lurette_use_oracle lurette_env \
	lurette_step_nb seed precision \
	draw_all_formula draw_all_vertices \
	dnb dne dni dnv \
	step_mode seed_is_random verbose_level display_local pp \
	lurettetop_is_running

    set root_node [GetRootNode $project]	
    set proj [slashPath  $lurette_projectPathName]
    set rif_file [GetNewRifFileName] 
    
    set tmp_dir [ComputeTmpDir]

    if { $tmp_dir == "none" } {
	set lurettetop_is_running false
	return "none"
    } else {
	set tmp_dir_cmd "set_tmp_dir \"$tmp_dir\""
    }
### set_oracle \"$proj\" $lurette_oracle\n\

    if { $lurette_use_oracle == "false" || $lurette_oracle == "" } {
	set set_oracle_cmd "set_oracle \"\" \"\""
    } else {
	set set_oracle_cmd "set_oracle \"$proj\" $lurette_oracle"
    }
    
    if { $verbose_level == 0 } {
	set set_verbose_cmd "set_verbose 0"
    } else {
	set i [expr $verbose_level - 1]
	set set_verbose_cmd "set_verbose $i"	
    }

   if {  [expr $seed_is_random]  } {
	set set_seed_cmd "set_seed_randomly"
   } else {
	set set_seed_cmd "set_seed $seed"
    }

    set lurette_cmd "\
set_sut \"$proj\" $lurette_sut\n\
$set_oracle_cmd\n\
set_env $lurette_env\n\
set_test_length $lurette_step_nb\n\
set_root_node $root_node\n\
\
set_draw_all_formula $draw_all_formula\n\
set_draw_all_vertices $draw_all_vertices\n\
set_draw_nb $dnb\n\
set_draw_edges $dne\n\
set_draw_inside $dni\n\
set_draw_vertices $dnv\n\
set_step_mode $step_mode\n\
$set_seed_cmd\n\
$set_verbose_cmd\n\
set_display_local_var $display_local\n\
set_preprocessor \"$pp\"\n\
$tmp_dir_cmd\n\
\
set_precision $precision\n\
set_output \"$rif_file\"\n\
set_sut_compiler scade_gui\n\
set_oracle_compiler scade_gui\n\
set_show_step true\n\
\
\
\
"
    OutputDebugVerb $lurette_cmd
    return $lurette_cmd
}


proc OnActivateRunCmd { $dummy } {
    global lurette_sut ioc lurette_dir 
    
    CheckConf
    if { [LaunchLuretteTop] } {
    
	if { $lurette_sut == "" } {
	    Dialog LuretteSettingsDlg
	    Dialog DialogLurettePref
	}
		
	set lurette_cmd [BuildLurettetopCmds $lurette_sut]
	if { $lurette_cmd == "none" } {
	    return false
	}
	append lurette_cmd "run\n"
	
	OutputDebugVerb $lurette_cmd
	puts $ioc $lurette_cmd
	flush $ioc
	
	update
    } else {
	return 0
    }
}

proc OnEnableRunCmd { $dummy } {
    return [isProjectLoaded]
}
proc OnCheckRunCmd { $dummy } {}


#############

proc isNodeObject {obj} {
    if {[Class $obj] == "Node"} {
	return 1
    } else {
	return 0
    }
}


proc getSelectedNode {} {
    global selection
    return [lindex $selection 0]
}


proc isNodeSelected {} {
    global selection
    if {[isProjectLoaded]} {
        if {[llength $selection] == 1} {
	    if {[isNodeObject $selection]} {
		return 1
	    } else {
	        return 0
	    }
	} else {
	    return 0
	}
    } else {
	return 0
    }
}


############### Type retrieving #################


# Returns a string representing the definition of type
proc getTypedef { type } {
    if { [string compare $type "" ] == 0 } { 
	return "Anonymous type not supported, sorry."
    }
    #named types
    if { [Call $type IsNamedType ] } {
	# code mort ??????
	
	#defered type
	if { [ Call $type IsDeferred] } { 
	    return "defered type not supported, sorry."
	}
	#defined type
	set definition [ GetRole $type definition ]
	if { [string compare "" $definition ] != 0 } { 
	    return [getTypedef $definition ] 
	}
	#default
	return "[Get $type name]"
    }
    #array : forward to subtype
    if { [ Call $type IsTable ] } { 
	set arity [Get $type size]
	return "[ getTypedef [ GetRole $type type ] ] ^ $arity"
    }

    # Structure Types
    if { [ Call $type IsStructure ] } {
	return "{ [Call $type ToString] }";
    }

    # Tuple Types
    # XXX pour l'intant, y'a pas de tuple en Lucky, donc ca risque de na pas marcher
    # mais bon, le jour o j'y rajouterais, ca devrait marcher comme ca...
    if { [ Call $type IsTuple ] } {
  	concatDeepRoleToList $type { element type } subTypes
  	set text ""
	set i 0
  	foreach t $subTypes {
  	    set text "$text _F$i:[getTypedef $t]; "
	    incr i
  	}
	return "tuple not yet supported, sorry."
  	return "{ $text }";
	
    }

    # Enumerated type
    if  { [ Call $type IsEnumeration ] } {
	return "[Call $type ToString]"
    }
    
    # default
    return "Could not find type"
}




##############################################################################

# in order to collect the list of all types of the model (to be used wiht MapRole)
proc collectAllTypes { acc type } {
    upvar $acc accu

    lappend accu $type
    return 1
}

# Compute the parameters to be used by system calls such as gen_stubs or 
# gen_fake_lucky. It returns the part that deals with the input/output var names 
# and types, as well as the typedef.
#
proc ComputeIoParameters {} {
    global scade

    set inputs [BuildInputList [getSelectedNode]]
    set outputs [BuildOutputList [getSelectedNode]]
    set node [Get [getSelectedNode] name]

    set cmd " -inputs"

# Generate the list of inputs
    foreach i $inputs {
        set iName [Get $i name]
        set iType [GetRole $i type]
        set iTypeName [Get $iType name]	
	append cmd " $iName" " $iTypeName"
    }

# Generate the list of outputs
    append cmd " -outputs"
    foreach o $outputs {
        set oName [Get $o name]
        set oType [GetRole $o type]
        set oTypeName [Get $oType name]
	append cmd " $oName" " $oTypeName"
    }

# Generate the list of typedefs
    append cmd " -typedef "

    set model [GetRole $scade model]
    set all_types {}
    set res [MapRole $model allNamedType "collectAllTypes all_types"]
    set all_types_sorted  [lsort -unique $all_types]

    foreach type $all_types {
	set typedef [getTypedef $type]
	set typeName [Get $type name]
	append cmd " $typeName \" $typedef \" "
    }

# Pour traiter les Types utilisateurs (structure, tableaux, alias),
# j'envoie des trucs du style :
#
#      append cmd " type_enum \"(zero, un, deux)\""
#      append cmd " type_array \"int ^ 2 ^ 3\""
#      append cmd " type_struct \"{f1:bool; f2:type_enum ; f: int ^ 6}\""
#
# c'est--dire que les types structurs utiliss sont dfinis avec
# la syntaxe de Lucky.
#

    return $cmd
}


##############################################################################
#
# generate a fake environment for the selected Node

Declare command -variable "GenLucCmd" -name \
    "Generate skeleton environment files for the selected node" \
    -imagefile "./Images/env.bmp" \
    -statusmessage "Select a node in the Framework thumbnail and click" \
    -tooltip "Generate skeleton environment files for the selected node" \
    -accelerator "" \
    -OnActivateCommand OnActivateGenLucCmd \
    -OnEnableCommand OnEnableGenLucCmd \
    -OnCheckCommand OnCheckGenLucCmd



# create the "Lurette $node" folder if necessary
proc GetLuretteNodeFolder { project name } {
    set LuretteFolderVar ""
    MapRole $project root "SearchFolder LuretteFolderVar \"$name\""
    if { $LuretteFolderVar == "" } {
	output "Creating $name folder...\n"
	set LuretteFolderVar [CreateFolder $project $name $project]
    }
    
    return $LuretteFolderVar
}

# This procedure generates 3 files: 
#   o ${node}_env.luc,   (a environement for the currently selected node)
#   o ${node}_env_UD.luc, (an other environement) 
#   o ${node}_always_true.lus (a trivial oracle) 
#
proc OnActivateGenLucCmd { $dummy } {
    global project lurette_projectPathName

    set lurette_projectPathName [Get [lindex $project 0] pathname]
    set lurette_dir [file dirname $lurette_projectPathName]
    set node [Get [getSelectedNode] name]

    GenLuc $node
    OutputDebugVerb "OnActivateGenLucCmd Done."
}



proc GenLuc { node } {
    global lurette_dir lurette_projectPathName project ioc lurette_sut gen_fake_lucky

    if { [LaunchLuretteTop] } {
	
	set bSucess 1
	set env_file_name "${node}_env.luc"
	set path_env_file_name [file join $lurette_dir $env_file_name]
	set env_file_name_exists [file exists $path_env_file_name]
	
	set env_file_name2 "${node}_env_UD.luc"
	set path_env_file_name2 [file join $lurette_dir $env_file_name2]
	set env_file_name2_exists [file exists $path_env_file_name2]
	
#      set oracle_file_name "${node}_always_true.lus"
#      set path_oracle_file_name [file join $lurette_dir $oracle_file_name]
#      set oracle_file_name_exists [file exists $path_oracle_file_name]


	set cmd "exec -keepnewline \"[slashPath $gen_fake_lucky]\" "

        append cmd [ComputeIoParameters]
        append cmd " -name " $node " \"[file join ${lurette_dir} ]\" &"

#  	OutputDebugVerb "sending the following command to lurettetop: $cmd"
#  	OutputDebugVerb "$cmd\n"

	set cmd_protected [string map {\[  \\\[  \] \\\] } $cmd]
	

	output "\nLaunching the following system call: $cmd_protected"
	
	
	if { [ catch $cmd_protected result] } {
	    MessageDialog -name "Lurette Error " \
		-message "Error when launching gen_fake_lucky: $result\n"
	    output  "Error when launching gen_fake_lucky: $result\n"
	    return 0
	}


    # Create (if necessary) a Folder dedicated to the testing of the current node.
    # This folder is used to store the environment and the oracle files 
    # for $node
	set lurette_folder [file join "Lurette $node"] 

	set LuretteFolderVar [GetLuretteNodeFolder $project $lurette_folder]


    # XXX Ca n'est pas l'existence du fichier qu'il faudrait tester, mais 
    # l'existence d'une fileref !!
    #
    # create file reference iff files were existing before the gen_fake_lucky 
    # command is launched
	if { !$env_file_name_exists } {
	    output "Adding $path_env_file_name in the FileView thumbnail\n"
	    set fileRef [CreateFileReference $project $path_env_file_name \
			     $LuretteFolderVar]
	}
	
	if { !$env_file_name2_exists } {
	    output "Adding $path_env_file_name2 in the FileView thumbnail\n"
	    set fileRef2 [CreateFileReference $project $path_env_file_name2 \
			      $LuretteFolderVar]
	}
#      if { !$oracle_file_name_exists } {
#  	output "Adding $path_oracle_file_name in the FileView thumbnail\n"
#  	set fileRef3 [CreateFileReference $project $path_oracle_file_name \
#  			  $LuretteFolderVar]
#      }


	update
	
	return $bSucess
    } else {
	return 0
    }
}

proc OnEnableGenLucCmd { $dummy } {
    return [isNodeSelected]
}
proc OnCheckGenLucCmd { $dummy } {}


###################################



proc SockServerIO {channel clientaddr clientport} {
    global ioc socket_state
    
    OutputDebugVerb "IO: Connection from $clientaddr registered ...\n"

    fconfigure $channel -blocking   0 -translation lf -buffering line

    fileevent $channel readable [list ReadDataOnSocket $channel]

    puts $channel "hello.\n"
    flush $channel
    
    set ioc $channel
    set socket_state "connected"

    update
    return 1
}


proc FilterLuretteOutput { line } {
    global lurette_pid oracle_pid sut_pid project lurette_sut \
	lurette_step_nb lurette_is_running seed EditSeed

    set line_length [string length $line]
    # Filtering some of the messages coming from lurettetop
    if {
	$line_length <= 1
    } then {
	output ""
    } elseif {
	[string equal -length 40 $line \
	     "  Type h for help, or man for a small user manual."]
	||
	[string equal $line ""]
    } then {
	
    } elseif {
	      [string equal -length 9 $line "--- step "]
	  } then {
	set lurette_is_running true
	set stop_char [expr [string first ":" $line 8] -1]
	set cstep_nb [string range $line 9 $stop_char]
	set progress [expr { ($cstep_nb + 0.0) / $lurette_step_nb * 100.0 } ]
	set progress_str [format "%.1f" $progress]
	#OutputDebug "test sequence generation ... progress = $progress_str% ($cstep_nb/$lurette_step_nb)\n" 
	# XXX "\r" does not work. Why?
        # Ca existe les progress bar dans studio ??
        # update

    } elseif {
	      [string equal -length 22 $line "The Pid of lurette is "]
	  } then {
	set lurette_pid [string range $line 22 [string length $line]]
	set lurette_is_running true
	output "The lurette PID is $lurette_pid\n"
	
    } elseif { 
	      [string equal -length 22 $line "The Pid of the sut is "] 
	  } then {
	set sut_pid [string range $line 22 [string length $line]]
	set lurette_is_running true
	output "The SUT PID is $sut_pid\n"
    } elseif { 
	      [string equal -length 34 $line \
		   "The random engine was initialized "] 
	  } then {
	set seed [string range $line 48 [string length $line]]
	output "The seed was set to $seed\n"
    } elseif { 
	      [string equal -length 17 $line \
		   "Lurette finished."] 
	  } then {
	set lurette_is_running false
    } elseif { 
	      [string equal -length 44 $line \
		   "The generated data can be found in the file "] 
	  } then {
	# Add the generated rif file in the fileview thumbnail
	
	set lurette_is_running false
	set rif_file [string range $line 44 [expr [string length $line] - 2]]
	
	if { [file exists "$rif_file"] } {
	    set lurette_folder [file join "Lurette $lurette_sut"] 
	    output "Adding $rif_file in the $lurette_folder folder of the FileView thumbnail\n"
	    set LuretteFolderVar [GetLuretteNodeFolder $project $lurette_folder]
#  	    set lurette_folder [file join "Lurette RIF files"] 
#  	    set LuretteFolderVar [GetLuretteNodeFolder $project $lurette_folder]
	    set fileRef [CreateFileReference $project $rif_file $LuretteFolderVar]
	} else {
	    output "$rif_file Does not exist\n"
	}
	
	
    } else {
	output "$line\n"
    }
    # XXX R1 : cf xlurette_glade_main pour traiter les cas a mettre sur stderr
 
    return 1
}

# XXX LP : l'affichage effectu par cette fonction n'est effectif que
# quand un "update" est execut. Du coup, j'en ai mis un peu partout,
# mais ca n'est ni trs satisfaisant, ni suffisant dans certains cas...
proc ReadDataOnSocket {chan} {

    if {![eof $chan]} {
        set line [gets $chan]
	# XXX LP : o (et comment) dois-je mettre les stdout et stderr de 
	#    lurettetop que je recupere via les sockets ?
	#
	# pour l'instant, je ne met que stdout dans l'onglet "script" avec output
	# Il faudrait peut-etre creer des onglets ad hoc ?
	# "Lurette stdout" et "Lurette stderr" ???
	# Si oui, comment ?
	
	FilterLuretteOutput $line		
	update idletasks
    }
}

proc Sleep {N} {
   after [expr {int($N * 1000)}]
}
proc ReadData {str chan} {
    global project

    set line [gets $chan]
    if {![string equal $line ""]} {
	if { $str == "stderr:" } {
	    OutputDebug "$str $line\n" 
	} else {
	    FilterLuretteOutput $line
	}
	ReadData $str $chan
	update
    }
    # XXX parce fileevent ne marche pas !!!
    # Il croit qu'il y a toujours des choses  lire, 
    # meme quand ca n'est pas vrai !
    after 1000 [list after 0 ReadData $str $chan]
}


proc ComputeFinalTargetDir {targetDir nodeName confName codeGen computeTargetDir} {
    upvar $computeTargetDir result

     set result $targetDir
     regsub {\$\(NodeName\)} $result $nodeName result
     regsub {\$\(Configuration\)} $result $confName result
     regsub {\$\(CG\)} $result $codeGen result

    return $result
 } 


proc ComputeTmpDir {} {
    global project lurette_dir luretteincludedir
    
    CheckConf
    set conf [GetLuretteConfiguration]
    set confname [Get $conf name]
    set root_node [GetRootNode $project]
    set targetdir  [call $project GetToolProp "GENERATOR" "TARGET_DIR" $conf]
    
    # set targetcode [call $project GetToolProp "GENERATOR" "TARGETCODE" $conf]
    # XXX En attendant que le ligne d'audessus rende le bon resultat...   
    set targetcode "C"
    
    # XXX par ailleurs, si je suis en mode "selected node" dans la fenetre 
    # "Settings/code generator", ca ne marche pas: il me renvoie un chemin du style
    # ".../mon_neuneu/_C/Lurette" ...
    set computeTargetDir ""
    ComputeFinalTargetDir $targetdir $root_node $confname $targetcode \
	computeTargetDir
    
    set computeTargetDirPath [ file join $lurette_dir $computeTargetDir]
    
    if { ![file exists $computeTargetDirPath] } {
	output "$computeTargetDirPath does not exist!\n"
	output "  - first possible reason: \"Root node\" should not be set to <selected node> \
for Lurette.\n"
	output "  - second possible reason: you need to click on the Build button of\
the \"simulator and code generation\" toolbar before using Lurette.\n"
	return "none"
    }

    return $computeTargetDirPath
}

proc LaunchLuretteTop {} {
    global socket_state lurettetop log_oc log_ec project \
	lurette_projectPathName lurettetop_is_running lurette_dir env

    if { $lurettetop_is_running } {
	OutputDebugVerb "ltop running...\n"
    } else {
	OutputDebug "Launching lurettetop...\n"
	set lurettetop_is_running true

	# To be able to quit Lurettetop properly
 	set bindir $env(ETBIN)
 	load [file join $bindir LuretteTcl.dll]
 	Lurette 

	set lurette_projectPathName [Get [lindex $project 0] pathname]
	set lurette_dir [file dirname $lurette_projectPathName]
	cd $lurette_dir

	set tmp_dir [ComputeTmpDir]
	if { $tmp_dir == "none" } {
	    output "Can not launch lurette\n\n"
	    set lurettetop_is_running false
	    return 0
	}

	# socket stuff
	set inet_addr "127.0.0.1"
	set io_port 2000 
	
	set connected false
	while {! $connected } {
	    set connected true
	    if { [catch { set server_ioc [socket -server SockServerIO \
					      -myaddr $inet_addr $io_port] } ] } {
		output "socket connection on port $io_port failed. I try again ...\n"
		incr io_port
		set connected false
	    }
	}
	OutputDebugVerb "socket connected on port $io_port\n"

	# log files 
	set stdout_log_file [file join $tmp_dir ".." ".." "lurette_stdout.log"]
	set stderr_log_file [file join $tmp_dir ".." ".." "lurette_stderr.log"]
	catch { [file remove $stdout_log_file] }
	catch { [file remove $stderr_log_file] }

	# Launching... 	
	set cmd "exec -keepnewline \"[slashPath $lurettetop]\" --sut-compiler scade-gui \
 --tmp-dir \"$tmp_dir\" --log --socket-inet-addr $inet_addr \
 --socket-io-port $io_port &"
	
	OutputDebugVerb "$cmd\n"
#   	MessageDialog -name "Lurettetop  " \
#   	    -message "$cmd\n computeTargetDir=$tmp_dir" -icon warning 

	set bSucess 1
	if { [ catch $cmd result] } {
	    MessageDialog -name "Lurette Error " \
		-message "Error when launching lurettetop: $result\n"
	    output  "Error when launching lurettetop: $result\n"
	    close ioc
	    set bSucess 0
	}
	
	# To avoid blocking
	after 5000 set socket_state timeout
	vwait socket_state
	
	switch $socket_state {
	    timeout {
		MessageDialog -name "Lurette error " -message "Socket connection timeout! Lurette will not work.\n"
		output "Socket connection timeout!\n"
	    } 
	    connected {
		OutputDebugVerb  "Sockets connected...\n"
	    }
	}
	

	set cont true
	while { $cont } {
	    if { [catch {
		set log_oc [open $stdout_log_file r]
		set log_ec [open $stderr_log_file r]
		set cont false
	    } ] } {
		# wait that lurettetop have indeed created the log files
		Sleep 1
	    }
	}
	    
	
	fconfigure $log_oc  -blocking 0 -buffering line -translation lf 
	fconfigure $log_ec  -blocking 0 -buffering line -translation lf 

	
	# Display log files
	# XXX fileevent ne marche pas !!!
	ReadData "stdout:" $log_oc
	#    fileevent $log_oc readable [list ReadData $log_oc]
	
	ReadData "stderr:" $log_ec
	
	return $bSucess
    }
    return 1
}


###################################

proc OnEnableCtxMenuLurette { $dummy } {
    return [isNodeSelected]
}

#Declare ContextMenu -Variable "CtxMenuLurette" \
# -OnEnableMenu OnEnableCtxMenuLurette -commands {GenLucCmd}    

# main entry procedure ################################################

proc Init {} {
    
    addcommand OpenLuretteSettingsCmd
    addcommand OpenLurettePrefCmd
    addcommand GenLucCmd

    addcommand GnuPlotCmd
    addcommand UpdateCmd
    addcommand RebuildCmd
    addcommand BuildCmd
    addcommand RunCmd
    addcommand StopRunCmd

    addtoolbar -variable "Lurette" \
	-commands {BuildCmd RebuildCmd separator RunCmd separator StopRunCmd \
		       separator OpenLuretteSettingsCmd OpenLurettePrefCmd \
		       separator GenLucCmd separator GnuPlotCmd UpdateCmd  \
		       }

    addmenu \
	-variable "MenuLurette" \
	-commands { separator BuildCmd RebuildCmd RunCmd separator \
			OpenLuretteSettingsCmd \
			OpenLurettePrefCmd separator \
			GenLucCmd separator \
			GnuPlotCmd  separator \
		    }\
	-path {&Project &Lurette} -position {after "Co&nfigurations..."}
    
    AddContextMenu -variable "CtxMenuLurette" \
	-OnEnableMenu OnEnableCtxMenuLurette \
	-commands {separator GenLucCmd OpenLuretteSettingsCmd}

    addcommand AboutCmd
    addmenu -variable "MenuAbout" -commands { AboutCmd } \
	-path { &Help } -position {last}


# XXX Does not fix the  update  problem
#    UpdateLoop
}

proc UpdateLoop { } {
    update
    output "updating...\n"
    after idle 2000 [list after 0 UpdateLoop]
}



#######################################################################
# End GUI custom code						      #
#######################################################################


# start ###############################################################
global project

Init


