#include <stdio.h>
#include <string.h>
#include <caml/mlvalues.h>
#include <caml/callback.h>

/* ********************************************************************** */
/* Expressions */
/* ********************************************************************** */

/* for type safety, we embed caml values in specific C types */

typedef struct bool_expr {
  value v;
} bool_expr;
typedef struct int_expr {
  value v;
} int_expr;
typedef struct float_expr {
  value v;
} float_expr;
typedef struct solutions_set {
  value v;
} solutions_set;

/* ********************************************************************** */
/* Values, Substitutions, Solutions */
/* ********************************************************************** */

typedef enum val_type { val_float, val_int, val_bool } val_type;

typedef struct val {
  enum val_type val_type;
  union {
    char valb;;
    int vali;
    double valf;
  } u;
} val;

typedef struct subst {
  char* name;
  struct val val;
} subst;

typedef struct solution {
  subst* array;
  size_t size;
} solution;

typedef struct solutions {
  solution* array;
  size_t size;
} solutions;

typedef enum draw_mode { draw_inside, draw_edge, draw_vertex } draw_mode;

/* ********************************************************************** */
/* Initialisation of Caml runtime */
/* ********************************************************************** */

void lucky_caml_init();

/* ********************************************************************** */
/* Boolean Constructors */
/* ********************************************************************** */

bool_expr lucky_var_b(const char* name);
bool_expr lucky_true_b();
bool_expr lucky_false_b();

bool_expr lucky_and_b(bool_expr e1, bool_expr e2);
bool_expr lucky_or_b(bool_expr e1, bool_expr e2);
bool_expr lucky_xor_b(bool_expr e1, bool_expr e2);
bool_expr lucky_impl_b(bool_expr e1, bool_expr e2);
bool_expr lucky_eq_b(bool_expr e1, bool_expr e2);
bool_expr lucky_not_b(bool_expr e1);
bool_expr lucky_ite_b(bool_expr e1, bool_expr e2, bool_expr e3);

/* ********************************************************************** */
/* Integer Constructors */
/* ********************************************************************** */

int_expr lucky_var_i(const char* name);
int_expr lucky_val_i(int n);
bool_expr lucky_eq_i(int_expr e1, int_expr e2);
bool_expr lucky_sup_i(int_expr e1, int_expr e2);
bool_expr lucky_supeq_i(int_expr e1, int_expr e2);
bool_expr lucky_inf_i(int_expr e1, int_expr e2);
bool_expr lucky_infeq_i(int_expr e1, int_expr e2);
int_expr lucky_ite_i(bool_expr e1, int_expr e2, int_expr e3);
int_expr lucky_sum_i(int_expr e1, int_expr e2);
int_expr lucky_diff_i(int_expr e1, int_expr e2);
int_expr lucky_prod_i(int_expr e1, int_expr e2);
int_expr lucky_quot_i(int_expr e1, int_expr e2);
int_expr lucky_mod_i(int_expr e1, int_expr e2);
int_expr lucky_div_i(int_expr e1, int_expr e2);
int_expr lucky_uminus_i(int_expr e1);

/* ********************************************************************** */
/* Float Constructors */
/* ********************************************************************** */

float_expr lucky_var_f(const char* name);
float_expr lucky_val_f(double d);
bool_expr lucky_eq_f(float_expr e1, float_expr e2);
bool_expr lucky_sup_f(float_expr e1, float_expr e2);
bool_expr lucky_supeq_f(float_expr e1, float_expr e2);
bool_expr lucky_inf_f(float_expr e1, float_expr e2);
bool_expr lucky_infeq_f(float_expr e1, float_expr e2);
float_expr lucky_ite_f(bool_expr e1, float_expr e2, float_expr e3);
float_expr lucky_sum_f(float_expr e1, float_expr e2);
float_expr lucky_diff_f(float_expr e1, float_expr e2);
float_expr lucky_prod_f(float_expr e1, float_expr e2);
float_expr lucky_quot_f(float_expr e1, float_expr e2);
float_expr lucky_uminus_f(float_expr e1);

/* ********************************************************************** */
/* Constraint solver and drawer */
/* ********************************************************************** */

/* Deallocating functions */
void subst_free(subst subst);
void solution_free(solution sol);
void solutions_free(solutions sols);

solutions_set lucky_solve(bool_expr e);
solutions lucky_draw(int inside, int edge, int vertex, int verbose, solutions_set solset);
void lucky_set_efficient_mode();
void lucky_set_fair_mode();

/* ********************************************************************** */
/* Pretty-printing and misc */
/* ********************************************************************** */

char* lucky_bool_expr_to_string(bool_expr e);

void fprint_bool_expr(FILE* stream, bool_expr e);
void fprint_val(FILE* stream, val val);
void fprint_subst(FILE* stream, subst subst);
void fprint_solution(FILE* stream, solution sol);
void fprint_solutions(FILE* stream, solutions sols);

void lucky_set_seed(int seed);
