
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <caml/mlvalues.h>
#include <caml/callback.h>
#include <caml/memory.h>
#include <caml/alloc.h>

#include "luckyDrawC_stubs.h"

dnl ***************************************************************************
dnl M4 Macros
dnl ***************************************************************************
dnl
dnl parameter 1: type of the returned value
dnl parameter 2: name of the registered caml closure
dnl              and of the generated C function
dnl paramater 3: type of the single paramater
dnl
define(`OP1',`
$1 $2($3 e1)
{
  static value* closure=NULL;
  $1 e = { 0 };

  Begin_roots2(e.v,e1.v);
  if (closure == NULL)
    closure = caml_named_value("$2");

  e.v = callback_exn(*closure,e1.v);
  assert(! Is_exception_result(e.v));
  End_roots();
  return e;
}
')dnl
dnl
dnl parameter 1: type of the returned value
dnl parameter 2: name of the registered caml closure
dnl              and of the generated C function
dnl paramaters 3 & 4: type of the parameters
dnl
define(`OP2',`
$1 $2($3 e1, $4 e2)
{
  static value* closure=NULL;
  $1 e = { 0 };

  Begin_roots3(e.v,e1.v,e2.v);
  if (closure == NULL)
    closure = caml_named_value("$2");
  e.v = callback2_exn(*closure,e1.v,e2.v);
  assert(! Is_exception_result(e.v));
  End_roots();
  return e;
}
')dnl
dnl
dnl parameter 1: type of the returned value
dnl parameter 2: name of the registered caml closure
dnl              and of the generated C function
dnl paramaters 3,4,5 : type of the parameters
dnl
define(`OP3',`
$1 $2($3 e1, $4 e2, $5 e3)
{
  static value* closure=NULL;
  $1 e = { 0 };


  Begin_roots4(e.v,e1.v,e2.v,e3.v);
  if (closure == NULL)
    closure = caml_named_value("$2");
  e.v = callback3_exn(*closure,e1.v,e2.v,e3.v);
  assert(! Is_exception_result(e.v));
  End_roots();
  return e;
}
')dnl
dnl
dnl parameter 1: type of the returned value
dnl parameter 2: name of the registered caml closure
dnl              and of the generated C function
dnl
define(`OP1S',`
$1 $2(const char* name)
{
  static value* closure=NULL;
  value str = 0;
  $1 e = { 0 };

  Begin_roots2(e.v,str);
  if (closure == NULL)
    closure = caml_named_value("$2");

  str = caml_copy_string(name);
  e.v = callback_exn(*closure,str);
  assert(! Is_exception_result(e.v));
  End_roots();
  return e;
}
')dnl

/* ********************************************************************** */
/* Initialisation of Caml runtime */
/* ********************************************************************** */

void lucky_caml_init()
{
  char* argv[1] = { NULL };

  caml_startup(argv);
}

/* ********************************************************************** */
/* Boolean Constructors */
/* ********************************************************************** */

OP1S(bool_expr, lucky_var_b)
bool_expr lucky_true_b()
{
  static value* closure=NULL;
  bool_expr e = { 0 };

  if (closure == NULL)
    closure = caml_named_value("lucky_true_b");

  e.v = *closure;
  return e;
}
bool_expr lucky_false_b()
{
  static value* closure=NULL;
  bool_expr e = { 0 };

  if (closure == NULL)
    closure = caml_named_value("lucky_false_b");

  e.v = *closure;
  return e;
}
OP2(bool_expr, lucky_and_b, bool_expr, bool_expr)
OP2(bool_expr, lucky_or_b, bool_expr, bool_expr)
OP2(bool_expr, lucky_xor_b, bool_expr, bool_expr)
OP2(bool_expr, lucky_impl_b, bool_expr, bool_expr)
OP2(bool_expr, lucky_eq_b, bool_expr, bool_expr)
OP1(bool_expr, lucky_not_b, bool_expr)
OP3(bool_expr, lucky_ite_b, bool_expr, bool_expr, bool_expr)

/* ********************************************************************** */
/* Integer Constructors */
/* ********************************************************************** */

OP1S(int_expr, lucky_var_i)
int_expr lucky_val_i(int n)
{
  static value* closure=NULL;
  value cst = 0;
  int_expr e = { 0 };


  Begin_roots2(e.v,cst);
  if (closure == NULL)
    closure = caml_named_value("lucky_val_i");
  cst = Val_int(n);
  e.v = callback_exn(*closure,cst);
  assert(! Is_exception_result(e.v));
  End_roots();
  return e;
}
OP2(bool_expr, lucky_eq_i, int_expr, int_expr)
OP2(bool_expr, lucky_sup_i, int_expr, int_expr)
OP2(bool_expr, lucky_supeq_i, int_expr, int_expr)
OP2(bool_expr, lucky_inf_i, int_expr, int_expr)
OP2(bool_expr, lucky_infeq_i, int_expr, int_expr)
OP3(int_expr, lucky_ite_i, bool_expr, int_expr, int_expr)
OP2(int_expr, lucky_sum_i, int_expr, int_expr)
OP2(int_expr, lucky_diff_i, int_expr, int_expr)
OP2(int_expr, lucky_prod_i, int_expr, int_expr)
OP2(int_expr, lucky_quot_i, int_expr, int_expr)
OP2(int_expr, lucky_mod_i, int_expr, int_expr)
OP2(int_expr, lucky_div_i, int_expr, int_expr)
OP1(int_expr, lucky_uminus_i, int_expr)

/* ********************************************************************** */
/* Float Constructors */
/* ********************************************************************** */

OP1S(float_expr, lucky_var_f)
float_expr lucky_val_f(double d)
{
  static value* closure=NULL;
  value cst = 0;
  float_expr e = { 0 };


  Begin_roots2(e.v,cst);
  if (closure == NULL)
    closure = caml_named_value("lucky_val_f");
  cst = copy_double(d);
  e.v = callback_exn(*closure,cst);
  assert(! Is_exception_result(e.v));
  End_roots();
  return e;
}
OP2(bool_expr, lucky_eq_f, float_expr, float_expr)
OP2(bool_expr, lucky_sup_f, float_expr, float_expr)
OP2(bool_expr, lucky_supeq_f, float_expr, float_expr)
OP2(bool_expr, lucky_inf_f, float_expr, float_expr)
OP2(bool_expr, lucky_infeq_f, float_expr, float_expr)
OP3(float_expr, lucky_ite_f, bool_expr, float_expr, float_expr)
OP2(float_expr, lucky_sum_f, float_expr, float_expr)
OP2(float_expr, lucky_diff_f, float_expr, float_expr)
OP2(float_expr, lucky_prod_f, float_expr, float_expr)
OP2(float_expr, lucky_quot_f, float_expr, float_expr)
OP1(float_expr, lucky_uminus_f, float_expr)

/* ********************************************************************** */
/* Constraint solver */
/* ********************************************************************** */

solutions_set lucky_solve(bool_expr e)
{
  static value* closure=NULL;
  solutions_set sol = { 0 };
  
  Begin_roots2(sol.v,e.v);
  if (closure == NULL)
    closure = caml_named_value("lucky_solve");
  sol.v = callback_exn(*closure,e.v);
  assert(! Is_exception_result(sol.v));
  End_roots();
  return sol;
}

/* ********************************************************************** */
/* Constraint drawer */
/* ********************************************************************** */

/* ====================================================================== */
/* Type conversion functions */
/* ====================================================================== */

size_t size_of_list(value v)
{
  size_t size = 0;

  while (v != Val_int(0)){
    v = Field(v,1);
    size++;
  }	
  return size;
}

/* ---------------------------------------------------------------------- */
/* value */

val val_of_value(value v)
{
  val val;

  switch(Tag_val(v)){
  case 0:
    /* float */
    val.val_type = val_float;
    val.u.valf = Double_val(Field(v,0));
    break;
  case 1:
    /* integer */
    val.val_type = val_int;
    val.u.vali = Int_val(Field(v,0));
    break;
  case 2:
    /* boolean */
    val.val_type = val_bool;
    val.u.valb = Int_val(Field(v,0));
    break;
  default:
    assert(0);
  }
  return val;
}

/* ---------------------------------------------------------------------- */
/* substitution */

subst subst_of_value(value v)
{
  subst subst;

  subst.name = strdup(String_val(Field(v,0)));
  subst.val = val_of_value(Field(v,1));
  return subst;
}

/* ---------------------------------------------------------------------- */
/* solution */

solution solution_of_value(value v)
{
  solution sol;
  size_t size;
  int i;
  
  size = size_of_list(v);
  sol.size = size;
  sol.array = (subst*)malloc(size*sizeof(subst));
  /* Iterates on substitutions */
  for (i=0; i<size; i++){
    sol.array[i] = subst_of_value(Field(v,0));
    v = Field(v,1);
  }
  return sol;
}

/* ---------------------------------------------------------------------- */
/* solutions */

solutions solutions_of_value(value v)
{
  solutions sols;
  size_t size;
  int i;
  
  size = size_of_list(v);
  sols.size = size;
  sols.array = (solution*)malloc(size*sizeof(solution));
  /* Iterates on solutions */
  for (i=0; i<size; i++){
    sols.array[i] = solution_of_value(Field(v,0));
    v = Field(v,1);
  }
  return sols;
}

/* ====================================================================== */
/* Deallocating functions */
/* ====================================================================== */

void subst_free(subst subst)
{
  free(subst.name);
}

void solution_free(solution sol)
{
  int i;
  for (i=0; i<sol.size; i++){
    subst_free(sol.array[i]);
  }
  free(sol.array);
}

void solutions_free(solutions sols)
{
  int i;

  for (i=0; i<sols.size; i++){
    solution_free(sols.array[i]);
  }
  free(sols.array);
}

/* ====================================================================== */
/* Printing functions */
/* ====================================================================== */

void fprint_bool_expr(FILE* stream, bool_expr e)
{
  char* str;

  str = lucky_bool_expr_to_string(e);
  fprintf(stream,"%s\n",str);
  free(str);
}

void fprint_val(FILE* stream, val val)
{
  switch(val.val_type){
  case val_bool:
    fprintf(stream, "%s", val.u.valb ? "true" : "false");
    break;
  case val_int:
    fprintf(stream, "%d", val.u.vali);
    break;
  case val_float:
    fprintf(stream, "%f", val.u.valf);
    break;
  default:
    assert(0);
  }
}

void fprint_subst(FILE* stream, subst subst)
{
  fprintf(stream,"%s = ", subst.name);
  fprint_val(stream,subst.val);
}

void fprint_solution(FILE* stream, solution sol)
{
  int i;
  for(i=0; i<sol.size; i++){
    if (i>0) fprintf(stream,", ");
    fprint_subst(stream,sol.array[i]);
  }
  fprintf(stream,"\n");
}
void fprint_solutions(FILE* stream, solutions sols)
{
  int i;
  for(i=0; i<sols.size; i++){
    fprint_solution(stream,sols.array[i]);
  }
}

/* ====================================================================== */
/* Constraint drawing */
/* ====================================================================== */

solutions lucky_draw(int inside, int edge, int vertex, int verbose, solutions_set solset)
{
  static value* closure=NULL;
  value triplet=0;
  value otriplet=0;
  value sols = 0;
  value overbose=0;
  solutions solutions;

  fprintf(stdout,"lucky_draw\n");	
  Begin_roots5(triplet,otriplet,overbose,sols,solset.v);
  if (closure == NULL)
    closure = caml_named_value("lucky_draw");
  /* Build the optional argument */
  triplet = caml_alloc_tuple(3);
  Store_field (triplet, 0, Val_int(inside));
  Store_field (triplet, 1, Val_int(edge));
  Store_field (triplet, 2, Val_int(vertex));
  otriplet = caml_alloc(1,0);
  Store_field (otriplet, 0, triplet);
  overbose = caml_alloc(1,0);
  Store_field (overbose, 0, Val_int(verbose));
  /* call to caml function */
  sols = callback3_exn(*closure,otriplet,overbose,solset.v);
  assert(! Is_exception_result(sols));

  /* conversion from list of list to array of array */
  solutions = solutions_of_value(sols);
  End_roots();
  return solutions;
}

void lucky_set_efficient_mode()
{
  static value* closure=NULL;
  value v=0;

  if (closure == NULL)
    closure = caml_named_value("lucky_set_efficient_mode");
  v = callback_exn(*closure,Val_unit);
  assert(! Is_exception_result(v));
  return;
}
void lucky_set_fair_mode()
{
  static value* closure=NULL;
  value v=0;

  if (closure == NULL)
    closure = caml_named_value("lucky_set_fair_mode");
  v = callback_exn(*closure,Val_unit);
  assert(! Is_exception_result(v));
  return;
}

/* ********************************************************************** */
/* Pretty-printing and misc */
/* ********************************************************************** */

char* lucky_bool_expr_to_string(bool_expr e)
{
  static value* closure=NULL;
  value v = 0;


  Begin_roots2(e.v,v);
  if (closure == NULL)
    closure = caml_named_value("lucky_bool_expr_to_string");
  v = callback_exn(*closure,e.v);
  assert(! Is_exception_result(v));
  End_roots();
  return strdup(String_val(v));
}

void lucky_set_seed(int seed)
{
  static value* closure=NULL;
  value vseed = 0;
  value v = 0;
  if (closure == NULL)
    closure = caml_named_value("lucky_set_seed");

  vseed = Val_int(seed);
  v = callback_exn(*closure,vseed);
  assert(! Is_exception_result(v));
  return;
}
